(* $Id: rpc_client.mli 185 2004-07-13 13:09:04Z gerd $
 * ----------------------------------------------------------------------
 *
 *)

(* This module implements an RPC client, i.e. provides means to connect
 * to an RPC service and call remote procedures.
 * In general, this module works in an asynchronous way and is implemented
 * event-driven. All events are handled by an event queue of type
 * Unixqueue.t that must already exist and to which this module adds its
 * own event handlers and event resources. This means that this module
 * can co-exist with other services and share the same event queue with
 * them.
 * You can push several procedure calls on the event queue at once.
 * The queue serves then as a pipeline; the calls are sent to the
 * server as long as the server accepts new calls. Replies are received
 * in any order, and the return values of the remote procedures are
 * delivered using a callback function.
 * You can set timeouts and force automatic retransmission if you want
 * this; these features are enabled by default if the underlying transport
 * mechanism is UDP. Timeouts and other exceptions are delivered to the
 * callback functions, too.
 * The whole mechanism is designed to allow maximum parallelism without
 * needing to use the multi-threading features of O'Caml. Especially,
 * the following parallelisms can be done:
 * - Call several procedures of the same server in parallel. Note that
 *   this does not necessarily mean that the procedures are run in
 *   parallel since the server is free to decide whether to work
 *   in a synchronous or asynchronous way.
 * - Call several procedures of different servers in parallel. To do so,
 *   simply add several RPC clients to the same event queue.
 * - Call a procedure and do something completely different in the
 *   background; this works well as long as the other task can be
 *   programmed using file descriptor events, too.
 * However, there are still some restrictions concerning asynchronous
 * calls. Some of them will be removed in the future, but others are
 * difficult to tackle:
 * - TCP connects are currently always done synchronously. If the [create]
 *   function returns the connection could be established.
 * - Portmapper queries are done synchronously.
 * - Authentication methods requiring RPC calls or other network services are
 *   performed in an synchronous way, too.
 * - Name service lookups are synchronous, too.
 *)

open Rpc
open Xdr
open Rtypes

(* The following exceptions are delivered to the callback function: *)

exception Message_lost
  (* got EOF when some pending procedure calls were not replied or even sent *)

exception Message_timeout
  (* After all retransmissions, there was still no reply *)

exception Communication_error of exn
  (* an I/O error happened *)

exception Client_is_down
  (* The RPC call cannot be performed because the client has been shut down
   * in the meantime. You can get this exception if you begin a new call,
   * but the connection is closed now.
   *)

exception Keep_call
  (* This exception can be raised by the callback function that is invoked
   * when the server response arrives. It causes that the RPC call record
   * is kept in the housekeeping structure of the client. If the server
   * sends another response, the callback function will be invoked again.
   * I.e. one call can be replied several times (batching).
   *)


type t
  (* The type of RPC clients *)


type connector =
    Inet of (string * int)                    (* Hostname or IP address, port *)
  | InetNumeric of (int * int * int * int * int)             (* IP addr, port *)
  | Unix of string                                   (* path to unix dom sock *)
  | Descriptor of Unix.file_descr
                                    (* Pass an already open socket descriptor *)
  | Dynamic_descriptor of (unit -> Unix.file_descr)
                       (* The function is called to get the socket descriptor *)
  | Dynamic_descriptor' of ((unit -> Unix.file_descr) * 
			    (t -> Unix.file_descr -> unit))
            (* The first function is called to get the socket descriptor, the *
	     * second function is called to shut down the socket.             *
	     *)

(* TODO:
 * - Remove InetNumeric. It won't work for IPv6.
 * - Add InetAddr of (inet_addr * port)
 *)

(* Note: The file descriptor passed with Descriptor must be a socket. *)

val shutdown_connector : t -> connector -> Unix.file_descr -> unit
  (* The default implementation to shut down the connector.
   * For [Descriptor] this is a no-op, 
   * for [Dynamic_descriptor'] the second function is called,
   * for the other connector types the socket is closed.
   *)

val create :
      ?program_number:uint4 ->    (* Override the program number *)
      ?version_number:uint4 ->    (* Override the version number *)
      ?initial_xid:int ->         (* the initial xid *)
      ?shutdown:(t -> connector -> Unix.file_descr -> unit) -> (* see below *)
      Unixqueue.event_system ->   (* the event queue to be used *)
      connector ->            (* Connect to which RPC server? *)
      protocol ->             (* which protocol type *)
      Rpc_program.t ->        (* signatures of the remote program *)
      t

  (* Opens a connection to the server specified by the 'connector'.
   * The server is assumed to implement an RPC program as specified by
   * the Rpc_program.t argument. (You can override the program and version
   * numbers stored in this argument by the optional parameters
   * ?program_number and ?version_number.)
   * All communication to the server is handled using the given queue.
   * All handlers, events and resources are associated to a new group.
   *
   * If the protocol is Tcp, the communication will be handled stream-
   * oriented. In this case, no timeout is detected and no retransmissions
   * are done.
   * If the protocol is Udp, a datagram-oriented communication style is
   * used. This works only for Internet UDP sockets because these are
   * bidirectional (Unix domain sockets are unidirectional and do not
   * work). For Udp, there is a timeout of 15 seconds and a maximum
   * of 3 retransmissions (i.e. a total of 4 transmission trials).
   *
   * @param shutdown This function is called when the client is shut down
   *   to close the client socket. By default, [shutdown_connector] is
   *   called.
   *)

val configure : t -> int -> float -> unit

  (* configure client retransmissions timeout:
   * sets the number of retransmissions and the timeout for the next calls.
   * (These values are defaults; the actual values are stored with each
   * call.)
   * Values of retransmissions > 0 are semantically only valid if the
   * called procedures are idempotent, i.e. invoking them several times
   * with the same values has the same effect as only one invocation.
   * Positive values for 'retransmissions' should only be used for Udp-style
   * communication.
   * The timeout value determines how long the client waits until the
   * next retransmission is done, or, if no more retransmissions are
   * permitted, a Message_timeout exception is delivered to the receiving
   * callback function. A timeout value of 0.0 means immediate timeout
   * and is usually senseless. A negative timeout value means 'no timeout'.
   * Positive timeout values are possible for both Udp and Tcp connections.
   * Timeout values are measured in seconds.
   *
   * There is a special application for the timeout value 0.0: If you
   * don't expect an answer from the server at all ("batch mode"), this
   * timeout value will cause that the message handler will get
   * a Message_timeout exception immediately. You should ignore this
   * timeout for batch mode. The positive effect from the timeout is that
   * the internal management routines will remove the remote call from
   * the list of pending calls such that this list will not become too long.
   *)

val set_exception_handler : t -> (exn -> unit) -> unit
  (* sets an exception handler (the default is a 'do nothing' exception
   * handler). Only exceptions resulting from invocations of a
   * callback function are forwarded to this handler.
   * Exceptions occuring in the handler itself are silently dropped.
   *)

(* NOTE ON EXCEPTIONS:
 *
 * Uncaught exceptions are problematic in an asynchronous environment. It is
 * unclear what to do with them, since there is usually no 'caller' who can
 * get them.
 * The Unixqueue module implements a special handling for uncaught exceptions.
 * A special 'abort' routine is performed that can be specified. This module
 * adds such an 'abort' routine that does all necessary cleanup, i.e. the
 * file descriptors are shut down and so on. After this has been done,
 * the client is in a clean state; but nobody has been notified about
 * this special situation. As the callback functions serve as communication
 * endpoints, these functions are telled that such an exception has
 * happened using Communication_error x. (If there are no more pending
 * or waiting calls, exceptions are silently dropped.)
 * If these functions do not catch this kind of exception, the global
 * exception handler is used.
 *)

val add_call :
    t ->                    (* which client *)
    string ->               (* name of the procedure *)
    xdr_value ->            (* parameter of the procedure *)
    ((unit -> xdr_value) -> unit) ->
                            (* this function is called back when the result has
                             * been received. The function passed as
                             * parameter either evaluates to the returned
                             * value or raises an exception.
			     * The callback function normally just returns ().
			     * However, it is also possible to raise the
			     * special exception Keep_call (see above) to
			     * work in batch mode.
                             *)
      unit

  (* add a call to the queue of unprocessed calls *)


val event_system : t -> Unixqueue.event_system
  (* Returns the unixqueue to which the client is attached *)

val program : t -> Rpc_program.t
  (* Returns the program the client represents *)

val get_socket_name : t -> Unix.sockaddr
val get_peer_name : t -> Unix.sockaddr
  (* Return the addresses of the client socket and the server socket, resp.
   *)

val get_protocol : t -> Rpc.protocol
  (* Get the protocol flavour *)


val sync_call :
    t ->            (* which client *)
    string ->       (* which procedure (name) *)
    xdr_value ->    (* the parameter of the procedure *)
    xdr_value       (* the result of the procedure *)
  (* Calls the procedure synchronously.
   * Note that this implies that the underlying unixqueue is started and that
   * all events are processed regardless of whether they have something to do
   * with this call or not.
   *)



val shut_down : t -> unit

  (* Shuts down the connection. Any unprocessed calls get the exception
   * Message_lost.
   * Note that the file descriptor is only shut down if the connection
   * has been opened by this module itself, i.e. if the 'connector' was
   * not Descriptor.
   *)

class type auth_session =
object
  method next_credentials : t ->
                            (string * string * string * string)
         (* Returns (cred_flavour, cred_data, verifier_flavor, verifier_date)
	  *)
  method server_rejects : server_error -> unit
         (* Called if the server rejects the credentials or the verifier
	  * (Auth_xxx). This method may
	  * raise an exception to indicate that authentication has finally
	  * failed.
	  *)
  method server_accepts : string -> string -> unit
         (* Called if the server accepts the credentials. The two strings
	  * are the returned verifier_flavor and verifier_data.
	  * This method may raise Rpc_server Rpc_invalid_resp to indicate
	  * that the returned verifier is wrong.
	  *)
end
  (* An [auth_session] object is normally created for every client instance.
   * It contains the current state of authentication. The methods are only
   * interesting for implementors of authentication methods.
   *
   * This class type might be revised in the future to allow asynchronous
   * authentication (authentication often uses some network service).
   *)


class type auth_method =
object
  method name : string
         (* The name of this method, used for errors etc. *)
  method new_session : unit -> auth_session
         (* Begin a new session *)
end
  (* An [auth_method] object represents a method of authentication. Such an
   * object can be shared by several clients.
   *)


val auth_none : auth_method
  (* The authentication method that does not perform authentication. *)


val set_auth_methods : t -> auth_method list -> unit
  (* Set the authentication methods for this client. The passed methods
   * are tried in turn until a method is accepted by the server.
   * The default is [ auth_none ]
   *)

val verbose : bool -> unit

  (* set whether you want debug messages or not *)
  (* Caution! This function is not protected against concurrent invocations
   * from several threads.
   *)
