(* $Id: server.ml 182 2004-05-25 16:49:11Z gerd $
 * ----------------------------------------------------------------------
 *
 *)

(* Server:
 *
 * In order to define a server the various remote procedure numbers must be
 * bound to real O'Caml functions. There are two ways of doing this:
 * - binding as synchronous call: The function returns immediately the result.
 * - binding as asynchronous call: The function receives the call but need
 *     not to produce a reply immediately. The function can do further I/O,
 *     for example call another remote procedure without blocking the server,
 *     and reply when the answer is available.
 *
 * Here, the procedure "plus1" and "sortarray" are implemented as synchronous
 * calls while "sortlist" works in an asynchronous way. "sortlist" first
 * converts its argument list into an array, calls then "sortarray" and
 * converts the result back. (This is a bit artificial...)
 *)

open Rtypes
open Xdr
open Rpc
open Rpc_server
open Simple_aux

let server_port = ref 0;;
  (* this variable will contain the Internet port where this server is
   * listening
   *)



(***** Implementation of the procedures *****)



let plus1 session n reply =
  (* Note that 'n' is a 32 bit signed integer. On 32 bit architectures,
   * such numbers cannot always be represented as O'Caml uses 31 bit signed
   * integers. So the following might generate a 'Cannot_represent' exception.
   * Note that this exception does not terminate the server.
   *)

  reply(n + 1)
;;


let sortarray session l reply =
  Sort.array ( <= ) l;
  reply l
;;


let sortlist session l (reply : Simple_aux.t_P'V1'sortlist'res -> unit) =
  (* some conversion functions: *)

  let rec convert_to_plain_list l =
    match l with
      None -> []
    | Some { value = v; next = l' } ->
	v :: convert_to_plain_list l'
  in

  let rec convert_to_xdr_list l =
    match l with
      []    -> None
    | x::l' -> Some { value = x; next = convert_to_xdr_list l' }
  in

  let l_as_array = Array.of_list (convert_to_plain_list l) in

  let esys = Rpc_server.get_event_system session in
  (* the event system behind the server *)

  (* Call this server recursively. To do so, we need a 'client' that is
   * attached to the same event system as the server.
   *)

  let client =
    Simple_clnt.P.V1.create_client
      ~esys:esys
      (Rpc_client.InetNumeric (127,0,0,1, !server_port))
      Tcp
  in

  Rpc_client.configure client 0 10.0;        (* set a timeout of 10 seconds *)

  (* Push the call onto the queue: *)

  Simple_clnt.P.V1.sortarray'async
    client
    l_as_array
    (fun get_result ->
      (* This function is called when the result has been arrived. *)
      try
	(* Obtain the result and convert it back: *)
	let result = get_result () in
	let result_as_xdr_list = convert_to_xdr_list (Array.to_list result) in
	(* Push the result onto the reply queue: *)
	reply result_as_xdr_list;
	(* The client is not needed any longer: *)
	Rpc_client.shut_down client;
      with
	any_exception ->
	  (* Shut down the client in this case, too: *)
	  Rpc_client.shut_down client;
	  (* Print the exception: *)
	  prerr_endline ("sortlist exception: " ^
			 Printexc.to_string any_exception);
	  raise any_exception)

  (* This was all. The rest is done in an event-driven way. *)
;;



(***** Building a server *****)



let main() =
  let esys = Unixqueue.create_unix_event_system() in

     (* esys: the "event system", i.e. the means where events arrive and are
      * processed by forwarding them to event handlers. "esys" contains an
      * event queue of unprocessed events so far, a set of handlers and
      * a set of resources which are conditions on file descriptors producing
      * events.
      *)

  let server =
    Simple_srv.P.V1.create_async_server
      ~proc_plus1: plus1
      ~proc_sortlist: sortlist
      ~proc_sortarray: sortarray
      Rpc_server.Portmapped          (* register with the portmapper *)
      Tcp
      Socket
      esys
  in

  (* Set signal handler. Signals are the only way to stop the server;
   * the default behaviour does not clean up the server, so we define
   * appropriate handlers.
   * Clean-up to do is mostly unregistering the program with the portmapper.
   *)

  List.iter
    (fun signal ->
      Sys.set_signal
        signal
	(Sys.Signal_handle (fun _ -> Rpc_server.stop_server server)))
    [ Sys.sighup; Sys.sigint; Sys.sigquit; Sys.sigterm ];

  Sys.set_signal
    Sys.sigpipe
    Sys.Signal_ignore;

  (* Initialize the 'server_port' variable *)
  server_port := Rpc_portmapper.port_of_program
                   Simple_aux.program_P'V1 "localhost" Tcp;

  (* Now start serving *)

  let rec auto_restart f arg =
    try f arg
    with err ->
      prerr_endline ("Server: Uncaught exception: " ^ Printexc.to_string err);
      auto_restart f arg
  in

  auto_restart Unixqueue.run esys
;;


(***** running the server *****)


(* Rpc_server.verbose true; *)
main();;
