<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    hvmenu-defs.php                                         */
/* Author:      Paul Waite                                              */
/* Description: Definitions for a Javascript-based multi-level menu.    */
/*              The system creates a semi-static config file in a       */
/*              dynamic manner and the javascript content is generated  */
/*              from that. The semi-static file (menu-vars.js) is only  */
/*              rebuilt when either the stylesheet or the menu data in  */
/*              the database is changed. The stylesheet should contain  */
/*              .menu and .submenu classes as per the default Axyl      */
/*              installation.                                           */
/*                                                                      */
/* ******************************************************************** */
/** @package menu */

/** Menu classes */
include_once("menu-defs.php");

// ----------------------------------------------------------------------
/**
* The HV Menu Javascript variables file. This file is
* dynamically generated by this class.
* @access private
*/
define("MENU_VARS_FILE", "menu_vars.js");
/**
* HVMenu class
* This module is being retained for legacy applications which still use
* the core javascript system (version 5.41) which was written by
* Ger Versluis 24 December 2001.
* @deprecated HVmenu javascript no longer packaged with Axyl. Use Xmenu instead.
* @package menu
*/
class HVmenu extends RenderableObject {
  // Public
  /** Menu name eg: 'main' */
  var $menu_name = "";
  /** Menu language */
  var $language = 0;
  /** Name of stylesheet file to get menu styles from */
  var $stylesheet = "";

  // Private
  /** Unique database menu ID
      @access private */
  var $menu_id = 0;
  /** Path to menu variables file - theme aware
      @access private */
  var $menuvarsfile = "";
  /** WWW menu URL - theme aware
      @access private */
  var $menuhref = "";
  /** Positioned relatively, or not
      @access private */
  var $relative_positioned = false;
  /** Level of menu to render
      @access private */
  var $menu_level = 0;
  /** Width of all menu items in pixels
      @access private */
  var $item_width = 150;
  /** Height of menu items in pixels
      @access private */
  var $item_height = 18;
  // ....................................................................
  /**
  * Constructor
  * Create the HV menu object.
  * @param string  $menu_name  Menu name
  * @param object  $webpage    Webpage object that this menu is being created for
  * @param string  $stylsheet  Name of stylesheet file to reference for menu styles
  * @param integer $lang       Optional language variant of this menu (zero = default)
  */
  function HVmenu($menu_name="main", $webpage=false, $stylesheet="", $lang=-1) {
    global $RESPONSE;
    $this->menu_name = $menu_name;
    // Set the menu language..
    if ($lang != -1) {
      $this->language = $lang;
    }
    elseif ($webpage !== false && $webpage->multilang && isset($webpage->languages[0])) {
      $this->language = $webpage->languages[0];
    }
    elseif (isset($RESPONSE) && $RESPONSE->multilang && isset($RESPONSE->languages[0])) {
      $this->language = $RESPONSE->languages[0];
    }
    // Get menu, fall back to default if not found..
    $tryagain = true;
    do {
      $tryagain = ($this->language != 0);
      $q  = "SELECT menu_id";
      $q .= "  FROM ax_menu";
      $q .= " WHERE menu_name='" . addslashes($this->menu_name) . "'";
      $q .= "   AND lang_id=$this->language";
      $mID = dbrecordset($q);
      if ($mID->hasdata) {
        $tryagain = false;
      }
      else {
        debugbr("menu language not found ($this->language): falling back to default", DBG_DEBUG);
        $this->language = 0;
      }
    } while ($tryagain);

    if ($mID->hasdata) {
      $this->menu_id = $mID->field("menu_id");
    }
    if ($webpage) {
      $this->stylesheet = $webpage->site_docroot . $webpage->head->stylesheet;
      global $CMDIR;
      $this->menuvarsfile = $webpage->site_docroot . "$CMDIR/" . $webpage->theme . "_" . MENU_VARS_FILE;
      $this->menuhref = "$CMDIR/" . $webpage->theme . "_" . MENU_VARS_FILE;
      $this->display_in_webpage($webpage);
    }
    // Over-ridden stylesheet to use..
    if ($stylesheet != "") {
      $this->stylesheet = $webpage->site_docroot . "/" . $stylesheet;
    }
  } // HVmenu
  // ....................................................................
  /**
  * Requires build
  * Check if the database records containing the menu
  * have been modified since the last modification time
  * of the menu vars file. Returns true if so.
  * @return bool True if menu requires build (config has been changed)
  * @access private
  */
  function requires_build() {
    // First check existence and file timestamp of the
    // menu setup file that is used by the system..
    if (!file_exists($this->menuvarsfile)) {
      debugbr("rebuilding: vars file absent", DBG_DEBUG);
      return true;
    }
    clearstatcache();
    $mvarmod_ts = filemtime($this->menuvarsfile);
    $mvarmod_dt = timestamp_to_datetime($mvarmod_ts);

    // Now check file mod time vs database mod times..
    $q  = "SELECT COUNT(*) AS tot FROM ax_menu";
    $q .= " WHERE menu_id='$this->menu_id'";
    $q .= "   AND last_modified > '$mvarmod_dt'";
    $modQ = dbrecordset($q);
    if ($modQ->field("tot") > 0) {
      return true;
    }
    $q  = "SELECT COUNT(*) AS tot FROM ax_menuoption";
    $q .= " WHERE menu_id='$this->menu_id'";
    $q .= "   AND last_modified > '$mvarmod_dt'";
    $modQ = dbrecordset($q);
    if ($modQ->field("tot") > 0) {
      return true;
    }
    // If stylesheet file is provided, then check against
    // that also, since this influences things like fonts
    // and colours etc..
    if ($this->stylesheet != "") {
      $stylemod_ts = filemtime($this->stylesheet);
      if ($stylemod_ts > $mvarmod_ts) {
        return true;
      }
    }
    // If we got here, no need to build..
    return false;
  } // requires build
  // ....................................................................
  /**
  * Display in webpage
  * Inserts the javascript necessary to embed the menu into a given webpage.
  * NB: Normally the webpage passed in here is $RESPONSE.
  * @param object $webpage Webpage object that this menu is being created for
  */
  function display_in_webpage($webpage) {
    global $LIBDIR;
    $webpage->head->add_scriptsrc($this->menuhref);
    $webpage->head->add_script("var TargetLoc='menu_$this->menu_id';\n");
    $webpage->head->add_script($this->menu_structure());
    $webpage->body->add_script("function Go(){return;}\n");
    $webpage->body->add_scriptsrc("$LIBDIR/js/menu_com.js");
  } // display_in_webpage
  // ....................................................................
  /**
  * This renders the field as HTML.
  * Inserts the HTML DIV tag which the HVmenu will use to position
  * itself to. The name of the DIV is taken from the unique menu ID,
  * and corresponds to the TargetLoc variable defined above..
  * @return string The menu anchor point (DIV) as HTML.
  */
  function html() {
    // Always build menu variables file if required..
    if ($this->requires_build()) {
      $this->build();
    }
    if ($this->relative_positioned) {
      return "<div id=\"menu_$this->menu_id\" style=\"position:relative;\"></div>";
    }
    else {
      return "<div id=\"menu_$this->menu_id\"></div>";
    }
  } // html
  // ....................................................................
  /**
  * Build menu config
  * Build the static menu_vars.js file which is used to set menu
  * look and feel, driven by the given stylesheet. This is a physical
  * file which gets (re-)built and saved to disk.
  * @access private
  */
  function build() {
    global $RESPONSE;
    global $LIBDIR;
    // Get fonts and colours etc. If this stylesheet is undefined or
    // doesn't exist on disk, then we should still be able to generate
    // the vars below, but will get the defaults in each case..
    $ss = new stylesheet($this->stylesheet);
    // Read in all the style settings..
    $FontLowColor         = defaulted($ss->style("menu",              "color"),               "#efefef");
    $LowBgColor           = defaulted($ss->style("menu",              "background-color"),    "black");
    $FontHighColor        = defaulted($ss->style("menu_highlight",    "color"),               "white");
    $HighBgColor          = defaulted($ss->style("menu_highlight",    "background-color"),    "grey");
    $BorderColor          = defaulted($ss->style("menu",              "border-color"),        "black");
    $MenuTextCentered     = defaulted($ss->style("menu",              "text-align"),          "left");
    $MenuCentered         = defaulted($ss->style("menu",              "menu-align"),          "left");
    $MenuVerticalCentered = defaulted($ss->style("menu",              "menu-vertical-align"), "top");
    $FontFamily           = defaulted($ss->style("menu",              "font-family"),         "arial");
    $FontSubLowColor      = defaulted($ss->style("submenu",           "color"),               "#efefef");
    $LowSubBgColor        = defaulted($ss->style("submenu",           "background-color"),    "black");
    $FontSubHighColor     = defaulted($ss->style("submenu_highlight", "color"),               "white");
    $HighSubBgColor       = defaulted($ss->style("submenu_highlight", "background-color"),    "grey");
    $BorderSubColor       = defaulted($ss->style("submenu",           "border-color"),        "black");

    // Elements with superfluous bits to be removed..
    $val                  = defaulted($ss->style("menu",    "menu-frame"),            "self");
    $FirstLineFrame       = str_replace("'", "", $val);
    $val                  = defaulted($ss->style("menu",    "submenu-frame"),         "self");
    $SecLineFrame         = str_replace("'", "", $val);
    $val                  = defaulted($ss->style("menu",    "content-frame"),         "self");
    $DocTargetFrame       = str_replace("'", "", $val);
    $val                  = defaulted($ss->style("menu",    "margin-top"),            "0");
    $StartTop             = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("menu",    "margin-left"),           "0");
    $StartLeft            = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("menu",    "vertical-correction"),   "0");
    $VerCorrect           = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("menu",    "horizontal-correction"), "0");
    $HorCorrect           = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("menu",    "padding-left"),          "0");
    $LeftPaddng           = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("menu",    "padding-top"),           "0");
    $TopPaddng            = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("menu",    "font-size"),             "8");
    $FontSize             = str_replace("pt", "", $val);
    $val                  = defaulted($ss->style("menu",    "disappear-delay"),       "100");
    $DisappearDelay       = str_replace("ms", "", $val);
    $val                  = defaulted($ss->style("menu",    "border-width"),          "1");
    $BorderWidth          = str_replace("px", "", $val);
    $val                  = defaulted($ss->style("menu",    "item-spacing"),          "0");
    $BorderBtwnElmnts     = str_replace("px", "", $val);

    // Real Numbers..
    $ChildOverlap = defaulted($ss->style("menu", "child-overlap"), "0.2");
    if ($ChildOverlap > 1.0) $ChildOverlap = 1.0;
    if ($ChildOverlap < 0.0) $ChildOverlap = 0.0;

    $ChildVerticalOverlap = defaulted($ss->style("menu", "child-vertical-overlap"), "0.2");
    if ($ChildVerticalOverlap > 1.0) $ChildVerticalOverlap = 1.0;
    if ($ChildVerticalOverlap < 0.0) $ChildVerticalOverlap = 0.0;
    $ChildVerticalOverlap = (1.0 - $ChildVerticalOverlap);

    // Integers, Logicals, et al..
    $val = defaulted($ss->style("menu", "font-weight"), "normal");
    if ($val == "bold") $FontBold = "1";
    else $FontBold = "0";

    $val = defaulted($ss->style("menu", "font-style"), "normal");
    if ($val == "italic") $FontItalic = "1";
    else $FontItalic = "0";

    $val = defaulted($ss->style("menu", "orientation"), "horizontal");
    if ($val == "horizontal") $FirstLineHorizontal = "1";
    else $FirstLineHorizontal = "0";

    $val = defaulted($ss->style("menu", "frames-columns"), "no");
    if ($val == "yes") $MenuFramesVertical = "1";
    else $MenuFramesVertical = "0";

    $val = defaulted($ss->style("menu", "takeover-background-color"), "no");
    if ($val == "yes") $TakeOverBgColor = "1";
    else $TakeOverBgColor = "0";

    $val = defaulted($ss->style("menu", "hide-top"), "no");
    if ($val == "yes") $HideTop = "1";
    else $HideTop = "0";

    $val = defaulted($ss->style("menu", "wrap"), "no");
    if ($val == "yes") $MenuWrap = "1";
    else $MenuWrap = "0";

    $val = defaulted($ss->style("menu", "wrap"), "no");
    if ($val == "yes") $MenuWrap = "1";
    else $MenuWrap = "0";

    $val = defaulted($ss->style("menu", "right-to-left"), "yes");
    if ($val == "yes") $RightToLeft = "1";
    else $RightToLeft = "0";

    $val = defaulted($ss->style("menu", "unfold-on-click"), "no");
    if ($val == "yes") $UnfoldsOnClick = "1";
    else $UnfoldsOnClick = "0";

    $val = defaulted($ss->style("menu", "debug"), "no");
    if ($val == "yes") $WebMasterCheck = "1";
    else $WebMasterCheck = "0";

    $val = defaulted($ss->style("menu", "show-arrows"), "no");
    if ($val == "yes") $ShowArrow = "1";
    else $ShowArrow = "0";

    $val = defaulted($ss->style("menu", "show-status"), "no");
    if ($val == "yes") $ShowStatus = "1";
    else $ShowStatus = "0";

    // This can take values: 'url' or 'description'. It determines
    // what is shown in the browser status bar..
    $StatusContent = defaulted($ss->style("menu", "status-content"), "url");

    $val = defaulted($ss->style("menu", "keep-highlight"), "no");
    if ($val == "yes") $KeepHilite = "1";
    else $KeepHilite = "0";

    $val = defaulted($ss->style("menu", "relative-positioned"), "no");
    if ($val == "yes") $this->relative_positioned = true;
    else $this->relative_positioned = false;

    // Write the variables file out..
    $newMV = new outputfile($this->menuvarsfile);
    if ($newMV->opened) {
      $newMV->writeln("//HVMENU (c)2001 Ger Versluis");
      $newMV->writeln("var LowBgColor='$LowBgColor';");
      $newMV->writeln("var LowSubBgColor='$LowSubBgColor';");
      $newMV->writeln("var HighBgColor='$HighBgColor';");
      $newMV->writeln("var HighSubBgColor='$HighSubBgColor';");
      $newMV->writeln("var FontLowColor='$FontLowColor';");
      $newMV->writeln("var FontSubLowColor='$FontSubLowColor';");
      $newMV->writeln("var FontHighColor='$FontHighColor';");
      $newMV->writeln("var FontSubHighColor='$FontSubHighColor';");
      $newMV->writeln("var BorderColor='$BorderColor';");
      $newMV->writeln("var BorderSubColor='$BorderSubColor';");
      $newMV->writeln("var BorderWidth=$BorderWidth;");
      $newMV->writeln("var BorderBtwnElmnts=$BorderBtwnElmnts;");
      $newMV->writeln("var FontFamily='$FontFamily';");
      $newMV->writeln("var FontSize='$FontSize';");
      $newMV->writeln("var FontBold=$FontBold;");
      $newMV->writeln("var FontItalic=$FontItalic;");
      $newMV->writeln("var MenuTextCentered='$MenuTextCentered';");
      $newMV->writeln("var MenuCentered='$MenuCentered';");
      $newMV->writeln("var MenuVerticalCentered='$MenuVerticalCentered';");
      $newMV->writeln("var ChildOverlap=$ChildOverlap;");
      $newMV->writeln("var ChildVerticalOverlap=$ChildVerticalOverlap;");
      $newMV->writeln("var StartTop=$StartTop;");
      $newMV->writeln("var StartLeft=$StartLeft;");
      $newMV->writeln("var VerCorrect=$VerCorrect;");
      $newMV->writeln("var HorCorrect=$HorCorrect;");
      $newMV->writeln("var LeftPaddng=$LeftPaddng;");
      $newMV->writeln("var TopPaddng=$TopPaddng;");
      $newMV->writeln("var FirstLineHorizontal=$FirstLineHorizontal;");
      $newMV->writeln("var MenuFramesVertical=$MenuFramesVertical;");
      $newMV->writeln("var DisappearDelay=$DisappearDelay;");
      $newMV->writeln("var TakeOverBgColor=$TakeOverBgColor;");
      $newMV->writeln("var FirstLineFrame='$FirstLineFrame';");
      $newMV->writeln("var SecLineFrame='$SecLineFrame';");
      $newMV->writeln("var DocTargetFrame='$DocTargetFrame';");
      $newMV->writeln("var HideTop=$HideTop;");
      $newMV->writeln("var MenuWrap=$MenuWrap;");
      $newMV->writeln("var RightToLeft=$RightToLeft;");
      $newMV->writeln("var UnfoldsOnClick=$UnfoldsOnClick;");
      $newMV->writeln("var WebMasterCheck=$WebMasterCheck;");
      $newMV->writeln("var ShowArrow=$ShowArrow;");
      $newMV->writeln("var ShowStatus=$ShowStatus;");
      $newMV->writeln("var StatusContent='$StatusContent';");
      $newMV->writeln("var KeepHilite=$KeepHilite;");
      $newMV->writeln("var Arrws=['$LIBDIR/img/_tri.gif',5,10,'$LIBDIR/img/_tridown.gif',10,5,'$LIBDIR/img/_trileft.gif',5,10];");
      $newMV->writeln("function BeforeStart(){return}");
      $newMV->writeln("function AfterBuild(){return}");
      $newMV->writeln("function BeforeFirstOpen(){return}");
      $newMV->writeln("function AfterCloseAll(){return}");
      // Finish up..
      $newMV->closefile();
    } // newMV file opened
  } // build
  // ....................................................................
  /**
  * Menu structure
  * This function returns a set of Javascript array definitions which
  * should be put into the page which will be hosting the menu. Note
  * that this will be built for each user, since each user has a
  * unique auth_code which might be a part of the menu..
  * @return string Javascript var definitions for menu structure
  * @access private
  */
  function menu_structure() {
    global $RESPONSE;
    // Menu Tree
    // MenuX=new Array(
    //    Text to show,
    //    Link,
    //    background image (optional),
    //    number of sub elements,
    //    height,
    //    width
    //    );
    // For rollover images set "Text to show" to:
    //    "rollover:Image1.jpg:Image2.jpg"
    $js = "";
    $q  = "SELECT *";
    $q .= "  FROM ax_menu m, ax_menuoption mo";
    $q .= " WHERE m.menu_id=$this->menu_id";
    $q .= "   AND mo.menu_id=m.menu_id";
    $q .= "   AND m.active=TRUE";
    $q .= "   AND mo.active=TRUE";
    $q .= " ORDER BY mo.menu_level,mo.display_order,mo.parent_id";
    $item = dbrecordset($q);
    if ($item->hasdata) {
      $topcount = 0;
      do {
        $mopid        = $item->field("menuoption_id");
        $mnu_ugroups  = $item->field("user_groups");
        $mnu_usertype = $item->field("user_type");
        if ($mnu_ugroups == "" || $RESPONSE->ismemberof_group_in($mnu_ugroups)) {
          if ($mnu_usertype == "" || ($RESPONSE->user_type == $mnu_usertype)) {
            $parent_id   = $item->field("parent_id");
            $menu_level  = $item->field("menu_level");
            $label       = $item->field("label");
            $description = $item->field("description");
            $action      = $item->field("action");
            $target      = $item->field("target");
            $authcode    = $item->istrue("auth_code");
            $width       = $item->field("width");
            $height      = $item->field("height");
            $is_parent   = $item->istrue("is_parent");
            if ($menu_level == 0) {
              $topcount += 1;
              $mno_top[$topcount] = $mopid;
            }
            $mno_details[$mopid] = "$label|$description|$action|$target|" . ($authcode ? "t" : "f") . "|$width|$height";
            $mno_level[$mopid]   = $menu_level;
            $mno_parent[$mopid]  = $parent_id;
            if ($parent_id != "") {
              if (isset($mno_children[$parent_id])) {
                $mno_children[$parent_id] .= "|";
              }
              $mno_children[$parent_id] .= $mopid;
            }
          } // user type check
        } // memberof
      } while ($item->get_next());

      // Number of top-level items..
      $js .= "var NoOffFirstLineMenus=$topcount;\n";

      if (is_array( $mno_top )) {
        while (list($item_no, $mopid) = each($mno_top)) {
          $prefix = "$item_no";
          $js .= HVmenu_entry($prefix, $mopid, $mno_children, $mno_details);
        }
      }
    } // hasdata
    return $js;
  } // menu_structure
} // HVmenu class

// ----------------------------------------------------------------------
/**
* Recursively produce HVmenu definition entries. These array
* definitions define the menu structure in terms of the actual
* menus and their sub-menus. This routine produces a single
* menu-option definition, but will recursively call all child
* (sub-menu) definitions.
@param string  $prefix          The existing menu level prefix
@param string  $mopid           Menu option ID
@param string  $mno_children    List of children IDs of a menu option
@param string  $mno_details     Menu option details
@access private
*/
function HVmenu_entry($prefix, $mopid, $mno_children, $mno_details) {
  global $RESPONSE;
  $childcount = 0;
  $js = "";
  if (isset($mno_children[$mopid]) && $mno_children[$mopid] != "") {
    $childoptions = explode("|", $mno_children[$mopid]);
    $childcount = count($childoptions);
    $pfxcount = 1;
    foreach ($childoptions as $childmopid) {
      $childprefix = $prefix . "_" . $pfxcount;
      $js .= HVmenu_entry($childprefix, $childmopid, $mno_children, $mno_details);
      $pfxcount += 1;
    }
  }
  // Now finally write this current menu option to file..
  $details = explode("|", $mno_details[$mopid]);
  $label = $details[0];
  if ($label == MENU_ITEM_SPACER) {
    $label = "";
  }
  $description = $details[1];
  $action      = $details[2];
  $target      = $details[3];
  $authcode    = $details[4];
  $width       = $details[5];
  $height      = $details[6];
  if ($authcode == "t") {
    if (strstr($action, "?")) $action .= "&";
    else $action .= "?";
    $action .= "auth_code=" . $RESPONSE->get_auth_code();
  }
  // Make menu entry..
  $js .= "Menu" . $prefix . "=new Array(\"$label\",\"$action\",\"$description\",\"\",$childcount,$height,$width);\n";
  // Return the javascript..
  return $js;
}

// ----------------------------------------------------------------------
?>