/****************************************************************
**
** Attal : Lords of Doom
**
** mapDispositionEditor.cpp
** Edit map disposition
**
** Version : $Id: mapDispositionEditor.cpp,v 1.6 2004/09/19 19:43:34 lusum Exp $
**
** Author(s) : Pascal Audoux
**
** Date : 22/07/2004
**
** Licence :
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/

#include "mapDispositionEditor.h"

// generic include files
// include files for QT
#include <qbuttongroup.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpoint.h>
#include <qptrlist.h>
#include <qpushbutton.h>
#include <qspinbox.h>
// application specific include files
#include "libCommon/genericMapDisposition.h"
#include "libCommon/log.h"

#include "libClient/gui.h"

extern QString IMAGE_PATH;

//
// ----- MapDispositionEditor -----
//

MapDispositionEditor::MapDispositionEditor( QWidget * parent, const char * name )
: QWidget( parent, name )
{
	_mapDisposition = 0;

	QButtonGroup * group = new QButtonGroup( this );



	QPushButton * pbFree = new QPushButton( group );
	pbFree->setText( tr( "Free" ) );
	pbFree->setToggleButton( true );
	pbFree->setOn( true );
	FIXEDSIZE( pbFree );

	QPushButton * pbOccupied = new QPushButton( group );
	pbOccupied->setText( tr( "Occupied" ) );
	pbOccupied->setToggleButton( true );
	FIXEDSIZE( pbOccupied );

	QPushButton * pbDoor = new QPushButton( group );
	pbDoor->setText( tr( "Door" ) );
	pbDoor->setToggleButton( true );
	FIXEDSIZE( pbDoor );

	group->setExclusive( true );
	group->insert( pbFree, 0 );
	group->insert( pbOccupied, 1 );
	group->insert( pbDoor, 2 );

	QVBoxLayout * layV1 = new QVBoxLayout( group );
	layV1->setMargin( 5 );
	layV1->addStretch( 1 );
	layV1->addWidget( pbFree );
	layV1->addStretch( 1 );
	layV1->addWidget( pbOccupied );
	layV1->addStretch( 1 );
	layV1->addWidget( pbDoor );
	layV1->addStretch( 1 );
	layV1->activate();
	
	QLabel * labCol = new QLabel( this );
	labCol->setText( tr( "Columns:" ) );
	FIXEDSIZE( labCol );
	
	QLabel * labRow = new QLabel( this );
	labRow->setText( tr( "Rows:" ) );
	FIXEDSIZE( labRow );
	
	QVBoxLayout * layV2 = new QVBoxLayout();
	layV2->setMargin( 5 );
	layV2->addStretch( 1 );
	layV2->addWidget( labCol );
	layV2->addStretch( 1 );
	layV2->addWidget( labRow );
	layV2->addStretch( 1 );
	
	_spinCol = new QSpinBox( this );
	_spinCol->setMinValue( 1 );
	_spinCol->setValue( 5 );
	FIXEDSIZE( _spinCol );
	
	_spinRow = new QSpinBox( this );
	_spinRow->setMinValue( 1 );
	_spinRow->setValue( 5 );
	FIXEDSIZE( _spinRow );
	
	
	QVBoxLayout * layV3 = new QVBoxLayout();
	layV3->setMargin( 5 );
	layV3->addStretch( 1 );
	layV3->addWidget( _spinCol );
	layV3->addStretch( 1 );
	layV3->addWidget( _spinRow );
	layV3->addStretch( 1 );

	_dispositionCanvas = new MapDispositionCanvas( this );
	_dispositionCanvasView = new MapDispositionCanvasView( _dispositionCanvas, this );

	QHBoxLayout * layout = new QHBoxLayout( this );
	layout->setMargin( 5 );
	layout->setSpacing( 5 );
	layout->addWidget( group );
	layout->addStretch( 1 );
	layout->addWidget( _dispositionCanvasView );
	layout->addLayout( layV2 );
	layout->addLayout( layV3 );
	layout->addStretch( 1 );
	layout->activate();

	connect( group, SIGNAL( clicked( int ) ), SLOT( slot_clicked( int ) ) );
	connect( _spinCol, SIGNAL( valueChanged( int ) ), SLOT( slot_resize() ) );
	connect( _spinRow, SIGNAL( valueChanged( int ) ), SLOT( slot_resize() ) );
}

void MapDispositionEditor::slot_clicked( int id )
{
	_dispositionCanvas->setCurrentType( ( GenericMapDisposition::DispositionType )id );
}

void MapDispositionEditor::slot_resize()
{
	_dispositionCanvas->clear();
	_mapDisposition->resize( _spinRow->value(), _spinCol->value() );
	_dispositionCanvas->init( _mapDisposition, _pixmapPath );
	_dispositionCanvas->resize( 30 * _mapDisposition->getWidth(), 30 * _mapDisposition->getHeight() );
	_dispositionCanvasView->setMinimumSize( 30 * _mapDisposition->getWidth(), 30 * _mapDisposition->getHeight() );
	//_dispositionCanvas->reinit();
	update();
}

void MapDispositionEditor::init( GenericMapDisposition * mapDisposition, const QString & pixmapPath )
{
	int width, height;
	_mapDisposition = mapDisposition;
	_pixmapPath = pixmapPath;
	_dispositionCanvas->init( mapDisposition, pixmapPath );
	if( mapDisposition ) {
		_dispositionCanvas->resize( 30 * mapDisposition->getWidth(), 30 * mapDisposition->getHeight() );
		width = mapDisposition->getWidth();
		height = mapDisposition->getHeight();
		_dispositionCanvasView->resize( 30 * mapDisposition->getWidth() + 10, 30 * mapDisposition->getHeight() + 10 );
		/* we need to use width and height on this point cause using directly mapDisposition->getWidth() 
		*  and mapDisposition->getHeigth() is not possible because _spinCol and _spinRow 
		*  call (by a signal) a function that change the value of mapDisposition 
		* in this case _spinCol change the value of mapDisposition->getHeigth() before _spinRow
		*/
		_spinCol->setValue( width );
		_spinRow->setValue( height );
		setMinimumHeight( 30 * height + 10 );
		_dispositionCanvas->reinit();
		update();
	}
}

void MapDispositionEditor::clear()
{
	logEE( "Not yet implemented." );
}

//
// ----- MapDispositionCanvas -----
//
MapDispositionCanvas::MapDispositionCanvas( QObject * parent, const char * name )
: QCanvas( parent , name )
{
	resize( 150, 150 );
	setBackgroundColor( Qt::white );

	QPtrList<QPixmap> listPixmaps;
	listPixmaps.setAutoDelete( true );
	QPtrList<QPoint> listPoints;
	listPoints.setAutoDelete( true );

	listPixmaps.append( new QPixmap( IMAGE_PATH + "misc/editorFree.png" ) );
	listPoints.append( new QPoint( 0, 0 ) );
	listPixmaps.append( new QPixmap( IMAGE_PATH + "misc/editorOccupied.png" ) );
	listPoints.append( new QPoint( 0, 0 ) );
	listPixmaps.append( new QPixmap( IMAGE_PATH + "misc/editorDoor.png" ) );
	listPoints.append( new QPoint( 0, 0 ) );

	_pixmaps = new QCanvasPixmapArray( listPixmaps, listPoints );
	_dispositionSprites = 0;
	_disposition = 0;
	_background = 0;
}

MapDispositionCanvas::~MapDispositionCanvas()
{
	clear();
}

void MapDispositionCanvas::clear()
{
	uint height, width;
	uint i, j;

	if( _background ) {
		delete _background;
		_background = 0;
	}
	
	if( _disposition && _dispositionSprites ) {
		height = _disposition->getHeight();
		width = _disposition->getWidth();
		for( i = 0; i < height; i++ ) {
			for( j = 0; j < width; j++ ) {
				delete _dispositionSprites[ i ][ j ];
				_dispositionSprites[ i ][ j ] = 0;
			}
			delete [] _dispositionSprites[ i ];
		}
		delete [] _dispositionSprites;
		_dispositionSprites = 0;
	}
	
	setAllChanged();
	update();
}


void MapDispositionCanvas::init( GenericMapDisposition * mapDisposition, const QString & pixmapPath )
{
	uint height, width;
	uint i, j;
	
	clear();
	
	_disposition = mapDisposition;
	height = _disposition->getHeight();
	width = _disposition->getWidth();
	_background = new DispositionSprite( this );
	_background->setBackground( pixmapPath );
	_background->move( 0, (height*30) - _background->height() );


	_dispositionSprites = new DispositionSprite ** [ height ];
	for( i = 0; i < height; i++ ) {
		_dispositionSprites[ i ] = new DispositionSprite * [ width ];
		for( j = 0; j < width; j++ ) {
			_dispositionSprites[ i ][ j ] = new DispositionSprite( this );
			/// XXX: change 30 by const
			_dispositionSprites[ i ][ j ]->move( j * 30, i * 30 );
			_dispositionSprites[ i ][ j ]->setType( _disposition->getDisposition( i, j ) );
		}
	}
	setAllChanged();
	update();
}

void MapDispositionCanvas::reinit()
{
	uint height = _disposition->getHeight();
	uint width = _disposition->getWidth();

	for( uint i = 0; i < height; i++ ) {
		for( uint j = 0; j < width; j++ ) {
			_dispositionSprites[ i ][ j ]->setType( _disposition->getDisposition( i, j ) );
		}
	}
	
	setAllChanged();
	update();
}

void MapDispositionCanvas::changeCell( uint row, uint col )
{
	_disposition->setDisposition( row, col, _currentType );
	_dispositionSprites[ row ][ col ]->setType( _currentType );
	setChanged( _dispositionSprites[ row ][ col ]->boundingRect() );
	reinit();
}


//
// ----- MapDispositionCanvasView -----
//

MapDispositionCanvasView::MapDispositionCanvasView( MapDispositionCanvas * canvas , QWidget * parent, const char * name, WFlags f )
: QCanvasView( (QCanvas*)canvas, parent, name, f )
{
	viewport()->setMouseTracking( true );
	_canvas = canvas;
}

void MapDispositionCanvasView::contentsMouseReleaseEvent( QMouseEvent * e )
{
	QCanvasItemList list = canvas()->collisions(e->pos());
	for( unsigned int i = 0; i < list.count(); i++ ) {
		if( list[i]->rtti() == DispositionSprite::RTTI ) {
			if( e->button() == LeftButton ) {
				uint col = e->x() / 30;
				uint row = e->y() / 30;
				_canvas->changeCell( row, col );
			}
		}
	}
}

//
// ----- DispositionSprite -----
//

const int DispositionSprite::RTTI = 61;

DispositionSprite::DispositionSprite( MapDispositionCanvas * canvas )
: QCanvasSprite( canvas->getPixmaps(), (QCanvas*)canvas )
{
	collisions( false );
	setEnabled( true );
	setFrame( 0 );
	setZ( 10 );
	show();
}

void DispositionSprite::setType( GenericMapDisposition::DispositionType type )
{
	setFrame( (uint)type );
}

void DispositionSprite::setBackground( const QString & pixmapPath )
{
	setZ( 5 );
	QPtrList<QPixmap> listPixmap;	
	listPixmap.setAutoDelete( true );
	listPixmap.append( new QPixmap( pixmapPath ) );
	QPtrList<QPoint> listPoint;
	listPoint.setAutoDelete( true );
	listPoint.append( new QPoint( 0, 0 ) );
	QCanvasPixmapArray * tryi = new QCanvasPixmapArray( listPixmap, listPoint );
	setSequence( tryi );
}



