/*
 * atanks - obliterate each other with oversize weapons
 * Copyright (C) 2003  Thomas Hudson
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the 
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License 
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * */

#include "environment.h"
#include "globaldata.h"
#include "explosion.h"
#include "missile.h"
#include "decor.h"
#include "tank.h"
#include "player.h"

EXPLOSION::EXPLOSION ()
{
	_align = LEFT;
	_current.x = 0;
	_current.y = 0;
	_current.w = 0;
	_current.h = 0;
	initialise();
}

EXPLOSION::~EXPLOSION ()
{
	_env->removeObject (this);
}

EXPLOSION::EXPLOSION (GLOBALDATA *global, ENVIRONMENT *env, double xpos, double ypos, int weaponType)
{
	initialise ();
	setEnvironment (env);
	player = NULL;
	_align = LEFT;
	_global = global;
	x = xpos;
	y = ypos;
	type = weaponType;
	if (type < WEAPONS)
		weap = &weapon[type];
	else
		weap = &naturals[type - WEAPONS];
	radius = weap->radius;
	etime = weap->etime;
	damage = weap->damage;
	eframes = weap->eframes;
}

EXPLOSION::EXPLOSION (GLOBALDATA *global, ENVIRONMENT *env, double xpos, double ypos, double xvel, double yvel, int weaponType)
{
	initialise ();
	setEnvironment (env);
	_align = LEFT;
	_global = global;
	x = xpos;
	y = ypos;
	xv = xvel;
	yv = yvel;
	angle = (int)(atan2 (xv, yv) / PI * 180);
	if (angle < 0)
		angle += 360;
	type = weaponType;
	if (type < WEAPONS)
		weap = &weapon[type];
	else
		weap = &naturals[type - WEAPONS];
	radius = weap->radius;
	etime = weap->etime;
	damage = weap->damage;
	eframes = weap->eframes;
}

void EXPLOSION::initialise ()
{
	PHYSICAL_OBJECT::initialise ();
	drag = 0.95;
	mass = 3;
	a = 1;
	peaked = 0;
	exclock = 500;
}

int EXPLOSION::applyPhysics ()
{
	if (type == NAPALM_JELLY) {
		if (rand () % 300 == 0) {
			DECOR *decor;
			decor = new DECOR (_global, _env, x, y, xv, yv, radius / 2, DECOR_SMOKE);
			if (!decor) {
				perror ("explosion.cc: Failed allocating memory for decor in applyPhysics");
				exit (1);
			}
		}
	}
/*	if (dispersing)
	{
		double accel = (_env->wind - xv) / mass * drag * _env->viscosity;
		xv += accel;
		x += xv;
		y += yv;
	}
*/
	return (hitSomething);
}


void EXPLOSION::explode ()
{
	int calcblow;
	double distance;
	int z;
	calcblow = 1;

	if (peaked == 1 && a <= EXPLOSIONFRAMES + 1) {
		exclock++;
		if (exclock > weap->etime) {
			exclock = 0;
			a++;
			requireUpdate ();
		}
		if (a > EXPLOSIONFRAMES + 1) {
			destroy = TRUE;
		}
	}
	if (a <= EXPLODEFRAMES + 1) {
		TANK *tank;
		if (a == 1) {
			for (int index = 0; (tank = (TANK*)_env->getNextOfClass (TANK_CLASS, &index)) && tank; index++) {
				// is tank directly above explosion?
				if ((fabs (x - tank->x) < radius) && (y - tank->y >= 0))
					tank->creditTo = player;
			}
		}
		if ((a == 1) && (type <= TECTONIC || type >= WEAPONS)) {
			for (int index = 0; (tank = (TANK*)_env->getNextOfClass (TANK_CLASS, &index)) && tank; index++) {
				if (type >= SHAPED_CHARGE && type <= CUTTER) {
					double ydist = fabs ((y - tank->y) * 20);
					if (ydist > TANKHEIGHT/2)
						ydist -= TANKHEIGHT/2;
					else
						ydist = 0;
					distance = sqrt (pow (x - tank->x, 2) + pow (ydist, 2));
				} else {
					distance = sqrt (pow (x - tank->x, 2) + pow (y - tank->y, 2));
				}
				if (distance <= (radius + TANKHEIGHT/2) && tank->l > 0) {
					_global->updateMenu = 1;
					tank->damage = (int) ((float) damage * ((float) 1 - ((fabs (distance) / (float)radius) / 2)) * player->damageMultiplier);

					tank->creditTo = player;
					tank->applyDamage ();
				}
			}
			if (type >= TREMOR && type <= TECTONIC) {
				angle = (int)(atan2 (yv, xv) / PI * 180);
				if (angle < 0)
					angle = angle + 360;
				angle = angle % 360;
			}
		}
		exclock++;
		if (exclock > weap->etime) {
			exclock = 0;
			a++;
			requireUpdate ();
		}
		if (a >= EXPLODEFRAMES + 1 && !peaked) {
			calcblow = 0;
		}
	}
	if (!calcblow) {
		if (type >= SHAPED_CHARGE && type <= CUTTER) {
			ellipsefill (_env->terrain, (int)x, (int)y, radius, radius / 20, PINK);
			setUpdateArea ((int)x - (radius + 1), (int)y - (radius / 20 + 1), (radius + 1) * 2, (radius / 20 + 1) * 2);

		} else if (type <= LAST_EXPLOSIVE || type >= WEAPONS) {

			if (type != NAPALM_JELLY) {

				circlefill (_env->terrain, (int)x, (int)y, radius, PINK);
				setUpdateArea ((int)x - (radius + 1), (int)y - (radius + 1), (radius + 1) * 2, (radius + 1) * 2);
			}
		} else if ((type >= RIOT_BOMB) && (type <= HVY_RIOT_BOMB)) {
			circlefill (_env->terrain, (int)x, (int)y, radius, PINK);
			setUpdateArea ((int)x - (radius + 1), (int)y - (radius + 1), (radius + 1) * 2, (radius + 1) * 2);
		} else if ((type >= RIOT_CHARGE) && (type <= RIOT_BLAST)) {
			double sx = x - _global->slope[angle][0] * 15;
			double sy = y - _global->slope[angle][1] * 15;
			triangle (_env->terrain, (int)sx, (int)sy, (int)(sx + _global->slope[(angle + 45) % 360][0] * radius), (int)(sy + _global->slope[(angle + 45) % 360][1] * radius),(int)(sx + _global->slope[(angle + 315) % 360][0] * radius),(int)(sy + _global->slope[(angle + 315) % 360][1] * radius), PINK);
			setUpdateArea ((int)sx - (radius + 1), (int)sy - (radius + 1), (radius + 1) * 2, (radius + 1) * 2);
		} else if ((type >= DIRT_BALL) && (type <= SUP_DIRT_BALL)) {
			BITMAP *tmp;		//for mixing
			
			tmp = create_bitmap(radius * 2, radius * 2);
			clear_to_color(tmp, PINK);
			for (int count = 0; count < radius ; count++) {
				circle (tmp, radius, radius, count, (player)?player->color:WHITE);
			}
			setUpdateArea ((int)x - (radius + 1), (int)y - (radius + 1), (radius + 1) * 2, (radius + 1) * 2);

			//copy terrain over explosion
			masked_blit(_env->terrain, tmp, (int)x - radius, (int)y - radius, 0, 0, radius*2, radius*2);

			//blit back exploded terrain
			masked_blit(tmp, _env->terrain, 0, 0,(int)x - radius, (int)y - radius, radius*2, radius*2);

			destroy_bitmap(tmp);
		}

		for (z = 0; z < _current.w; z++)
			setSlideColumnDimensions (_global, _env, _current.x + z, TRUE);
		calcblow = 1;
		peaked = 1;
		dispersing = 1;
	}
}

void EXPLOSION::draw (BITMAP *dest)
{
	if (type >= SHAPED_CHARGE && type <= CUTTER) {
		if (a > 1 && a <= EXPLOSIONFRAMES + 1) {
			rotate_scaled_sprite (dest, (BITMAP *) _global->gfxData.flameFront[(a + (EXPLOSIONFRAMES * weap->eframes)) - 2], (int)x - radius, (int)y - (radius / 20), itofix (0), ftofix ((double) radius / 300));
			setUpdateArea ((int)x - (radius + 1), (int)y - (radius / 20 + 1), (radius + 1) * 2, (radius / 20 + 1) * 2);
		}
	} else if (type == NAPALM_JELLY) {
		int blobSize = (int)(radius - ((double)a / EXPLOSIONFRAMES) * radius + 1);
		circlefill (_env->db, (int)x, (int)y, blobSize, makecol (255, 0, 0));
		setUpdateArea ((int)x - (radius + 1), (int)y - (radius + 1), (radius + 1) * 2, (radius + 1) * 2);
	} else if (type <= LAST_EXPLOSIVE || type >= WEAPONS) {
		if (a > 1 && a <= EXPLOSIONFRAMES + 1) {
			rotate_scaled_sprite (dest, (BITMAP *) _global->gfxData.explosions[(a + (EXPLOSIONFRAMES * weap->eframes)) - 2], (int)x - radius, (int)y - radius, itofix (0), ftofix ((double) radius / 107));
			setUpdateArea ((int)x - (radius + 1), (int)y - (radius + 1), (radius + 1) * 2, (radius + 1) * 2);
		}
	} else if (type <= TECTONIC) {
		if (a > 1 && a <= EXPLODEFRAMES + 1) {
			drawFracture (_global, _env, _env->terrain, &_current, (int)x, (int)y, angle, (int)(((double)a / EXPLODEFRAMES) * (radius / 4)), radius, 5, 0);
		}
	} else if ((type >= RIOT_BOMB) && (type <= HVY_RIOT_BOMB)) {
		if (a > 1 && a <= EXPLODEFRAMES + 1) {
			int startCirc = (radius / EXPLODEFRAMES) * a;
			circlefill (_env->terrain, (int)x, (int)y, startCirc, PINK);
			circle (dest, (int)x, (int)y, startCirc, player->color);
			setUpdateArea ((int)x - (radius + 1), (int)y - (radius + 1), (radius + 1) * 2, (radius + 1) * 2);
		}
	} else if ((type >= RIOT_CHARGE) && (type <= RIOT_BLAST)) {
		if (a > 1 && a <= EXPLODEFRAMES + 1) {
			double sx = x - _global->slope[angle][0] * 15;
			double sy = y - _global->slope[angle][1] * 15;
			int startCirc = (radius / EXPLODEFRAMES) * a;
			triangle (dest, (int)sx, (int)sy, (int)(sx + _global->slope[(angle + 45) % 360][0] * startCirc), (int)(sy + _global->slope[(angle + 45) % 360][1] * startCirc),(int)(sx + _global->slope[(angle + 315) % 360][0] * startCirc),(int)(sy + _global->slope[(angle + 315) % 360][1] * startCirc), player->color);
			setUpdateArea ((int)sx - (startCirc + 1), (int)sy - (startCirc + 1), (startCirc + 1) * 2, (startCirc + 1) * 2);
		}
	} else {
		if (a > 1 && a <= EXPLODEFRAMES + 1) {
			int startCirc = (radius / EXPLODEFRAMES) * a;
			circlefill (dest, (int)x, (int)y, startCirc, (player)?player->color:WHITE);
			startCirc += (radius / EXPLODEFRAMES) * 2;
			setUpdateArea ((int)x - startCirc, (int)y - startCirc, startCirc * 2, startCirc * 2);
		}
	}
}

int EXPLOSION::isSubClass (int classNum)
{
	if (classNum == EXPLOSION_CLASS)
		return (TRUE);
	else
		return (FALSE);
		//return (PHYSICAL_OBJECT::isSubClass (classNum));
}
