/* $Id: Visual.cpp,v 1.11 2003/03/26 15:43:02 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the ArkRPG Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <Ark/ArkFontBitmap.h>

#include "config.h"
#include "Visual.h"
#include "GLRenderer.h"

#include <GL/gl.h>

// Due to some NVIDIA includes, I had to add this for linux.
#ifndef WIN32
#undef GLAPIENTRY
#undef GLAPI
#define GLAPIENTRY
#define GLAPI
#endif

#include <GL/glu.h>
#include <stdio.h>
#include <algorithm>

namespace Ark
{

   GLCache::GLCache () :
      Cache ()
   {}

   GLCache::~GLCache ()
   {
   }

   void
   GLCache::AddRenderer(GLRenderer *rnd)
   {
      m_Renderers.push_back(rnd);
   }

   
   void
   GLCache::RemoveRenderer(GLRenderer *rnd)
   {
      m_Renderers.erase(std::remove
			(m_Renderers.begin(), m_Renderers.end(), rnd),
			m_Renderers.end());
   }
   
   void
   GLCache::ResetTexture (GLTexture *newt)
   {
      GLRendererList::iterator it;
      for (it = m_Renderers.begin(); it != m_Renderers.end(); it++)
	 (*it)-> m_State.m_Passes [(*it)->m_TexUnit].m_Texture = TexturePtr(newt);
   }

   Object *
   GLCache::CreateObject (ObjectType t, const String &name)
   {
      if (t == V_TEXTURE)
	 return new GLTexture (name);
      if (t == V_FONT)
	 return new BitmapFont (name);
      
      return Cache::CreateObject (t, name);
   }


////////////////////////////////////////////////////////////////////////////   
   extern bool g_TexSize_Power2;
   extern int g_TexSize_Maximum;

   int ConvertTextureSize(int n)
   {
      if (g_TexSize_Power2)
      {
	 if (n <= 16)        n = 16;
	 else if (n <= 32)   n = 32;
	 else if (n <= 64)   n = 64;
	 else if (n <= 128)  n = 128;
	 else if (n <= 256)  n = 256;
	 else if (n <= 512)  n = 512;
	 else if (n <= 1024) n = 1024;
      }

      if (n > g_TexSize_Maximum)
	 n = g_TexSize_Maximum;

      return n;
   }

   GLTexture::~GLTexture()
   {
   }
   
   bool GLTexture::Load (Cache *cache, const String &filename)
   {
      GLCache *glcache = static_cast<GLCache*> (cache);
      assert (glcache != NULL);

      ImagePtr img;
      glcache->Get (V_IMAGE, filename, img);
      glcache->ResetTexture(this);

      if (img)
      {
	  m_Image = img;
	  return true;
      }

      // image not loaded, we fail
      return false;
   }

   bool GLTexture::Configure ()
   {
       if (!m_Image)
	   return false;

       const bool result = SetImage(m_Image);
       m_Image = ImagePtr();

       return result;
   }

   bool GLTexture::SetImage (const ImagePtr& img)
   {
      if (m_GLID != 0 || img->m_Data == NULL)
	 return false;
      
      m_TexFormat = GL_RGB;
      m_Format = img->m_Format;
      
      m_Width = img->m_Width; 
      size_t w = ConvertTextureSize (img->m_Width);
      
      m_Height = img->m_Height; 
      size_t h = ConvertTextureSize (img->m_Height);
      
      const int depth = img->GetBytesPerPixel();
      switch (m_Format)
      {
	 case Image::RGB_888:
	    m_TexFormat = GL_RGB;
	    break;
	    
	 case Image::RGBA_8888:
	    m_TexFormat = GL_RGBA;
	    break;
	    
	 case Image::I_8:
	    m_TexFormat = GL_LUMINANCE;
	    break;
	    
	 case Image::A_8:
	    m_TexFormat = GL_ALPHA;
	    break;
	    
	 default:
	    return false;
      }

      uchar *data = img->m_Data;
      bool deletedata = false;

      if (w != img->m_Width || h != img->m_Height)
      {
	 data = new uchar[w * h * depth];
	 deletedata = true;

	 gluScaleImage
	    ((GLenum) m_TexFormat,
	     img->m_Width, img->m_Height, GL_UNSIGNED_BYTE, img->m_Data,
	     w, h, GL_UNSIGNED_BYTE, data);
      }
      
      glGenTextures (1, &m_GLID);
      glBindTexture (GL_TEXTURE_2D, m_GLID);

      glPixelStorei (GL_UNPACK_ALIGNMENT, 1);
      glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);

      switch (m_RepeatMode)
      {
	  case Image::CLAMP:
	      glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
	      glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
	      break;
	  case Image::REPEAT:
	  default:
	      glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
	      glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
	      break;
      }
     
      // FIXME: is it right to disable mipmapping for "small" textures ? 
      if (w <= 64 && h <= 64)
      {
	 glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	 glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);

	 glTexImage2D(GL_TEXTURE_2D, 0, depth,
		      w, h, 0, GLenum(m_TexFormat),
		      GL_UNSIGNED_BYTE, data);
      }
      else
      {
	 glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
	 glTexParameteri (GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER,
			  GL_LINEAR_MIPMAP_NEAREST);
      
	 gluBuild2DMipmaps (GL_TEXTURE_2D, depth, w, h,
			    GLenum(m_TexFormat), GL_UNSIGNED_BYTE, data);
      }

      if (deletedata)
	 delete[] data;

      return true;
   }

   bool
   GLRenderer::RenderParticleSys (ParticleSys *psys)
   {
      ParticleTmpl *tpl = psys->m_Template;
      std::vector<Particle>::iterator cur = psys->m_Particles.begin();

      if (cur == psys->m_Particles.end() || tpl == NULL)
	 return false;
      
      /* ====== LINE ======= */
      if (tpl->m_pType == ParticleTmpl::TYPE_LINE)
      {
	 Color white;
	 SetTexture (TexturePtr());
	 SetBlend (true, BLEND_SRC_ALPHA, BLEND_ONE);
	 //SetBlend (true, BLEND_SRC_ALPHA, BLEND_ONE_MINUS_SRC_ALPHA);
	 SetLighting (false, white, white, white);
	 
	 glDisable (GL_TEXTURE_2D);
	 glBegin (GL_LINES);
     
	 while (cur != psys->m_Particles.end())
	 {
	    if (!cur->m_Dead)
	    {
	       Vector3 pos = cur->m_Pos;
	       Vector3 vel = cur->m_Vel.GetUnitVector();
	       
//      scalar alpha = 1.0 - (cur->m_Age / cur->m_LifeTime);
//      glColor4f (1.0, 1.0, 1.0, alpha);
	       glColor4fv (&tpl->m_Color.R);
	       
	       glVertex3fv (&pos.X);
	       pos += vel.Scale (tpl->m_LineLength);
	       glVertex3fv (&pos.X);
	    }
	    
	    cur++;
	 }
	 
	 glEnd ();
      }
      /* ====== SPRITE ======= */
      else if (tpl->m_pType == ParticleTmpl::TYPE_SPRITE)
      {
	 Color white;
	 Color col = tpl->m_Color;
	 scalar rad = tpl->m_Radius;

	 SetTexture (TexturePtr(tpl->m_Sprite));

	 SetBlend (true, BLEND_SRC_ALPHA, BLEND_ONE);
	 //SetBlend (true, BLEND_SRC_ALPHA, BLEND_ONE_MINUS_SRC_ALPHA);
	 SetCulling (false);
	 SetDepthWrite (false);
	 SetDepthFunc (DEPTHFUNC_LEQUAL);
	 SetLighting (false, white, white, white);

	 // Get Up and Right vector with inverse rotations to build
	 // billboard polygons.
	 // See the article on Billboarding on Nate Miller's website :
	 // http://nate.scuzzy.net

	 Matrix44 &M = m_ViewMatrix;
	 Vector3 up (M.M(0,0), M.M(1,0), M.M(2,0));
	 Vector3 ri (M.M(0,1), M.M(1,1), M.M(2,1));

	 Vector3 mup (-up.X, -up.Y, -up.Z);
	 Vector3 mri (-ri.X, -ri.Y, -ri.Z);

	 glBegin (GL_QUADS);
	 while (cur != psys->m_Particles.end())
	 {
	    if (!cur->m_Dead)
	    {
	       scalar lifefactor = cur->m_Age / cur->m_LifeTime;
	       col.A = tpl->m_Color.A * (1.0f - lifefactor);
	       
	       rad = tpl->m_Radius * (cur->m_Age / tpl->m_MaxLife);

	       Vector3 A;

	       glColor4fv (&col.R);

	       A = cur->m_Pos + rad * (mri + mup);
	       glTexCoord2f (0,0);
	       glVertex3fv (&A.X);

	       A = cur->m_Pos + rad * (ri + mup);
	       glTexCoord2f (1,0);
	       glVertex3fv (&A.X);

	       A = cur->m_Pos + rad * (ri + up);
	       glTexCoord2f (1,1);
	       glVertex3fv (&A.X);

	       A = cur->m_Pos + rad * (mri + up);
	       glTexCoord2f (0,1);
	       glVertex3fv (&A.X);
	    }
	    
	    ++cur;
	 }
	 glEnd ();

	 SetDepthWrite (true);
      }

      return true;
   }

   
/* namespace Ark */
}
