/* $Id: ArkFileSys.h,v 1.11 2002/10/18 21:05:31 mrq Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_FILESYS_H
#define ARK_FILESYS_H

#include <Ark/ArkTypes.h>
#include <Ark/ArkString.h>
#include <map>

namespace Ark
{

//  ==========================================================================
/**
* This is the filesystem class. The only filesytem to be used is the one
* the system has created : Sys()->FS().
* A file system provides regular file testing (IsFile, IsDir),
* but it also allows more complex ones (like Mount/UnMount), which can be
* used to "abstract" the way the programs access to data. You can get the
* real name of a file using the GetFileName() function, and it is even
* automated by the AutoReadStream class.
*/
//  ==========================================================================

class ARK_DLL_API FileSystem
{
public:
  FileSystem();
  virtual ~FileSystem();
  /** \brief Tests wether 'path' points to a file */
  virtual bool  IsFile (const String &name);

  /** \brief Tests wether 'path' points to a directory */
  virtual bool  IsDir  (const String &name);

  /** \brief Creates a directory named 'path' */
  virtual bool  MakeDir  (const String &path, int mode);

  /** \brief Mount a path to a mountpoint.
  * \param name is the name of the mountpoint
  * \param dir is the path
  *
  * If you mount /home/toto to home, like
  * this :
  * \code
  *           Mount ("toto", "/home/toto");
  * \endcode
  *
  * You'll be able to open files in the toto mountpoint without having to know
  * where it does point to, using the AutoReadStream :
  * \code
  *           AutoReadStream s ("{toto}/lettre_a_tata.txt", "rt");
  * \endcode
  */
  virtual bool  Mount  (const String &name, const String &dir);

  /** Remove an existing mountpoint
  * \param name is the name of the mountpoint.
  * \return false on error.
  */
  virtual bool  UnMount(const String &name);

  /** Try to map a file name with mount points in it to a real file name.
  * This can be done if a directory is mounted to another (eg it doesn't work
  * with tarballs).
  */
  virtual String GetFileName (const String &name, bool err = true);
  
  private:
    std::map<String, String> m_MountPoints;
};

}

#endif
