/* $Id: ArkConfig.h,v 1.18 2002/10/11 01:10:01 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_CONFIG_H
#define ARK_CONFIG_H

#include <Ark/Ark.h>
#include <Ark/ArkStream.h>
#include <Ark/ArkVariable.h>

namespace Ark
{

   class ConfigP;

   //  =======================================================================
   /** This class is used to read "configuration".
    * What I call a configuration is the association of values for variables.
    * The way it works :
    *   \arg You create the object. There is no var.
    *   \arg You Load() files. The variables defined are those which can be
    *        found in the file.
    *   \arg For each variable you need, you call GetVar, with a default value.
    *        If the variable wasnt defined, or if its type is wrong, then the
    *        default value is used.
    */
   //  =======================================================================
   class ARK_DLL_API Config
   {
      public:
	 /// Definition of function pointer for call to AddFunction
	 typedef bool (*CfgCb) (void *misc, Config *self, int nargs,
				String *args);
	 
      public:
	 /// Inits an empty configuration object
	 Config ();
	 
	 /// Destroy this configuration, its variables & function callbacks
	 ~Config ();
	 
	 /** Parse a file : set variables values and call callbacks */
	 bool Load (const String &file);
	 
	 /** Parse a file : set variables values and call callbacks */
	 void Load (const String& name, Stream &stream);

	 /** Load a "system" configuration, meaning it will be searched for
	  *  (in this order) in the system configuration directory, in the
	  *  user's home directory and in the current directory.
	  */
	 bool LoadSystem (const String &name);

	 /** Write all config variables in a file. Beware that it does
	  *  overwrite comments (of course) and does not reproduce function
	  *  calls, neither does it take care of includes, etc. In brief : 
	  *  use with caution.
	  **/
	 void Write (const String &file) const;
	 
	 /** Remove all variables from the config object */
	 void Clear ();

	 /**
	  * Replaces all references to variables in the source string by
	  * the actual value of these variables. A variable reference is
	  * of the form "%{variable_name}".
	  * 
	  * \return The string with variable content. If a variable reference
	  * cant be resolved, then it is not changed at all.
	  */
	 String Expand (const String &src);
	 
	 /** Add a function callback to this configuration object.
	  *     \param name is the name the function will have in the script
	  *     \param nargs is the number of arguments this function needs
	  *     \param cb is a pointer to the function
	  *     \param misc it a pointer to user data which is given as the
	  *            first argument to the function.
	  */
	 void AddFunction
	 (const String &name, int nargs, CfgCb cb, void *misc = 0);
	
	 /**
	  * Return a string contained in the variable \c name.
	  * \c def is returned if no value is found.
	  */
	 String GetStr(const String &name, const String &def) const;

	 /**
	  * Return a scalar contained in the variable \c name.
	  * \c def is returned if no value is found.
	  */
	 scalar GetScalar (const String &name, scalar def) const;

	 /**
	  * Return an integer contained in the variable \c name
	  * \c def is returned if no value is found.
	  */
	 int GetInt (const String &name, int def) const;

      private:
	 /// Private implementation of the configuration class.
	 friend class ConfigP;
	 ConfigP *m_Cfg;
   };

}

#endif
