/* aprsdigi: APRS-style in-band UI frame digipeater and node.
 *
 *  APRS has some peculiar digipeating ideas, so let's test them here:
 *
 *  1. Several digipeater aliases such as RELAY, WIDE, GATE.
 *  2. WIDEn-n flooding algorithm.
 *  3. SSID-based shorthand for digi path.
 *  
 *  See Bob Bruninga's (WB4APR) README/MIC-E.TXT for a description of
 *  methods 2 and 3.  #1 is conventional TNC digipeating with MYAlias, etc.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License, version 2 as
 *  published by the Free Software Foundation.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * (See the file "COPYING" that is included with this source distribution.)
 * 
 * derived from ax25-utils/listen.c
 *
 * portions Copyright (c) 1996,1997,1999 Alan Crosswell
 * Alan Crosswell, N2YGK
 * 144 Washburn Road
 * Briarcliff Manor, NY 10510, USA
 * n2ygk@weca.org
 *
 * TODO:
 *  see ./TODO
 *  Figure out GATE function vs. split rx/tx.
 *  Combine unproto() and parsecalls().
 *  Figure out why I can't just use standard full_sockaddr instead of
 *   struct ax_calls.
 *  Sanity check args more rigorously.
 */
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/socket.h>
#include <net/if.h>
#include <netinet/if_ether.h>
#include <netinet/in.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <linux/ax25.h>
#include <linux/rose.h>
#include <signal.h>
#include <errno.h>
#include "axconfig.h"
#include "axutils.h"
#include "mic_e.h"

#ifndef PACKAGE
#define PACKAGE "aprsdigi"
#endif
#ifndef VERSION
#define VERSION "$Revision: 1.28 $";
#endif

/* some defines that really belong in a header file! */
#define	ALEN		6	/* index to where the SSID and flags go */
#define	AXLEN		7	/* length of a callsign */
/* SSID mask and various flags that are stuffed into the SSID byte */
#define	SSID		0x1E
#define	HDLCAEB		0x01
#define	REPEATED	0x80

#define	UI		0x03	/* unnumbered information (unproto) */
#define	PID_NO_L3	0xF0	/* no level-3 (text) */
#define	C		0x80
#define	SSSID_SPARE	0x40	/* must be set if not used */
#define	ESSID_SPARE	0x20


struct statistics {		/* counters */
  int rx;			/* packets received */
  int rx_ign;			/*  rx packets ignored */
  int rx_dup;			/*  dupe packets killed */
  int rx_loop;			/*  looping packets killed */
  int rx_mic;			/*  mic_e packets killed */
  int tx;			/* packets transmitted, sum of: */
  int digi;			/*  regular digipeats */
  int flood;			/*  flood-N digipeats */
  int ssid;			/*  ssid digipeats */
  int ids;			/*  id packets */
};

/* the per-interface information */
#define MAXALIASES 5
#define MAXINTF 10
static struct interface {
  char *port;			/* axports port name */
  char *dev;			/* kernel device name */
  char *tag;			/* optional tag text */
  int taglen;
  int idinterval;		/* seconds between IDs; 0 for none. */
  int i_flags;			/* status flags */
  ax25_address aliases[MAXALIASES]; /* intf call & aliases */
  int n_aliases;
  int rsock,tsock;		/* socket fds */
  struct sockaddr rsa,tsa;	/* and their sockaddrs */
  time_t next_id;		/* next time we ID */
  u_char idbuf[AX25_MTU];	/* An ID packet for this intf */
  int idlen;
  struct statistics stats;	/* statistics */
} Intf[MAXINTF];
#define MYCALL(n) Intf[n].aliases[0]
#define I_MYCALL(i) i->aliases[0]
static int N_intf = 0;

struct interface_list {		/* a list of interfaces (duh) */
  struct interface *i;
  struct interface_list *next;
};

struct stuff {			/* maybe I should learn C++... */
  u_char *cp;			/* pointer into the packet */
  int len;			/* length of it */
  struct ax_calls in;		/* the received packet's calls, flags, etc. */
  struct ax_calls out;		/* the transmitted packet's calls, etc. */
  struct interface *i;		/* the interface received on */
}; 

/* some global stuff */

/* General options: */
static int Verbose = 0;
static int Testing = 0;
static int Digi_SSID = 0;
static int Kill_dupes = 0;	/* kill dupes even in conventional mode */
static int Kill_loops = 0;	/* kill loops */
static char *Logfile = NULL;
static ax25_address Aprscall;	/* replace mic-e encoded to-call with this */
static ax25_address Trace_dummy; /* dummy call for tracen-n substitution */
static ax25_address Widecall;	/* dummy call for tracen-n substitution */
static int Have_digipath = 0;
#define DIGISIZE 7		/* one more than usual! */
static ax25_address Digipath[DIGISIZE]; /* for SSID-[1:7] w/o flooding */
static struct full_sockaddr_ax25 Path[4]; /* for SSID-[8:15] N S E W */
static char Dirs[5] = "NSEW";

/* per-interface default options: */
static char *Tag = NULL;	/* tag onto end of posit in SSID mode */
static int Taglen = 0;
static int Idinterval = (9*60)+30; /* default to 9:30 */
static int Keep = 20;		/* seconds to remember for dupe test */
static int I_flags = 0;		/* interface default flags */
#define I_NEED_ID 1		/* need to ID */
#define SUBST_MYCALL 2		/* replace digi alias w/mycall */
#define MICE_XLATE   4		/* translate mic-e to tracker format */
#define X1J4_XLATE   8		/* translate x1j4 to plain format */

/*
 * A table of recognized flooding alias prefixes.  In current practice, 
 * the only recognized flood aliases are WIDEn and TRACEn(where n is 0-7)
 * where the latter has truly odd behavior.  The flavors of WIDE and TRACE
 * are:
 *  1. WIDE:  A conventional alias, eligible for MYCALL substitution.
 *     Roughly equivalent to WIDE0-0.
 *  2. WIDEn-n: Flooding algorithm.  Decrement SSID and repeat the packet
 *     without setting the DIGIPEATED flag until it reaches zero at which
 *     point do set the DIGIPEATED flag and move on to the next callsign
 *     in the digipeat list.  Does not get MYCALL substituted (even WIDEn-0)
 *     so the user can see that the packet came in via n (anonymous) WIDE
 *     digis.
 *  3. TRACE: Like WIDE but MYCALL substitution is not required.
 *  4. TRACEn-n: Do the SSID substitution like WIDE, but also insert MYCALL
 *     into the digi list:
 *	RELAY*,TRACE3-3
 *	RELAY,N2YGK-7*,TRACE3-2
 *	RELAY,N2YGK-7,WB2ZII*,TRACE3-1
 *	RELAY,N2YGK-7,WB2ZII,N2MH-15*,TRACE3
 *	RELAY,N2YGK-7,WB2ZII,N2MH-15,WA2YSM-14*
 *     (What happens when the digi list gets too long for the AX.25 protocol
 *     appears not to have been considered in this design.  I guess we'll
 *     just insert as many as we can.)
 */

/*
 * These flags are shared by the calltab and floods tables and by 
 * the floodtype of a specific received call.  That is, while there is
 * only one WIDE calltab entry, a received callsign can be WIDE-n or
 * WIDEn-n.
 */
#define C_IS_NOFLOOD 0		/* not a flood */
#define C_IS_FLOOD 1		/* this callsign is a flood (WIDE) */
#define C_IS_FLOODN 2		/* this is an N-N type flood (WIDEn-n) */
#define C_IS_TRACE 4		/* this is a trace-type flood (TRACE*) */

struct flood {
  ax25_address call;		/* the flooding callsign prefix */
  int len;			/* length */
  int flags;
} Floods[MAXALIASES];
static int N_floods = 0;
#define FLOODCALL Floods[0].call
#define FLOODLEN Floods[0].len

/* forward declarations */
static int unproto(struct full_sockaddr_ax25 *, char *);
static int parsecalls(ax25_address*, int, char *);
static void print_it(FILE *,struct ax_calls *,unsigned char *,int len);
static void add_text(u_char **,int *,u_char *,int,char *,int);
static int dupe_packet(struct ax_calls *,u_char *,int);
static int loop_packet(struct ax_calls *,u_char *,int);
static int xmit(struct stuff *s);
static void set_id(void);
static void rx_packet(struct interface *i, u_char *buffer, int len);
static int floodcmp(ax25_address *a, int len, ax25_address *b);
static int floodtype(ax25_address *c, int i);
static void sked_id(struct interface *iface);
static void usage(void);
static void newusage(void);
static void check_config();
static void print_dupes(void);
static struct interface_list *intf_of(ax25_address *callsign);

/* signal handlers */
static void cleanup(int),identify(int),identify_final(int),
  print_stats(int),reset_stats(int);

static void do_opts(int argc, char **argv);
static void do_ports(void);
static void set_sig(void);
static int rx_loop(void);

int
main(int argc, char **argv)
{
  int r;

  bzero(Intf,sizeof(Intf));
  do_opts(argc,argv);
  do_ports();
  set_id();
  set_sig();
  check_config();
  r = rx_loop();
  exit(r);
}

/* Listen for received packets and farm out the work */
static int
rx_loop()
{
  unsigned char buffer[AX25_MTU];
  int n,selfds = 0;
  struct interface *i;
  fd_set select_mask;

  /* set up the initial select mask */
  FD_ZERO(&select_mask);
  for (n  = 0, i = Intf; n < N_intf; n++, i++) {
    FD_SET(i->rsock, &select_mask);
    selfds = (i->rsock>selfds)?i->rsock:selfds;
  }
  ++selfds;
  for (;;) {
    int len;
    fd_set rmask = select_mask;
    int size = sizeof(struct sockaddr);

    if (select(selfds,&rmask,NULL,NULL,NULL) < 0) {
      if (errno == EINTR)
	continue;
      perror("select");
      return 1;
    }

    /* find which sockets have data */
    for (n = 0, i = Intf; n < N_intf; n++, i++) {
      if (FD_ISSET(i->rsock,&rmask)) {
	if ((len = recvfrom(i->rsock,buffer,sizeof(buffer),0,&i->rsa,&size)) < 0) {
	  if (errno == EINTR)
	    continue;
	  perror(i->port);
	  return 1;
	}
#ifndef USE_TWO_SOCKS
	i->tsa = i->rsa;
#endif
	rx_packet(i,buffer,len); /* process the received packet */
      }
    }
  } /* end of for(;;) */
  return 0;			/* NOTREACHED */
}

/* Parse a received packet, convert and digipeat if necessary. */

static int rx_nodigi(struct stuff *s,int r);
static int rx_dupe(struct stuff *s);
static int rx_to_me(struct stuff *s);
static int rx_mice(struct stuff *s);
static int rx_flood(struct stuff *s);
static int rx_digi(struct stuff *s);

static void
rx_packet(struct interface *i,	/* which interface received on */
	  u_char *buffer,	/* what was received */
	  int len)		/* length received */
{
  int r;
  struct stuff s;

  bzero(&s,sizeof(s));
  s.i = i;
  ++s.i->stats.rx;
  s.cp = buffer;
  s.len = len;

  /* parse the raw kiss frame */
  r = parse_kiss(&s.cp,&s.len,&s.in);
  if (Verbose) {
    fprintf(stderr,"%s: RX: ",s.i->port);
    print_it(stderr,&s.in,s.cp,s.len);
  }

  /* go through a bunch of choices until we eat the packet or die:-) */

  if (rx_mice(&s))		/* mic-E special handling */
    return;
  else if (rx_nodigi(&s,r))	/* Nothing to digi? */
    return;
  else if (rx_dupe(&s))		/* Is it a killed dupe or loop? */
    return;
  else if (rx_to_me(&s))	/* Addressed to me? */
    return;
  else if (rx_flood(&s))	/* flood special handling */
    return;
  else if (rx_digi(&s))		/* conventional digi */
    return;

  /* How'd we get here? */
  if (Testing) {
    fprintf(stderr,"%s: Not repeatable\n",s.i->port);
  }
}	  

static int
rx_nodigi(struct stuff *s,int r)
{
  int result = 0;
  /* bail if invalid or not doing SSID & no unrepeated digis */
  if (r == PK_INVALID || r == PK_VALID || (r != PK_VALDIGI && !Digi_SSID)) {
    ++s->i->stats.rx_ign;
    result = 1;
  }
  if (Verbose) {
    if (result)
      fprintf(stderr,"packet is %s. (r=%d)\n",
	      (r == PK_INVALID)?"invalid":"not repeatable",result);
    else
      fprintf(stderr,"packet is repeatable. (r=%d)\n",r);
  }
  return result;
}

static int
rx_dupe(struct stuff *s)
{
  int result = 0;
  static char *lupedupe[] = {"dupe or loop","dupe","loop"};

  /* If packet was last transmitted by me, then don't digipeat it! */
  if (Kill_dupes && dupe_packet(&s->in,s->cp,s->len)) {
    ++s->i->stats.rx_dup;
    result = 1;
  } else if (Kill_loops && loop_packet(&s->in,s->cp,s->len)) {
    ++s->i->stats.rx_loop;
    result = 2;
  }
  if (Verbose) {
    fprintf(stderr,"packet is%sa %s\n",result?" ":" not ",lupedupe[result]);
  }
  return result;
}  


/*
 * find interfaces belonging to a callsign or alias.  Returns a pointer to
 * a list of interfaces or NULL if not found.
 */
static struct callsign_list {
  ax25_address *callsign;
  int flags;
  int floodlen;			/* length of flood call */
  struct interface_list *l;	/* list of interfaces having this callsign */
  struct interface_list *illast;
  struct callsign_list *next;
} *calltab = NULL, *ctlast = NULL;

static void calltab_init(void);

static struct callsign_list *
calltab_entry(ax25_address *callsign,int *flags)
{
  struct callsign_list *c;
  int flagdummy;

  if (flags == NULL)
    flags = &flagdummy;		/* dummy the flags if not needed */
  *flags = 0;
  for (c = calltab; c; c = c->next) {
    if ((c->flags&C_IS_FLOOD 
	 && (*flags = floodcmp(c->callsign,c->floodlen,callsign)))
	|| (ax25cmp(c->callsign,callsign) == 0)) {
      *flags |= c->flags;	/* add stuff floodcmp doesn't know about */
      return c;
    }
  }
  return NULL;
}

static void 
calltab_init(void)
{
  struct interface *i;
  int n,m;
  
  /* iterate over all interfaces' callsigns */
  for (i = Intf, n = 0; n < N_intf; n++,i++) {
    for (m = 0; m < i->n_aliases; m++) {
      struct interface_list *new_il = 
	(struct interface_list *)calloc(1,sizeof(struct interface_list));
      struct callsign_list *c = (calltab)?calltab_entry(&i->aliases[m],0):NULL;

      if (!c) {			/* first time seeing this call */
	int f;
	c = (struct callsign_list *)calloc(1,sizeof(struct callsign_list));
	c->callsign = &i->aliases[m];
	/* see if this is a flood call */
	for (f = 0; f < N_floods; f++) {
	  if (floodcmp(&Floods[f].call,Floods[f].len,c->callsign)) {
	    c->flags =  Floods[f].flags;
	    c->floodlen = Floods[f].len;
	    break;
	  }
	}
	/* add the new callsign_list to calltab */
	if (ctlast) {
	  ctlast->next = c;
	  ctlast = c;
	} else {
	  calltab = ctlast = c;	/* very first entry */
	}
      }
      new_il->i = i;		/* initialize the interface list */
      if (c->illast) {		/* and link it in to the callsign's list */
	c->illast->next = new_il;
	c->illast = new_il;
      } else {
	c->l = c->illast = new_il;
      }
    }
  }
}

static struct interface_list *
intf_of(ax25_address *callsign)
{
  struct callsign_list *c;

  if (c = calltab_entry(callsign,0))
    return c->l;
  return NULL;
}

static int
rx_to_me(struct stuff *s)
{
  int result = 0;
  struct interface_list *l = intf_of(&s->in.ax_to_call);

  if (l) {
    result = 1;
  }
  if (Verbose) {
    fprintf(stderr,"packet is%saddressed to me.\n",result?" ":" not ");
  }
  return result;
}  

/* 
 * MIC-E SSID path selection only applies if:
 *  1. 1st byte of info field must be one of the "Mic-E cookies."
 *  2. SSID digipeating is enabled.
 *  3. To-call's SSID must be non zero: this is the route.
 *  4. Digipeater path must be empty.
 */
static int
rx_mice(struct stuff *s)
{
  int ssid, mic_e, result = 0;

  mic_e = (*(s->cp) == 0x60 || *(s->cp) == 0x27
	   || *(s->cp) == 0x1c || *(s->cp) == 0x1d);
  if (mic_e)
    ++s->i->stats.rx_mic;
  if (Verbose) {
    fprintf(stderr,"%s mic_e...\n",mic_e?"is":"is not");
  }  
  if (mic_e && Digi_SSID && s->in.ax_n_digis == 0 
      && (ssid = (s->in.ax_to_call.ax25_call[ALEN]&SSID)>>1)
      && s->len > 0) {
    if (Verbose) {
      fprintf(stderr,"Got an SSID route for path %d.\n",ssid);
    }
    s->out.ax_from_call = s->in.ax_from_call;
    s->out.ax_to_call = s->in.ax_to_call;
    s->out.ax_to_call.ax25_call[ALEN]&=~SSID; /* zero the SSID */
    s->out.ax_type = s->in.ax_type;
    s->out.ax_pid = s->in.ax_pid;
    if (ssid <= 7) {	/* omnidirectional */
      if (N_floods || ssid == 0) { /* in a flooding network? */
	s->out.ax_n_digis = 1;
	s->out.ax_digi_call[0] = FLOODCALL;
	s->out.ax_digi_call[0].ax25_call[FLOODLEN] = (ssid+'0') << 1;
	s->out.ax_digi_call[0].ax25_call[ALEN] |= SSID&(ssid<<1);
	if (Verbose) {
	  fprintf(stderr,"Flooding: setting path to %s\n",
		  ax2asc(&s->out.ax_digi_call[0]));
	}
      } else {		/* not in a flooding network */
	int startat,i;
	if (ssid < 4) {		/* RTFM for why this is */
	  startat = 0;		/* starting point in digipath */
	  s->out.ax_n_digis = ssid; /* number of digis from there. */
	} else {
	  startat = 3;
	  s->out.ax_n_digis = ssid-3;
	}
	if (Verbose) {
	  fprintf(stderr,"Non-flooding: converting SSID WIDE-%d path to DIGI[%d:%d]\n",
		  ssid,startat,startat+s->out.ax_n_digis-1);
	}
	/* fill in the digipeater list */
	for (i = 0; i < s->out.ax_n_digis; i++,startat++) {
	  s->out.ax_digi_call[i] = Digipath[startat];
	}
      } /* flooding/non-flooding network */
    } else {			/* SSID > 7 is directional */
      int j;
      if (Verbose) {
	fprintf(stderr,"setting path to %c UNPROTO%s\n",
		Dirs[ssid&3],(ssid&4)?" + WIDE":"");
      }
      s->out.ax_n_digis = Path[ssid&3].fsa_ax25.sax25_ndigis;
      for (j = 0; j <= s->out.ax_n_digis; j++)
	s->out.ax_digi_call[j] = Path[ssid&3].fsa_digipeater[j];
      if (ssid&4) {	/* directional + wide call */
	s->out.ax_digi_call[s->out.ax_n_digis++] = Widecall;
      }
    }
    ++s->i->stats.ssid;
    if (xmit(s) < 0)
      perror("xmit");
    result = 1;
  } /* mic-E */
  if (Verbose) {
    fprintf(stderr,"did%srequire special Mic-E handling.\n",
	    result?" ":" not ");
  }
  return result;
}  

/* see if special WIDEn-n & TRACEn-n handling applies. */
static int
rx_flood(struct stuff *s)
{
  int i,wide,thisflags,result=0;
  struct callsign_list *c;

  /* FLOODn-n algorithm: next digi with non-zero ssid callsign */
  c = calltab_entry(&s->in.ax_digi_call[s->in.ax_next_digi],&thisflags);
  if (c && thisflags&C_IS_FLOOD
      && (wide = (s->in.ax_digi_call[s->in.ax_next_digi].ax25_call[ALEN]&SSID)>>1)) {

    if (Verbose) {
      fprintf(stderr,"Got a flooding %s route.\n",
	      ax2asc(&s->in.ax_digi_call[s->in.ax_next_digi]));
    }

    /* flooding algorithm always kills dupes.  If kill_dupes
       option was not selected, then do the dupe checking here */
    if (!Kill_dupes && dupe_packet(&s->in,s->cp,s->len)) {
      if (Verbose) {
	fprintf(stderr,"flood packet is a dupe\n");
      }
      return 1;
    }
    /* decrement the flood counter (SSID) */
    s->out = s->in;		/* copy the input header */
    s->out.ax_digi_call[s->out.ax_next_digi].ax25_call[ALEN] &= ~SSID;
    s->out.ax_digi_call[s->out.ax_next_digi].ax25_call[ALEN] |= ((--wide) << 1)&SSID;
    /* TRACEn-n: insert dummy mycall in front: xmit will put the real one in */
    if ((thisflags&(C_IS_FLOODN|C_IS_TRACE)) == (C_IS_FLOODN|C_IS_TRACE)) {
      int n;
      if (s->out.ax_next_digi >= AX25_MAX_DIGIS) {
	/* XXX */
	fprintf(stderr,"%s: TRACEn-n list overflow. Last digi dropped.\n",
		s->i->port);
	s->out.ax_n_digis--;	/* XXX check off-by-1 */
      }
      /* shift remaining digis right one slot to make room */
      for (n = s->out.ax_n_digis; n >= s->out.ax_next_digi; n--) {
	s->out.ax_digi_call[n] = s->out.ax_digi_call[n-1];
      }
      /* then stuff in a dummy "TRACE" and mark it repeated */
      s->out.ax_digi_call[s->out.ax_next_digi] = Trace_dummy;
      s->out.ax_digi_call[s->out.ax_next_digi].ax25_call[ALEN] |= REPEATED;
      s->out.ax_n_digis++;
    }
    if (Verbose) {
      fprintf(stderr,"Rewriting it as %s:\n",
	      ax2asc(&s->out.ax_digi_call[s->out.ax_next_digi]));
    }
    ++s->i->stats.flood;
    if (xmit(s) < 0) 
      perror("xmit");
    return 1;
  }
  if (Verbose) {
    fprintf(stderr,"Did %s require special flooding handling.\n",
	    result?" ":" not ");
  }
  return result;
}

/* see if conventional digipeat handling applies. */
static int
rx_digi(struct stuff *s)
{
  int j, result = 0;
  struct interface_list *l;

  /* conventional: see if the next digipeater matches one of my calls */
  if (l = intf_of(&s->in.ax_digi_call[s->in.ax_next_digi])) {
    /* a packet for me to digipeat */
    if (Verbose) {
      fprintf(stderr,"Got a conventional digipeat.\n");
    }
      s->out = s->in;	/* copy input list to output unmodifed */
      s->out.ax_digi_call[s->out.ax_next_digi].ax25_call[ALEN] |= REPEATED;
      ++s->i->stats.digi;
      if (xmit(s) < 0) 
	perror("xmit");
      result  = 1;
  }
  if (Verbose) {
    fprintf(stderr,"Did%srequire conventional digipeat handling.\n",
	    result?" ":" not ");
  }
  return result;
}

static void
add_text(op,oleft,text,len,tag,taglen)
unsigned char **op;
int *oleft;
u_char *text;
int len;
char *tag;
int taglen;
{
  if ((*oleft -= len) > 0) {
    bcopy(text,*op,len);	/* copy the text */
    *op += len;
  }
  if (taglen && tag && (*oleft -= taglen) > 0) {
    bcopy(tag,*op,taglen); /* and tack on the tag */
    *op += taglen;
  }
}

/* watch out for overflow when adding mycall and/or wide */
static int
unproto(calls,str)		/* parse a via path into a calls struct */
struct full_sockaddr_ax25 *calls;
char *str;
{
  char buf[200];

  bzero(calls, sizeof(*calls));
  sprintf(buf,"dummy via %s",str);
  return convert_call(buf,calls);
}

static int
parsecalls(calls,ncalls,str)	/* parse a via path into a calls struct */
ax25_address *calls;
int ncalls;			/* max number */
char *str;
{
  char *cp;
  int i;

  bzero(calls,ncalls*sizeof(*calls));
  cp = strtok(str," \t\n,");
  for (i = 0; cp && i < ncalls; i++) {
    if (convert_call_entry(cp,calls[i].ax25_call) < 0)
      return -1;
    cp = strtok(NULL," \t\n,");
  }
  return i;
}

static void
print_it(FILE *f,
	 struct ax_calls *calls,
	 u_char *data,
	 int len)
{
  int j;
  char asc_from[12],asc_to[12];

  if (f == NULL)
    return;
  strncpy(asc_to,ax2asc(&calls->ax_to_call),sizeof(asc_to));
  strncpy(asc_from,ax2asc(&calls->ax_from_call),sizeof(asc_from));
  fprintf(f,"%s>%s",asc_from,asc_to);
  for (j = 0; j < calls->ax_n_digis; j++) {
    fprintf(f,",%s%s",ax2asc(&calls->ax_digi_call[j]),
	    (calls->ax_digi_call[j].ax25_call[ALEN]&REPEATED
		&& (j == calls->ax_next_digi-1))?"*":"");
  }
  fprintf(f,":%.*s\n",len,data);
}

/* 
 * packet reformatter
 *  depending on options flags on transmit interface, perform reformatting
 *  of the payload.  Current methods include mic_E expansion and X1J4
 *  stripping of the the "TheNet X1J4 (alias)" prefix.  Future methods
 *  to include GPS/APRS position compression, expansion, etc.
 */
static int
reformat(struct stuff *s,	/* "class data":-) */
	 struct interface *i,	/* this iteration's transmit interface */
	 u_char ***ovec,	/* returns ptr to array of u_char ptrs */
	 int **olen)		/* returns ptr to array of lengths */
{
  int r;
  time_t now;
  static u_char mic1[AX25_MTU], mic2[AX25_MTU];
  static u_char *vecp[2];
  static int vecl[2];

  time(&now);

  *ovec = vecp;			/* returned ptrs */
  *olen = vecl;
  vecp[0] = mic1;
  vecp[1] = mic2;
  /* does xmit intf want mic-E translation? */
  if (i->i_flags&MICE_XLATE && 
      fmt_mic_e(ax2asc(&s->out.ax_to_call),s->cp,s->len,
		mic1,&vecl[0],mic2,&vecl[1],now)) {
    s->out.ax_to_call = Aprscall; /* replace compressed lat w/"APRS" */
    return (vecl[0]>0)+(vecl[1]>0);
  } else if (i->i_flags&X1J4_XLATE &&   /* wantx1j4 translation? */
	     fmt_x1j4(ax2asc(&s->out.ax_to_call),s->cp,s->len,
		      mic1,&vecl[0],mic2,&vecl[1],now)) {
    return (vecl[0]>0)+(vecl[1]>0);
  } else {			/* no reformat; just pass cp, len thru */
    vecp[0] = s->cp;
    vecl[0] = s->len;
    return (vecl[0]>0);
  }
}

/* 
 * xmit looks at the next digipeater call to see which interfaces the
 * packet has to go out.  By giving several interfaces the same callsign
 * or alias, multi-way fanout and/or gatewaying can be performed.
 */
static int
xmit(struct stuff *s)
{
  struct callsign_list *c;
  struct interface_list *l;
  int r = 0;
  int thisflags = 0;

  /* transmit the packet on the interface(s) of the next callsign */
  if ((c = calltab_entry(&s->out.ax_digi_call[s->out.ax_next_digi],&thisflags)) == NULL) {
    fprintf(stderr,"xmit: Assertion failed: call %s not found\n",
	    ax2asc(&s->out.ax_digi_call[s->out.ax_next_digi]));
    return -1;
  }
  for(l = c->l; l; l = l->next) { /* loop over each tx interface */
    u_char obuf[AX25_MTU];
    u_char *op = obuf;
    int oleft = sizeof(obuf);
    int olen;
    int npkt,n;
    u_char **vecp;		/* output vector */
    int *vecl;			/* and lengths */

    /* do any required per-intf expansion/translation/compression of payload */
    npkt = reformat(s,l->i,&vecp,&vecl);
    for (n = 0; n < npkt; n++) {
      struct ax_calls calls = s->out;
      /* 
       * Adjust the next_digi pointer only if REPEATED was set.
       * Only substitute MYCALL if marked REPEATED and the callsign
       * flags permit it.  Flags and their rules:
       * FLOOD  FLOODN  TRACE  Rule
       * -----  ------  -----  -----------------------------------
       *     0       X      X  only if SUBST_MYCALL substitute MYCALL
       *     1       0      X  always substitute MYCALL (WIDE/TRACE->MYCALL)
       *     1       1      0  never substitute MYCALL (WIDEn-n)
       *     1       1      1  always substitute MYCALL (???->MYCALL)
       * rx_flood will have inserted tracecall in front of TRACEn-n.
       * but MYCALL subst happens here since it is iface-specific.
       */
      if (calls.ax_digi_call[calls.ax_next_digi].ax25_call[ALEN]&REPEATED) {
	if ((!(thisflags&C_IS_FLOOD) && l->i->i_flags&SUBST_MYCALL)
	    || ((thisflags&C_IS_FLOOD) && !(thisflags&C_IS_FLOODN))
	    || ((thisflags&(C_IS_FLOOD|C_IS_FLOODN|C_IS_TRACE))
		== (C_IS_FLOOD|C_IS_FLOODN|C_IS_TRACE))) {
	  calls.ax_digi_call[calls.ax_next_digi] = I_MYCALL(l->i);
	  calls.ax_digi_call[calls.ax_next_digi].ax25_call[ALEN] |= REPEATED;
	}
	++calls.ax_next_digi;
      }
      gen_kiss(&op,&oleft,&calls);	/* generate the kiss header */
      add_text(&op,&oleft,vecp[n],vecl[n],l->i->tag,l->i->taglen); /* fill in the info field */	

      olen = sizeof(obuf) - oleft;
      if (Logfile) {
	FILE *of;
	if (Logfile[0] == '-' && Logfile[1] == '\0')
	  of = stdout;
	else
	  of = fopen(Logfile,"a");
	print_it(of,&calls,op-(vecl[n]+l->i->taglen),vecl[n]+l->i->taglen);
	if (of != stdout)
	  fclose(of);
      }
      if (Verbose) {
	fprintf(stderr,"%s: TX: ",l->i->port);
	print_it(stderr,&calls,op-(vecl[n]+l->i->taglen),vecl[n]+l->i->taglen);
      }
      ++l->i->stats.tx;
      sked_id(l->i);
      l->i->tsa.sa_family = AF_AX25;
      r += sendto(l->i->tsock,obuf,olen,0,&l->i->tsa,sizeof(l->i->tsa));
    } /* end for n */
  } /* end for l  */
  return r;
}


/* 
 * packet dupe checking 
 *
 * Compare the to, from, and info.  Ignore the digipeater path.
 * If the packet matches one already received within the 'keep'
 * interval then it is a dupe.  Keep the list sorted in reverse
 * chronological order and throw out any stale packets.
 */
struct pkt {
  struct pkt *next,*prev;
  time_t t;			/* when recevied */
  ax25_address to;		/* destination */
  ax25_address fr;		/* source */
  int l;			/* length of text */
  u_char d[AX25_MTU];		/* the text */
};

static struct pkt *top;		/* stacked in reverse chronological order */

static int
dupe_packet(struct ax_calls *calls,u_char *cp,int len)
{
  struct pkt *p, *matched = NULL, *old = NULL;
  time_t now = time(0);
  time_t stale = now - Keep;

  if (Verbose) {
    print_dupes();
  }
  for (p = top; p; p = p->next) {
    if (p->t >= stale) {	/* packet is still fresh */
      if (p->l == len && bcmp(p->d,cp,len) == 0
	  && bcmp(&calls->ax_to_call,&p->to,sizeof(p->to)) == 0
	  && bcmp(&calls->ax_from_call,&p->fr,sizeof(p->fr)) == 0) {
	matched = p;
	break;
      } else
	continue;
    } else {			/* all following packets are stale */
      old = p;
      break;
    }
  }
  if (old) {			/* trim list of stale pkts */
    if (top == old)
      top = NULL;		/* entire list is stale */
    else {
      old->prev->next = NULL;
    }
    while (old) {
      p = old->next;
      free(old);
      old = p;
    }
  }
  if (matched) {		/* move matched packet to head of list? */
#ifdef DUPE_REFRESH_STAMP
    /* 
     * Should dupe-checking update the time stamp each time a dupe is
     * heard again or let it age out so that it can get digi'd
     * periodically?  For example, if a station beacons once every 15
     * seconds, then only the first packet received ever will be
     * digi'd if the "memory" is 20 sec.  Good example of where this
     * could be a problem is a MIC-E beaconing an emergency.  By not
     * refreshing the timestamp, the desired behavior of reducing channel
     * clutter is still achieved.
     */
    matched->t = now;		/* updated timestamp */
    if (matched == top)		/* already on top */
      return 1;
    /* unlink matched packet from list */
    if (matched->prev)
      matched->prev->next = matched->next;
    if (matched->next)
      matched->next->prev = matched->prev;
    matched->prev = matched->next = NULL;
    /* push matched packet on top of list */
    matched->next = top;
    if (top)
      top->prev = matched;
    top = matched;
#endif
    return 1;
  } else {			/* not matched: push new packet on top */
    if ((p = (struct pkt *)calloc(1,sizeof(struct pkt))) == NULL) {
      fprintf(stderr,"failed to calloc!\n");
      return 0;			
    }
    p->t = now;
    p->l = (len>AX25_MTU)?AX25_MTU:len;
    bcopy(cp,p->d,p->l);
    p->to = calls->ax_to_call;
    p->fr = calls->ax_from_call;
    /* push new packet on top of list */
    p->next = top;
    if (top)
      top->prev = p;
    top = p;
    return 0;
  }
}

static void
print_dupes(void) {
  struct pkt *p;

  for (p = top; p ; p = p->next) {
    fprintf(stderr,"dupe @ 0x%0x:  prev->0x%0x  next->0x%0x  time %d  len %d\n",
	    p,p->prev,p->next,p->t,p->l);
  }
}

/*
 * is packet looping?
 * e.g. FOO,mycall,WIDE*,mycall,somecall....
 * This will break some people's traces:-)  So will kill_dupes.
 */

static int
loop_packet(struct ax_calls *calls,u_char *cp,int len)
{
  int n, flags;
  struct callsign_list *cl;

  for (n = 0; n < calls->ax_next_digi; n++) {
    if ((cl = calltab_entry(&calls->ax_digi_call[n],&flags))
	&& flags == C_IS_NOFLOOD)
      return 1;
  }
  return 0;
}

static void
cleanup(sig)
int sig;			/* dont't really care what it was */
{
  if (alarm(0) > 0)		/* an alarm was pending, so... */
    sked_id(NULL);		/*  ID one last time */
  identify_final(sig);
  print_stats(sig);		/* dump final stats */
  exit(0);			/* and exit */
}

static void
print_stats(sig)
int sig;
{
  FILE *of = stderr;
  struct interface *i;
  int n;

  if (Logfile) {
    if (Logfile[0] == '-' && Logfile[1] == '\0')
      of = stdout;
    else
      of = fopen(Logfile,"a");
  }
  if (of == NULL)
    return;
  for (i = Intf, n = 0; n < N_intf; n++,i++) {
    fprintf(of,"# %s: rx %d (ign %d dup %d loop %d mic %d) tx %d (digi %d flood %d ssid %d ids %d)\n",
	    i->port,
	    i->stats.rx,i->stats.rx_ign,i->stats.rx_dup,i->stats.rx_loop,
	    i->stats.rx_mic,
	    i->stats.tx,i->stats.digi,i->stats.flood,i->stats.ssid,
	    i->stats.ids);
  }
  if (of != stdout)
    fclose(of);
  if (sig >= 0)
    signal(sig,print_stats);
}

static void
reset_stats(sig)
int sig;
{
  struct interface *i;
  int n;
  print_stats(-1);

  for (i = Intf, n = 0; n < N_intf; n++,i++) {
    bzero(&i->stats,sizeof(i->stats));
  }
  signal(sig,reset_stats);
}


/* setup a bare minimum legal ID (don't QRM the world with a digi path!) */
static void
set_id()
{
  struct ax_calls id;
  char idinfo[AX25_MTU];
  int i,n;
  struct interface *iface;

  for (n = 0, iface = Intf; n < N_intf; n++, iface++) {
    u_char *op = iface->idbuf;
    int oleft = sizeof(iface->idbuf);

    bzero(&id,sizeof(id));
    convert_call_entry("ID",id.ax_to_call.ax25_call);
    id.ax_to_call.ax25_call[ALEN] |= C; /* it's a command */
    id.ax_from_call = MYCALL(n);
    id.ax_type = UI;
    id.ax_pid = PID_NO_L3;
    id.ax_n_digis = 0;

    gen_kiss(&op,&oleft,&id);	/* generate the kiss header */
    *idinfo = '\0';
    for (i = 0; i < iface->n_aliases; i++) {
      strcat(idinfo,ax2asc(&iface->aliases[i]));
      strcat(idinfo,"/R ");
    }
    for (i = 0; i < N_floods; i++) {
      strcat(idinfo,ax2asc(&Floods[i].call));
      strcat(idinfo,"n-n/R ");
    }
    add_text(&op,&oleft,idinfo,strlen(idinfo),0,0);
    iface->idlen = sizeof(iface->idbuf) - oleft;
  }
}

static void
xmit_id(struct interface *i)
{
  if (Verbose) {
    fprintf(stderr,"%s: TX: ID\n",i->port);
  }
  if (sendto(i->tsock,i->idbuf,i->idlen,
	     0,&i->tsa,sizeof(i->tsa)) < 0) /* ship it off */
    perror(i->dev);
  i->i_flags &= ~I_NEED_ID;
  i->next_id = 0;
  ++i->stats.ids;
}

/*
 * sched alarm for next id.  Called with intf ptr if that interface
 * needs to schedule an ID, NULL during alarm sig handler.
 */
static struct interface *ID_next = NULL;
static void
sked_id(struct interface *iface)
{
  struct interface *i;
  int n;
  time_t now = time(0);
  time_t min = now+24*60*60;
  time_t when;

  if (iface && !(iface->i_flags&I_NEED_ID) && iface->idinterval) { 
    iface->next_id = now+iface->idinterval;
    iface->i_flags |= I_NEED_ID;
  }
  /* find minimum time among all interfaces 'til next needed ID. */
  for (i = Intf, n = 0, ID_next = NULL; n < N_intf; n++,i++) {
    if (i->i_flags&I_NEED_ID && i->idinterval && i->next_id > 0) {
      if ((when = i->next_id - now) <= 0) /* catch any already due */
	xmit_id(i);
      else if (i->next_id < min) {
	ID_next = i;		/* new minimum */
	min = ID_next->next_id;
      }    
    }
  }
  if (ID_next && ID_next->next_id && ID_next->next_id > 0) {
    alarm(when); /* next alarm in this many secs */
    if (Verbose) {
      fprintf(stderr,"next ID for %s in %d seconds\n",ID_next->port, when);
    }
  }
}

static void
identify(int sig)		/* wake up and ID! */
{
  if (ID_next && ID_next->i_flags&I_NEED_ID) {
    xmit_id(ID_next);
    ID_next = NULL;
    sked_id(NULL);		/* schedule next ID */
    signal(sig,identify);
  }
}

static void
identify_final(int sig)		/* wake up and ID! */
{
  struct interface *i;
  int n;

  for (i = Intf, n = 0; n < N_intf; n++,i++) {
    if (i->i_flags&I_NEED_ID && i->idinterval && i->next_id > 0) {
      xmit_id(i);
    }
  }
}

static void
do_opts(int argc, char **argv)
{
  char s;

  while ((s = getopt(argc, argv, "CcMmXxF:f:n:s:e:w:t:k:l:i:d:p:DLVvarT")) != -1) {
    int p = -1;			/* used for NSEW hack, below */
    int fflags = 0;		/* use for fF hack */
    switch (s) {
    case 'a':			/* aliases to listen to */
    case 'r':
      newusage();
      break;
    case 'v':
      ++Verbose;
      break;
    case 'T':
      ++Testing;
      break;
    case 'p':			/* -p port[:alias,alias,alias...] */
      if (N_intf >= MAXINTF) {
	fprintf(stderr,"too many interfaces!\n");
	exit(1);
      }
      {
	struct interface *i = &Intf[N_intf++];
	char *op = strchr(optarg,':');
	i->port = optarg;
	i->i_flags = I_flags;
	i->tag = Tag;
	i->taglen = Taglen;
	i->idinterval = Idinterval;
	if (op) {		/* additional aliases... */
	  *op++ = '\0';
	  if ((i->n_aliases=parsecalls(&i->aliases[1],MAXALIASES-1,op)) < 1) {
	    fprintf(stderr,"Don't understand port %s aliases %s\n",optarg,op);
	    exit(1);
	  }
	}
      }
      break;
    case 'F':			/* flooding trace alias */
      fflags = C_IS_TRACE;
    case 'f':			/* flooding wide alias */
      fflags |= C_IS_FLOOD;
      if (N_floods >= MAXALIASES) {
	fprintf(stderr,"too many flooding aliases!\n");
	exit(1);
      }
      if (convert_call_entry(optarg,Floods[N_floods].call.ax25_call) < 0) {
	fprintf(stderr,"Don't understand callsign %s\n",optarg);
	exit(1);
      }
      if ((Floods[N_floods].call.ax25_call[ALEN]&SSID) != 0) {
	fprintf(stderr,"-f %s: flooding callsign must have a zero SSID\n",
		optarg);
	exit(1);
      }
      Floods[N_floods].flags = fflags;
      Floods[N_floods++].len = strlen(optarg);
      break;
    case 'w':			/* directional unproto path */
      ++p;
    case 'e':
      ++p;
    case 's':
      ++p;
    case 'n':
      ++p;
      ++Digi_SSID;
      if (unproto(&Path[p],optarg) < 0) {
	fprintf(stderr,"Don't understand %c path %s\n",toupper(s),optarg);
	exit(1);
      }
      break;
    case 'd':			/* DIGI-style(non-SSID) unproto path */
      ++Have_digipath;
      if (parsecalls(Digipath,DIGISIZE,optarg)!=DIGISIZE) {
	fprintf(stderr,"Don't understand %c path %s\n",toupper(s),optarg);
	exit(1);
      }
      break;
    case 't':			/* tag to tack on to end of all packets */
      Tag = optarg;
      Taglen = strlen(optarg);
      if (*Tag == '-' && Taglen == 1) {
	Tag = NULL;
	Taglen = 0;
      }
      break;
    case 'k':
      if ((Keep = atoi(optarg)) <= 0)
	Keep = 20;		/* default keep is 20 */
      break;
    case 'l':
      Logfile = optarg;		/* log digipeated packets */
      break;
    case 'i':
      Idinterval = atoi(optarg);
      break;
    case 'C':
      I_flags |= SUBST_MYCALL;
      break;
    case 'c':
      I_flags &= ~SUBST_MYCALL;
      break;
    case 'M':
      I_flags |= MICE_XLATE;
      break;
    case 'm':
      I_flags &= ~MICE_XLATE;
      break;
    case 'X':
      I_flags |= X1J4_XLATE;
      break;
    case 'x':
      I_flags &= ~X1J4_XLATE;
      break;
    case 'D':
      Kill_dupes++;
      break;
    case 'L':
      Kill_loops++;
      break;
    case 'V':
      printf("aprsdigi: %s-%s\n",PACKAGE,VERSION);
      exit(0);
    case '?':
      usage();
    }
  }
  /* sanity-check args */
  if (N_intf <= 0) {
    fprintf(stderr,"aprsdigi: must specify at least one port with -p\n");
    exit(1);
  }
}

static void
usage()
{
  fprintf(stderr,"Usage: aprsdigi [-CcDLMXv] [-n|s|e|w path] [-fF call] [-t tag] [-k secs] [-l logfile] [-i interval] [-p port:alias1,alias2...] ...\n");
  fprintf(stderr," general options:\n");
  fprintf(stderr," -v       -- produce verbose debugging output\n");
  fprintf(stderr," -T       -- test mode: listen to my packets too\n");
  fprintf(stderr," -D       -- suppress Duplicate packets\n");
  fprintf(stderr," -L       -- suppress Looping packets\n");
  fprintf(stderr," -V       -- print program version and exit\n");
  fprintf(stderr," -n|s|e|w -- set North|South|East|West SSID directional path\n");
  fprintf(stderr," -f       -- set Flooding callsign (usually WIDE)\n");
  fprintf(stderr," -F       -- set Flooding TRACE callsign (usually TRACE)\n");
  fprintf(stderr," -k       -- seconds of old dupes to remember\n");
  fprintf(stderr," -l       -- log digipeated packets here\n");
  fprintf(stderr," per-interface options (put *before* each -p as needed):\n");
  fprintf(stderr," -C (-c)  -- do (not) perform Mycall substitution\n");
  fprintf(stderr," -M (-m)  -- do (not) perform Mic-E translation\n");
  fprintf(stderr," -X (-x)  -- do (not) perform X1J4 translation\n");
  fprintf(stderr," -i       -- seconds between ID transmissions\n");
  fprintf(stderr,"             (Use \"-i 0\" to disable IDs)\n");
  fprintf(stderr," -t       -- default tag text to add to received packets\n");
  fprintf(stderr,"             (Use \"-t -\" to reset to empty)\n");
  fprintf(stderr,"\n");
  fprintf(stderr," -p       -- ax25 port name and optional list of aliases\n");
  fprintf(stderr,"             (primary callsign is obtained from the port)\n");
  exit(1);
}

static void
newusage()
{
  fprintf(stderr,"aprsdigi: NEW VERSION: -a, -x and -r switches have been replaced with -p:\n");
  usage();
}

static void
do_ports()
{
  struct ifreq ifr;
  char *rxdev = NULL, *txdev = NULL;
  int proto = (Testing)?ETH_P_ALL:ETH_P_AX25;
  int n;
  struct interface *i;

  convert_call_entry("APRS",Aprscall.ax25_call);
  convert_call_entry("WIDE",Widecall.ax25_call);
  convert_call_entry("trace",Trace_dummy.ax25_call);
  if (ax25_config_load_ports() == 0)
    fprintf(stderr, "aprsdigi: no AX.25 port data configured\n");
  
  for (n = 0, i = Intf; n < N_intf; i++,n++) {
    ++i->n_aliases;	/* count the zeroth element too! */
    if ((i->dev = ax25_config_get_dev(i->port)) == NULL) {
      fprintf(stderr, "aprsmon: invalid port name - %s\n", i->port);
      exit(1);
    }
    if ((i->rsock = socket(AF_PACKET, SOCK_PACKET, htons(proto))) == -1) {
      perror(i->port);
      exit(1);
    }
#ifdef USE_TWO_SOCKS
    /* when transmitting via SOCK_PACKET get kprintf complaining,
       "protocol 0000 is buggy, dev sm0" so try to figure out the
       correct way to setup to transmit ax.25.  I think it's actually
       a kernel bug at this point....*/
    if ((i->tsock = socket(AF_PACKET, SOCK_PACKET, htons(proto))) == -1) {
      perror(i->port);
      exit(1);
    }
    strcpy(i->tsa.sa_data, i->dev);
    i->tsa.sa_family = AF_AX25;
#else
    i->tsock = i->rsock;
    i->tsa = i->rsa;
#endif
    strcpy(i->rsa.sa_data, i->dev);
    i->rsa.sa_family = AF_AX25;
    if (bind(i->rsock, &i->rsa, sizeof(struct sockaddr)) < 0) {
      perror(i->dev);
      exit(1);
    }
    strcpy(ifr.ifr_name, i->dev); /* get this port's callsign */
    if (ioctl(i->rsock, SIOCGIFHWADDR, &ifr) < 0) {
      perror(i->dev);
      exit(1);
    }
    if (ifr.ifr_hwaddr.sa_family != AF_AX25) {
      fprintf(stderr,"%s: not AX25\n",i->port);
      exit(1);
    }
    /* set mycall (zeroth alias) to the port's */
    bcopy(ifr.ifr_hwaddr.sa_data,MYCALL(n).ax25_call,sizeof(MYCALL(n).ax25_call));
    MYCALL(n).ax25_call[ALEN] |= ESSID_SPARE|SSSID_SPARE;
  }
  calltab_init();
}

static void
set_sig()
{
  signal(SIGHUP,cleanup);
  signal(SIGINT,cleanup);
  signal(SIGQUIT,cleanup);
  signal(SIGTERM,cleanup);
  signal(SIGPWR,cleanup);
  signal(SIGPIPE,cleanup);
  signal(SIGUSR1,print_stats);
  signal(SIGUSR2,reset_stats);
  signal(SIGALRM,identify);
}

/*
 * compare two ax.25 "flooding" addresses.
 * returns: floodtype
 * args: f is the known flood call prefix, len is its len.
 *       c is what we are testing against it to see if it is
 *       flood, floodN, or none.
 */
static int
floodcmp(ax25_address *f, int len, ax25_address *c)
{
  int i;

  for (i = 0; i < len; i++) 
    if ((f->ax25_call[i] & 0xFE) != (c->ax25_call[i] & 0xFE))
      return C_IS_NOFLOOD;
  return floodtype(c,i);
}

/*
 * floodtype: Is it a flood, or floodN, or not a flood at all?
 * The call should have blanks in the remainder
 * (e.g. WIDE-3), or a single digit followed by blanks
 * (WIDE3-3).  If it has other junk then is is not a flood (WIDENED).
 */
static int
floodtype(ax25_address *c, int i)
{
  int dig = (c->ax25_call[i]&0xFE)>>1;
  if (dig == ' ')
    return C_IS_FLOOD;
  else if (dig >= '0' && dig <= '9' 
	   && ((i>=5) || (c->ax25_call[i+1] & 0xFE)>>1 == ' '))
    return C_IS_FLOOD|C_IS_FLOODN;
  else
    return C_IS_NOFLOOD;
}

static void
check_config()
{
  struct interface *i;
  int n, j;
  struct callsign_list *cl;
  
  printf("Linux APRS(tm) digipeater\nportions Copyright (c) 1996,1997,1999 Alan Crosswell, n2ygk@weca.org\n");
  printf("Version: aprsdigi %s-%s\n",PACKAGE,VERSION);
  printf("This is free software covered under the GNU Public License.\n");
  printf("There is no warranty.  See the file COPYING for details.\n\n");
  printf("My interfaces:\n");
  printf("Interface Callsign  Aliases...\n");
  for (i = Intf, n = 0; n < N_intf; i++,n++) {
    printf("%-9.9s",i->dev);
    for (j = 0; j < i->n_aliases; j++) {
      printf(" %-9.9s",ax2asc(&i->aliases[j]));
    }
    printf("\n");
  }
  printf("My interface options:\n");
  printf("Interface Flags    ID interval tag\n");
  for (i = Intf, n = 0; n < N_intf; i++,n++) {
    printf("%-9.9s",i->dev);
#define onoff(f,c) ((i->i_flags&(f))?(c):(c+' '))
    printf(" %c%c%c    ",onoff(SUBST_MYCALL,'C'),
	   onoff(MICE_XLATE,'M'), onoff(X1J4_XLATE,'X'));
#undef onoff
    printf(" %4d (%02d:%02d)",i->idinterval, i->idinterval/60,i->idinterval%60);
    printf(" %s\n",(i->taglen)?i->tag:"(none)\n");
  }
  printf("My callsigns and aliases (routing table):\n");
  printf("Callsign  Interfaces...\n");
  for (cl = calltab; cl; cl = cl->next) {
    struct interface_list *l;
    char t[10];

    strcpy(t,ax2asc(cl->callsign));
    if (cl->flags&C_IS_FLOOD)
      strcat(t,"n-n");		/* indicate a WIDEn-n */
    printf("%-9.9s",t);
    for (l = cl->l; l; l = l->next) {
      printf(" %-9.9s",l->i->port);
    }
    printf("\n");
  }
  if (N_floods == 0)
    printf("No flooding callsigns\n");
  if (Have_digipath) {
    printf("SSID-based omnidirectional routing:\n");
    if (N_floods) {		/* doing WIDEn-n so this won't get used */
      fprintf(stderr,"ERROR: -d is inconsistent with WIDEn-n flooding\n");
      exit(1);
    }
    if (intf_of(&Digipath[0]) == NULL) {
      fprintf(stderr,"ERROR: %s: first callsign in omni route is not one of my callsigns.\n",
	      ax2asc(&Path[n].fsa_digipeater[0]));
      exit(1);
    }
    for (j = 0; j < DIGISIZE; j++) {
      printf("%-9.9s ",ax2asc(&Digipath[j]));
    }
    printf("\n");
  }
  if (Digi_SSID) {
    printf("SSID-based directional routing:\n");
    for (n = 0; n < 4; n++) {
      if (intf_of(&Path[n].fsa_digipeater[0]) == NULL) {
	fprintf(stderr,"ERROR: %s: first callsign in %c route is not one of my callsigns.\n",
		ax2asc(&Path[n].fsa_digipeater[0]),Dirs[n]);
	exit(1);
      }
      printf("\n%c:        ",Dirs[n]);
      for (j = 0; j < Path[n].fsa_ax25.sax25_ndigis; j++)
	printf("%-9.9s ",ax2asc(&Path[n].fsa_digipeater[j]));
    }
    printf("\n");
  } else {
    printf("SSID-based routing: (none)\n");
  }
  /* flags */
  printf("keep dupes for: %d seconds\n",Keep);
  printf("log file: %s\n",(Logfile)?Logfile:"(none)");
#define onoff(x) (x)?"ON":"OFF"
  printf("kill dupes: %s loops: %s  testing: %s\n",
	 onoff(Kill_dupes), onoff(Kill_loops), 
	 onoff(Testing));
#undef onoff
  fflush(stdout);
}
