// ---------------------------------------------------------------------------
// - cstr.cxx                                                                -
// - standard system library - c string function implementation              -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "cstr.hpp"
#include "cstr.hxx"

namespace aleph {

  // get the size of the c string. The function is safe with null pointer.

  long c_strlen (const char* s) {
    if (s == 0) return 0;
    return strlen (s);
  }

  // same as above but inlined locally

  static inline long c_lenstr (const char* s) {
    if (s == 0) return 0;
    return strlen (s);
  }

  // Compare two strings and returns true if they are equals. 

  bool c_strcmp (const char* s1, const char* s2) {
    if (!s1 && !s2) return true;
    if (!s1 &&  s2 && (*s2 == '\0')) return true;
    if (!s2 &&  s1 && (*s1 == '\0')) return true;
    if ((s1 && !s2) || (!s1 && s2)) return false;
    if (*s1 != *s2) return false;
    return (strcmp (s1,s2) == 0) ? true : false;
  }

  // compare two strings and returns true if they are equals up to n characters. 

  bool c_strncmp (const char* s1, const char* s2, const long size) {
    if (size == 0) return true;
    long len1 = c_lenstr (s1);
    long len2 = c_lenstr (s2);
    if ((len1 == 0) && (len2 == 0)) return false;
    if ((len1 == 0) && (len2 != 0)) return false;
    if ((len1 != 0) && (len2 == 0)) return false;

    // normal compare
    int status = strncmp (s1,s2,size);
    return (status == 0) ? true : false;
  }

  // create a string from a character.

  char* c_strmak (const char c) {
    char* result = new char[2];
    result[0] = c;
    result[1] = '\0';
    return result;
  }

  // duplicate a string. If the string is null or has a 0 length, 
  // the function returns the null pointer.
  
  char* c_strdup (const char* s) {
    // check for null string if the length is null
    long len = c_strlen (s);
    if (len == 0) return (char*) 0;
    
    // allocate a new string with new so delete can be used
    char* result = new char[len + 1];
    for (long i = 0; i < len + 1; i++)
      result[i] = s[i];
    
    return result;
  }

  // copy a string from a source to destination
  
  void c_strcpy (char* dst, const char* src) {
    // standard check as usual
    if (!dst) return;
    // if the source is null we simply return the null string
    if ((c_strlen (src) == 0) || (!dst)) {
      dst[0] = (char) 0;
      return;
    }
    // copy the string
    strcpy (dst,src);
  }

  // concatenate a string with another. The string must allocated enough.

  void c_strcat (char* dst, const char* src) {
    // standard check as usual
    if ((!src) || (c_strlen (src) == 0) || (!dst)) return;
    // concatenate the string
    strcat (dst,src);
  }

  // remove the leading blank from a string and return a new string

  char* c_rmlead (const char* s) {
    long len = c_strlen (s);
    if (len == 0) return (char*) 0;
    // remove leading blank
    while ((*s != '\0') && ((*s == ' ') || (*s == '\t'))) s++;
    // now copy
    char* result = c_strdup (s);
    return result;
  }

  // remove the trailing blank from a string and return a new string

  char* c_rmtrail (const char* s) {
    long len = c_strlen (s);
    if (len == 0) return (char*) 0;
    char* buffer = c_strdup (s);
    char* end    = buffer + len - 1;
    // remove trailing blank
    while ((end != s) && ((*end == ' ') || (*end == '\t'))) *end-- = '\0';
    // now copy
    char* result = c_strdup (buffer);
    return result;    
  }

  // convert a string to upper case
  
  char* c_toupper (const char* s) {
    long len = c_strlen (s);
    if (len == 0) return (char*) 0;
    char* result = new char[len + 1];
    for (long i = 0; i < len; i++) {
      char c = s[i];
      if ((c >= 'a') && (c <= 'z')) c = c - 'a' + 'A';
      result[i] = c;
    }
    result[len] = '\0';
    return result;
  }

  // convert a string to lower case
  
  char* c_tolower (const char* s) {
    long len = c_strlen (s);
    if (len == 0) return (char*) 0;
    char* result = new char[len + 1];
    for (long i = 0; i < len; i++) {
      char c = s[i];
      if ((c >= 'A') && (c <= 'Z')) c = c - 'A' + 'a';
      result[i] = c;
    }
    result[len] = '\0';
    return result;
  }
}
