// ---------------------------------------------------------------------------
// - Table.cpp                                                               -
// - aleph:odb library - table class implementation                          -
// ---------------------------------------------------------------------------
// - This program is free software;  you can rodbstribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Table.hpp"
#include "Reader.hpp"
#include "Odbsid.hpp"
#include "Lexical.hpp"
#include "Integer.hpp"
#include "Runnable.hpp"
#include "Exception.hpp"
#include "PrintTable.hpp"

namespace aleph {
  // the table supported quarks
  static const long QUARK_ADD       = String::intern ("add");
  static const long QUARK_GET       = String::intern ("get");
  static const long QUARK_SET       = String::intern ("set");
  static const long QUARK_LENGTH    = String::intern ("length");
  static const long QUARK_ADDKEY    = String::intern ("add-key");
  static const long QUARK_GETKEY    = String::intern ("get-key");
  static const long QUARK_MAPKEY    = String::intern ("map-key");
  static const long QUARK_GETNAME   = String::intern ("get-name");
  static const long QUARK_SETNAME   = String::intern ("set-name");
  static const long QUARK_ADDINFO   = String::intern ("add-info");
  static const long QUARK_GETINFO   = String::intern ("get-info");
  static const long QUARK_SETINFO   = String::intern ("set-info");
  static const long QUARK_ADDHEAD   = String::intern ("add-header");
  static const long QUARK_GETHEAD   = String::intern ("get-header");
  static const long QUARK_SETHEAD   = String::intern ("set-header");
  static const long QUARK_ADDFOOT   = String::intern ("add-footer");
  static const long QUARK_GETFOOT   = String::intern ("get-footer");
  static const long QUARK_SETFOOT   = String::intern ("set-footer");
  static const long QUARK_EVALINFO  = String::intern ("eval-info");
  static const long QUARK_EVALHEAD  = String::intern ("eval-header");
  static const long QUARK_EVALFOOT  = String::intern ("eval-footer");
  static const long QUARK_ADDRECORD = String::intern ("add-record");

  // this function computes the maximum between two numbers
  static inline long max (const long x, const long y) {
    return (x < y) ? y : x;
  }

  // this procedure returns a new table object for deserialization
  static Serial* mksob (void) {
    return new Table;
  }
  // register this cell serial id
  static const t_byte SERIAL_ID = Serial::setsid (SERIAL_SHTT_ID, mksob);

  // create a nil table

  Table::Table (void) {
    d_quark = 0;
  }

  // create a new table by name

  Table::Table (const String& name) {
    d_quark = name.toquark ();
  }

  // return the object name

  String Table::repr (void) const {
    return "Table";
  }

  // return the table serial id

  t_byte Table::serialid (void) const {
    return SERIAL_SHTT_ID;
  }

  // serialize a table

  void Table::wrstream (Output& os) const {
    rdlock ();
    // save the table name
    const String& name = String::qmap (d_quark);
    name.wrstream (os);
    // save the descriptor
    d_info.wrstream (os);
    // save the header
    d_head.wrstream (os);
    // save the footer
    d_foot.wrstream (os);
    // save the vector
    d_vrcds.wrstream (os);
    unlock ();
  }

  // deserialize this table

  void Table::rdstream (Input& is) {
    wrlock ();
    // get the table name
    String name;
    name.rdstream (is);
    d_quark = name.toquark ();
    // get the descriptor
    d_info.rdstream (is);
    // get the header
    d_head.rdstream (is);
    // get the footer
    d_foot.rdstream (is);
    // get the vector
    d_vrcds.rdstream (is);
    unlock ();
  }

  // import data in this table

  void Table::import (Input* is) {
    if (is == nilp) return;
    // create a new reader
    Reader rd (is);
    wrlock ();
    Form* form = nilp;
    try {
      // create a new reader
      while (true) {
	form = rd.parse ();
	if (form == nilp) break;
	add (form);
      }
    } catch (Exception& e) {
      if (form == nilp) {
	e.setlnum (rd.getlnum ());
      } else {
	e.setlnum (form->getlnum ());
	Object::cref (form);
      }
      unlock ();
      throw;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // dump a table into an output stream

  void Table::report (Output& os, long max, long start, bool flag) const {
    rdlock ();
    // get the number of rows
    long rows = length ();
    // check for start index
    if ((start < 0) || (start >= rows)) {
      unlock ();
      throw Exception ("table-error", "start index out of range for dump");
    }
    // check for max index
    long tlen = start + ((max == 0) ? rows : max);
    if (tlen > rows) {
      unlock ();
      throw Exception ("table-error", "max index is out of range");
    }
    try {
      // create a print table
      long cols = getcols ();
      PrintTable result (cols);
      for (long i = start; i < tlen; i++) {
	Record* rcd = get (i);
	if (rcd == nilp) continue;
	long row = result.add ();
	// get the record length
	long rlen = rcd->length ();
	for (long j = 0; j < rlen; j++) {
	  Literal* lobj = rcd->getlobj (j);
	  if (lobj == nilp) {
	    result.set (row, j, "nil");
	  } else {
	    String data = flag ? lobj->tostring () : lobj->toliteral ();
	    result.set (row, j, data);
	  }
	}
	// fill the rest of the table
	for (long j = rlen; j < cols; j++) result.set (row, j, "nil");
      }
      result.format (os);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // return the table name

  String Table::getname (void) const {
    rdlock ();
    String result = String::qmap (d_quark);
    unlock ();
    return result;
  }

  // set the table name

  void Table::setname (const String& name) {
    rdlock ();
    d_quark = name.toquark ();
    unlock ();
  }

  // add a literal in the info descriptor
  
  void Table::addinfo (Literal* lobj) {
    d_info.add (lobj);
  }

  // get an info cell by index

  Cell* Table::getinfo (const long index) const {
    rdlock ();
    try {
      Cell* result = dynamic_cast <Cell*> (d_info.get (index));
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // evaluate an info cell by index

  Literal* Table::evalinfo (const long index) const {
    rdlock ();
    try {
      Cell* cell = getinfo (index);
      Literal* result = (cell == nilp) ? nilp : cell->get ();
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set a literal in the info descriptor by index
  
  void Table::setinfo (const long index, Literal* lobj) {
    wrlock ();
    try {
      Cell* cell = dynamic_cast <Cell*> (d_info.get (index)); 
      if (cell != nilp) cell->set (lobj);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // add a literal in the header
  
  void Table::addhead (Literal* lobj) {
    d_head.add (lobj);
  }

  // get a header cell by index

  Cell* Table::gethead (const long index) const {
    rdlock ();
    try {
      Cell* result = dynamic_cast <Cell*> (d_head.get (index));
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // evaluate a header cell by index

  Literal* Table::evalhead (const long index) const {
    rdlock ();
    try {
      Cell* cell = gethead (index);
      Literal* result = (cell == nilp) ? nilp : cell->get ();
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set a literal in the header by index
  
  void Table::sethead (const long index, Literal* lobj) {
    wrlock ();
    try {
      Cell* cell = dynamic_cast <Cell*> (d_head.get (index)); 
      if (cell != nilp) cell->set (lobj);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // add a literal in the footer
  
  void Table::addfoot (Literal* lobj) {
    d_foot.add (lobj);
  }

  // get a footer cell by index

  Cell* Table::getfoot (const long index) const {
    rdlock ();
    try {
      Cell* result = dynamic_cast <Cell*> (d_foot.get (index));
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // evaluate a footer cell by index

  Literal* Table::evalfoot (const long index) const {
    rdlock ();
    try {
      Cell* cell = getfoot (index);
      Literal* result = (cell == nilp) ? nilp : cell->get ();
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set a literal in the footer by index
  
  void Table::setfoot (const long index, Literal* lobj) {
    wrlock ();
    try {
      Cell* cell = dynamic_cast <Cell*> (d_foot.get (index)); 
      if (cell != nilp) cell->set (lobj);
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // add a key in this table

  void Table::addkey (const String& key) {
    // make sure the key is valid
    if (Lexical::valid (key) == false)
      throw Exception ("key-error", "invalid table key", key);
    wrlock ();
    // check if the key does exist
    if (d_keys.exists (key) == true) {
      unlock ();
      throw Exception ("key-error", "key already exist", key);
    }
    d_keys.add (key);
    unlock ();
  }
   
  // find a key index by name

  long Table::mapkey (const String& name) const {
    return d_keys.index (name);
  }

  // return a key name by index

  String Table::getkey (const long index) const {
    return d_keys.get (index);
  }

  // add a record in this table

  void Table::add (Record* rcd) {
    if (rcd == nilp) return;
    wrlock ();
    d_vrcds.append (rcd);
    unlock ();
  }

  // add a list of literal as a record

  void Table::add (Cons* cons) {
    if (cons == nilp) return;
    wrlock ();
    Record* rcd = new Record;
    try {
      // iterate in the cons cell
      while (cons != nilp) {
	rcd->add (cons->getcar ());
	cons = cons->getcdr ();
      }
      d_vrcds.append (rcd);
    } catch (...) {
      Object::cref (rcd);
      unlock ();
      throw;
    }
  }

  // add a vector of literal as a record

  void Table::add (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc == 0) return;
    wrlock ();
    Record* rcd = new Record;
    try {
      for (long i = 0; i < argc; i++) rcd->add (argv->get (i));
      d_vrcds.append (rcd);
      unlock ();
    } catch (...) {
      Object::cref (rcd);
      unlock ();
      throw;
    }
  }

  // get a record by index

  Record* Table::get (const long index) const {
    rdlock ();
    Record* result = dynamic_cast <Record*> (d_vrcds.get (index));
    unlock ();
    return result;
  }

  // set a record in this table by index

  void Table::set (const long index, Record* rcd) {
    wrlock ();
    d_vrcds.set (index, rcd);
    unlock ();
  }

  // return the length of the table list

  long Table::length (void) const {
    rdlock ();
    long result = d_vrcds.length ();
    unlock ();
    return result;
  }

  // return the number of columns
  
  long Table::getcols (void) const {
    rdlock ();
    // get the table length
    long tlen = length ();
    // compute the maximum columns
    long result = 0;
    for (long i = 0; i < tlen; i++) {
      Record* rcd = get (i);
      if (rcd == nilp) continue;
      result = max (result, rcd->length ());
    }
    unlock ();
    return result;
  }

  // create a new table in a generic way

  Object* Table::mknew (Vector* argv) {
    // get number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 argument
    if (argc == 0) return new Table;
    // check for 1 argument
    if (argc == 1) {
      String name = argv->getstring (0);
      return new Table (name);
    }
    throw Exception ("argument-error", "too many argument with table");
  }

  // apply a table method with a set of arguments and a quark

  Object* Table::apply (Runnable* robj, Nameset* nset, const long quark,
			Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for generic quark
    if (quark == QUARK_ADD) {
      add (argv);
      return nilp;
    }
    if (quark == QUARK_ADDINFO) {
      for (long i = 0; i < argc; i++) {
	Object* obj = argv->get (i);
	if (obj == nilp) continue;
	Literal* lobj = dynamic_cast <Literal*> (obj);
	if (lobj == nilp) 
	  throw Exception ("type-error", "illegal object for info add",
			   Object::repr (obj));
	addinfo (lobj);
      }
      return nilp;
    }
    if (quark == QUARK_ADDHEAD) {
      for (long i = 0; i < argc; i++) {
	Object* obj = argv->get (i);
	if (obj == nilp) continue;
	Literal* lobj = dynamic_cast <Literal*> (obj);
	if (lobj == nilp) 
	  throw Exception ("type-error", "illegal object for header add",
			   Object::repr (obj));
	addhead (lobj);
      }
      return nilp;
    }
    if (quark == QUARK_ADDFOOT) {
      for (long i = 0; i < argc; i++) {
	Object* obj = argv->get (i);
	if (obj == nilp) continue;
	Literal* lobj = dynamic_cast <Literal*> (obj);
	if (lobj == nilp) 
	  throw Exception ("type-error", "illegal object for footer add",
			   Object::repr (obj));
	addfoot (lobj);
      }
      return nilp;
    }

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_LENGTH ) return new Integer (length  ());
      if (quark == QUARK_GETNAME) return new String  (getname ());
    }
    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETNAME) {
	String name = argv->getstring (0);
	setname (name);
	return nilp;
      }
      if (quark == QUARK_ADDKEY) {
	String key = argv->getstring (0);
	addkey (key);
	return nilp;
      }
      if (quark == QUARK_MAPKEY) {
	String key = argv->getstring (0);
	return new Integer (mapkey (key));
      }
      if (quark == QUARK_GETKEY) {
	long index = argv->getint (0);
	return new String (getkey (index));
      }
      if (quark == QUARK_GET) {
	long idx = argv->getint (0);
	rdlock ();
	try {
	  Object* result = get (idx);
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_ADDRECORD) {
	Object* obj = argv->get (0);
	Record* rcd = dynamic_cast <Record*> (obj);
	if (rcd != nilp) {
	  add (rcd);
	  return nilp;
	}
	throw Exception ("type-error", "invalid object to add in table",
			 Object::repr (obj));
      }
      if (quark == QUARK_GETINFO) {
	long index = argv->getint (0);
	rdlock ();
	try {
	  Object* result = getinfo (index);
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_EVALINFO) {
	long index = argv->getint (0);
	rdlock ();
	try {
	  Object* result = evalinfo (index);
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_GETHEAD) {
	long index = argv->getint (0);
	rdlock ();
	try {
	  Object* result = gethead (index);
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_EVALHEAD) {
	long index = argv->getint (0);
	rdlock ();
	try {
	  Object* result = evalhead (index);
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_GETFOOT) {
	long index = argv->getint (0);
	rdlock ();
	try {
	  Object* result = getfoot (index);
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_EVALFOOT) {
	long index = argv->getint (0);
	rdlock ();
	try {
	  Object* result = evalfoot (index);
	  robj->post (result);
	  unlock ();
	  return result;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
    }
    // dispatch 2 argument
    if (argc == 2) {
      if (quark == QUARK_SETINFO) {
	long    idx = argv->getint (0);
	Object*  obj  = argv->get (1);
	Literal* lobj = dynamic_cast <Literal*> (obj);
	if ((obj != nilp) && (lobj == nilp)) 
	  throw Exception ("type-error", "invalid object to set in info",
			   obj->repr ());
	setinfo (idx, lobj);
	return nilp;
      }
      if (quark == QUARK_SETHEAD) {
	long    idx = argv->getint (0);
	Object*  obj  = argv->get (1);
	Literal* lobj = dynamic_cast <Literal*> (obj);
	if ((obj != nilp) && (lobj == nilp)) 
	  throw Exception ("type-error", "invalid object to set in header",
			   obj->repr ());
	sethead (idx, lobj);
	return nilp;
      }
      if (quark == QUARK_SETFOOT) {
	long    idx = argv->getint (0);
	Object*  obj  = argv->get (1);
	Literal* lobj = dynamic_cast <Literal*> (obj);
	if ((obj != nilp) && (lobj == nilp)) 
	  throw Exception ("type-error", "invalid object to set in footer",
			   obj->repr ());
	setfoot (idx, lobj);
	return nilp;
      }
      if (quark == QUARK_SET) {
	long    idx = argv->getint (0);
	Object* obj = argv->get (1);
	Record* rcd = dynamic_cast <Record*> (obj);
	if ((obj != nilp) && (rcd == nilp)) 
	  throw Exception ("type-error", "invalid object to set in table",
			   obj->repr ());
	set (idx, rcd);
	return nilp;
      }
    }
    // call the persist method
    return Persist::apply (robj, nset, quark, argv);
  }
}
