// ---------------------------------------------------------------------------
// - Lexical.cpp                                                             -
// - aleph engine - lexical name class implementation                        -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2003 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Engsid.hpp"
#include "Lexical.hpp"
#include "Nameset.hpp"
#include "Integer.hpp"
#include "Runnable.hpp"
#include "Exception.hpp"


namespace aleph {
  // the lexical quarks
  static const long QUARK_NIL = String::intern ("nil");
  static const long QUARK_MAP = String::intern ("map");

  // this procedure returns a new lexical object for deserialization
  static Serial* mksob (void) {
    return new Lexical;
  }
  // register this lexical serial id
  static const t_byte SERIAL_ID = Serial::setsid (SERIAL_LEXL_ID, mksob);

  // return true if the character is a valid lexical one

  bool Lexical::valid (const char c) {
    if ((c >= 'a') && (c <= 'z')) return true;
    if ((c >= 'A') && (c <= 'Z')) return true;
    if ((c >= '0') && (c <= '9')) return true;

    if (c == '.') return true;
    if (c == '+') return true;
    if (c == '-') return true;
    if (c == '*') return true;
    if (c == '/') return true;
    if (c == '!') return true;
    if (c == '=') return true;
    if (c == '.') return true;
    if (c == '>') return true;
    if (c == '<') return true;
    if (c == '?') return true;
    return false;
  }

  // return true if the name is valid lexical one

  bool Lexical::valid (const String& name) {
    long len = name.length ();
    if (len == 0) return false;
    for (long i = 0; i < len; i++)
      if (Lexical::valid (name[i]) == false) return false;
    return true;
  }

  // create a default lexical
 
  Lexical::Lexical (void) {
    d_lnum  = 0;
    d_quark = 0;
  }

  // create a lexical with a name

  Lexical::Lexical (const String& name) {
    if (Lexical::valid (name) == false) 
      throw Exception ("syntax-error","invalid lexical name", name);
    d_name  = name;
    d_quark = name.toquark ();
    d_lnum  = 0;
  }

  // create a lexical with a name and a line number

  Lexical::Lexical (const String& name, const long lnum) {
    if (Lexical::valid (name) == false) 
      throw Exception ("syntax-error","invalid lexical name", name);
    d_name  = name;
    d_quark = name.toquark ();
    d_lnum  = lnum;
  }
  
  // copy constructor for this lexical

  Lexical::Lexical (const Lexical& that) {
    d_name  = that.d_name;
    d_quark = that.d_quark;
    d_lnum  = that.d_lnum;
  }

  // return the class name

  String Lexical::repr (void) const {
    return "Lexical";
  }

  // return a clone of this lexical

  Object* Lexical::clone (void) const {
    return new Lexical (*this);
  }

  // return the lexical serial code

  t_byte Lexical::serialid (void) const {
    return SERIAL_LEXL_ID;
  }

  // serialize this lexical

  void Lexical::wrstream (Output& os) const {
    rdlock ();
    Integer lnum (d_lnum);
    d_name.wrstream (os);
    lnum.wrstream   (os);
    unlock ();
  }

  // deserialize this lexical

  void Lexical::rdstream (Input& is) {
    wrlock ();
    String  sval;
    Integer ival;
    sval.rdstream (is);
    ival.rdstream (is);
    d_name  = sval;
    d_lnum  = ival.tointeger ();
    d_quark = sval.toquark ();
    unlock ();
  }

  // return true if the lexical is the nil quark

  bool Lexical::isnil (void) const {
    return (d_quark == QUARK_NIL);
  }

  // create a lexical in a generic way

  Object* Lexical::mknew (Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // check for 0 argument
    if (argc == 0) return new Lexical;

    // check for 1 argument
    if (argc == 1) {
      String name = argv->getstring (0);
      return new Lexical (name);
    }
    // illegal arguments
    throw Exception ("argument-error", "too many arguments with lexical");
  }

  // set this lexical name as a const symbol

  Object* Lexical::cdef (Runnable* robj, Nameset* nset, Object* object) {
    return nset->cdef (robj, nset, d_quark, object);
  }

  // set this lexical name with an object

  Object* Lexical::vdef (Runnable* robj, Nameset* nset, Object* object) {
    return nset->vdef (robj, nset, d_quark, object);
  }

  // evaluate this lexical in the current environment

  Object* Lexical::eval (Runnable* robj, Nameset* nset) {
    return nset->eval (robj,nset,d_quark);
  }

  // apply this lexical with a set of arguments and a quark

  Object* Lexical::apply (Runnable* robj, Nameset* nset, const long quark,
			  Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_MAP) {
	Object* result = (nset == nilp) ? nilp : nset->find (d_quark);
	robj->post (result);
	return result;
      }
    }

    // call the literal method
    return Literal::apply (robj, nset, quark, argv);
  }
}
