/**********************************************************************
 ** Pager class: Methods that provide the means to display only a portion
 **              of a screen dump at a time, defined by the number of
 **              lines the user has setup for their pager
 **
 **
 **
 **    Last reviewed: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef PAGER_C
#define PAGER_C

#include "config.h"
#include "sysdep.h"
#include "pager.h"
#include "memchk.h"

/****************************************************************************
 ** Pager (constructor) : opens a file for paged output
 ** 
 ** Parameters: char *fn = name of the file to open.
 **             disp_lines - the number of lines to display at a time
 **
 ** Returns: nothing
 ****************************************************************************/
Pager::Pager(char *fn, int disp_lines)
{  page_fp = NULL;
   open_pagefile(fn);
   pager_str = NULL;
   the_offset = 0;
   num_lines = disp_lines;

   if (num_lines == 0)
      num_lines = 10000;
}


/****************************************************************************
 ** Pager (constructor) : sets up a string for paged output
 ** 
 ** Parameters: the_str - a pointer to the string to read in
 **             disp_lines - the number of lines to display at a time
 **
 ** Returns: nothing
 ****************************************************************************/
Pager::Pager(Strings *the_str, int disp_lines)
{  page_fp = NULL;
   pager_str = the_str;
   the_offset = 0;
   num_lines = disp_lines;

   if (num_lines == 0)
      num_lines = 10000;
}


Pager::~Pager()
{
   if (page_fp != NULL)
      close_pagefile();
}

/****************************************************************************
 ** paged_read - opens a file or reads a string for paged output
 ** 
 ** Parameters: the_connection - the connection we send to
 **
 ** Returns: 1 for success, -1 for failure
 ****************************************************************************/

int Pager::paged_read(Connection *the_connection)
{
   if (pager_str != NULL)
   {
      return read_string(the_connection);
   }
   else
      return read_file(the_connection);
}


/****************************************************************************
 ** read_file - opens a *file* for paged output
 ** 
 ** Parameters: the_connection - the connection we send to
 **
 ** Returns: 1 for success, -1 for failure
 ****************************************************************************/

int Pager::read_file(Connection *the_connection)
{  char    the_line[250];
   Strings line_input;
   int     linect = 0;

   if (the_connection == NULL || page_fp == NULL)
       return -1;

   while ((!feof(page_fp)) && (linect < num_lines))
   {   
      (void)fgets(the_line,250,page_fp);
      linect++;
      
      if (feof(page_fp))
      {  rewind(page_fp);
         return 1;
      }

      line_input = the_line;
      line_input.remove_returns();

      if (the_line[0] != '^') 
         the_connection->send_to_socket(line_input.str_show());  

   }
      //   the_connection->flush();

   /* if we are at the end of the file, then return 1 saying we are done */
   if (feof(page_fp))
   {
      rewind(page_fp);
      return 1;
   }

   /* we aren't done displaying the file, return 0 */
   return 0;
}


/****************************************************************************
 ** read_string - reads the string for paged output
 ** 
 ** Parameters: the_connection - the connection we send to
 **
 ** Returns: 1 for success, -1 for failure
 ****************************************************************************/

int Pager::read_string(Connection *the_connection)
{
   char    *str_ptr;
   char    *start_ptr;
   char    tmp_char;
   int     num_newlines = 0;
   Strings line_input;

   if (the_connection == NULL || pager_str == NULL)
       return -1;

   str_ptr = pager_str->str_show();

   if ((!str_ptr) || (!(*str_ptr)))
      return 1;

   str_ptr += the_offset;
   

   if ((!str_ptr) || (!(*str_ptr)))
      return 1;

   start_ptr = str_ptr;

   /* find the end of this */
   while ((*str_ptr) && (num_newlines < num_lines))
   {
      if (*str_ptr == '\n')
         num_newlines++;

      the_offset++;
      str_ptr++;
   }

   if (*str_ptr)
   {
      tmp_char = *str_ptr;
      *str_ptr = '\0';
      line_input = start_ptr;
      *str_ptr = tmp_char;
   }
   else
      line_input = start_ptr;

   if (line_input.str_show() != NULL)
      the_connection->send_to_socket(line_input.str_show());

   if (!(*str_ptr))
      return 1;
   
   /* we aren't done displaying the string, return 0 */
   return 0;
}


int Pager::open_pagefile(char *fn)
{   
   if (page_fp != NULL)
      close_pagefile();

   if (fn != NULL)
   {   page_fp = xfopen(fn,"r", "pager_file");

       if (page_fp != NULL)
           pager_file = fn;
       else
          return -1;
   }   
   return 0;
}

int Pager::close_pagefile()
{  xfclose(page_fp, "pager_file");
   return 0;
}


int Pager::file_available() 
{ 
   return (page_fp != NULL); 
}

#endif






