/**********************************************************************
 ** Mask - An object to define a flag subset for a specific administrative
 **        position
 **    
 ** Last reviewed:
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef MASK_C
#define MASK_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "mask.h"
#include "lexer.h"
#include "objtype.h"
#include "newfuncts.h"
#include "builder.h"
#include "player.h"
#include "comflags.h"
#include "adminflags.h"

/***********************************************************************
 ** Mask (constructor) - loads mask name and attributes
 **
 ** Parameters: mask_name 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Mask::Mask(char *mask_name)
{
   Strings tmp_name;

   tmp_name.assign_word(mask_name, 1);

   set_name(tmp_name.str_show());

   modified = 0;
   obj_type = OBJ_TYPE_MASK;

   com_flags = new Flags(6);
   adm_flags = new Flags(6);
}


/***********************************************************************
 ** ~Mask (destructor) - cleans up the mask
 **
 ** Parameters: None 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Mask::~Mask(void)
{
   delete com_flags;
   delete adm_flags;
}


/***********************************************************************
 ** load_mask - loads a mask from a file into memory
 **
 ** Parameters: the_file - where we are getting the mask from
 **             error_log - the error log to write errors to
 **             is_builder - are we writing this for the builder port
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Mask::load_mask(FILE *the_file, ErrLog *error_log, int is_builder)
{
   token_record *the_token;
   Flags        *tmp_flags;
   Strings      holder;

   if (is_builder)
   {
       the_token = get_token(the_file,'\0');
 
       if (the_token->token_type != T_NUMERICAL)
       {
           error_log->log_err("Invalid format in masks file", "load_mask");
           return -1;
       }
       set_modified(atoi(the_token->the_string));
   }

   /* read in the command flags */
   tmp_flags = get_comflags();
   if (tmp_flags->read_flags(the_file, error_log) <= 0)
   {
      holder.sprintf("Error reading comflags for mask '%s'", get_name());
      error_log->log_err(holder.str_show(), "load_mask");
      return -1;
   }   

   /* read in the admin flags */
   tmp_flags = get_admflags();
   if (tmp_flags->read_flags(the_file, error_log) <= 0)
   {
      holder.sprintf("Error reading adminflags for mask '%s'", get_name()); 
      error_log->log_err(holder.str_show(), "load_mask");
      return -1;
   }   

   return 1;
}


/***********************************************************************
 ** mask - describes the mask to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Mask::describe(Builder *the_builder)
{
   the_builder->send_bldr("\n&+GMask: \t&+M%s&*\n", get_name());
   the_builder->send_bldr("&+GFlags Avail:&* comflags, adminflags\n"); 
   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the mask to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Mask::describe(Player *the_player)
{
   the_player->send_plr("\n&+GMask: \t&+M%s&*\n", get_name());
   the_player->send_plr("&+GFlags Avail:&* comflags, adminflags\n"); 
   the_player->send_plr("&+YSize: \t\t\t&+W%d&*\n", get_mem_size());

   the_player->send_plr("\n");
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **             the_parsed - the parsed structure for this
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Mask::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
        send_bldr("You can set the following attributes on a mask.\n"
               "   comflags\n");
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "comflags",
                               strlen(the_parsed->get_target1())))
   {
      Flags *tmp_comflags;
      int flagnum;
      Strings holder;

      tmp_comflags = get_comflags();
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("Set which command flag?\n");
         return -1;
      }
      holder.assign_word(the_parsed->get_speech(), 1);
     
      if ((flagnum = 
       tmp_comflags->get_by_name(holder.str_show(), comflagnames)) == -1)
      {
         the_builder->send_bldr("That is not a command flag.\n");
         return -1;
      }

      holder.assign_word(the_parsed->get_speech(), 2);
      if (holder.str_show() == NULL)
      {
         the_builder->send_bldr("Set that flag to what?\n"
                                "Valid choices are: On or Off\n");
         return -1;
      }

      if (holder.str_n_cmp("On", holder.str_len()))
         tmp_comflags->set_flag(flagnum);
      else if (holder.str_n_cmp("Off", holder.str_len()))
         tmp_comflags->clr_flag(flagnum);
      else
      {
         the_builder->send_bldr("That is not a valid setting.\n"
                             "Valid choices are: On or Off\n");
         return -1;
      }
      the_builder->send_bldr("Flag &+M%s&* has been set to: %s\n",
          comflagnames[flagnum], (tmp_comflags->get_flag(flagnum)) ? 
			     "&+GOn&*" : "&+ROff&*");
      return 1;
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "adminflags",
                               strlen(the_parsed->get_target1())))
   {
      Flags *tmp_admflags;
      int flagnum;
      Strings holder;

      tmp_admflags = get_admflags();
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("Set which administration flag?\n");
         return -1;
      }
      holder.assign_word(the_parsed->get_speech(), 1);
     
      if ((flagnum = 
       tmp_admflags->get_by_name(holder.str_show(), admflagnames)) == -1)
      {
         the_builder->send_bldr("That is not an administration flag.\n");
         return -1;
      }

      holder.assign_word(the_parsed->get_speech(), 2);
      if (holder.str_show() == NULL)
      {
         the_builder->send_bldr("Set that flag to what?\n"
                                "Valid choices are: On or Off\n");
         return -1;
      }

      if (holder.str_n_cmp("On", holder.str_len()))
         tmp_admflags->set_flag(flagnum);
      else if (holder.str_n_cmp("Off", holder.str_len()))
         tmp_admflags->clr_flag(flagnum);
      else
      {
         the_builder->send_bldr("That is not a valid setting.\n"
                             "Valid choices are: On or Off\n");
         return -1;
      }
      the_builder->send_bldr("Flag &+M%s&* has been set to: %s\n",
          admflagnames[flagnum], (tmp_admflags->get_flag(flagnum)) ? 
			     "&+GOn&*" : "&+ROff&*");
      return 1;
   }

   the_builder->send_bldr("The attribute '%s' is not a mask attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** write_object - writes the mask to a specified file in specified
 **                format
 **
 ** Parameters: the_file - the file to write to
 **             build_format - shall we use builder format or not
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
void Mask::write_object(FILE *the_file, int build_format)
{
   Flags *tmp_flags;

   fprintf(the_file, "\n%s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());

   tmp_flags = get_comflags();
   tmp_flags->write_flag(the_file);
   tmp_flags = get_admflags();
   tmp_flags->write_flag(the_file);

   return;
}


/***********************************************************************
 ** is_modified - has this mask been modified?
 **
 ** Parameters: None
 **
 ** Returns:  1 for yes, 0 for no
 **
 ***********************************************************************/
   
int Mask::is_modified(void)
{
   return modified;
}



/***********************************************************************
 ** set_modified - shall we set this?
 **
 ** Parameters: the_num - the number to set it to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
void Mask::set_modified(int the_num)
{
   modified = the_num;
}


/***********************************************************************
 ** get_comflags - gets the command flag structure
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the command flag
 **
 ***********************************************************************/
   
Flags *Mask::get_comflags()
{
   return com_flags;
}


/***********************************************************************
 ** get_admflags - gets the administration flag structure
 **
 ** Parameters: None
 **
 ** Returns: a pointer to the administration flag
 **
 ***********************************************************************/
   
Flags *Mask::get_admflags()
{
   return adm_flags;
}

/***********************************************************************
 ** copy_object - copies the mask to a mask of a different name
 **
 ** Parameters: copy_from - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Mask::copy_object(Entity *copy_obj)
{
   Mask *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_MASK)
      return 0;

   copy_from = (Mask *) copy_obj;

   /******* set the mask attributes *****/

   com_flags->copy_flags(copy_from->get_comflags());
   adm_flags->copy_flags(copy_from->get_admflags());

   return 1;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this mask is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Mask::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Mask::get_mem_size_dynamic()
{
   int  size = 0;

   size += com_flags->get_mem_size_dynamic();
   size += adm_flags->get_mem_size_dynamic();

   return size;
}

#endif

