/**********************************************************************
 ** Level - contains methods and attributes for a level object
 **
 ** Reviewed through:
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **    
 **********************************************************************/


#ifndef LEVEL_C
#define LEVEL_C

#include "sysdep.h"
#include "strings.h"
#include "level.h"
#include "lexer.h"
#include "builder.h"
#include "parse.h"
#include "utils.h"
#include "player.h"
#include "objtype.h"
#include "newfuncts.h"
#include "config.h"
#include "lvlflags.h"
#include "specials.h"
#include "newfuncts.h"

/***********************************************************************
 ** Level (constructor) - loads level name and attributes
 **
 ** Parameters: level_name - this level name 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Level::Level(char *level_name)
{
   Strings tmp_name;

   obj_type = OBJ_TYPE_LEVEL;
   tmp_name.assign_word(level_name, 1);

   set_name(tmp_name.str_show());

   modified = 0;

   set_chain_name("none");
   lvl_num = 1;
   min_str = min_dex = min_intel = min_con = min_wisdom = min_cha = min_exp = 0;

   special_req = when_awarded = NULL;
   lvl_flags = new Flags(1);
}


/***********************************************************************
 ** ~Level (destructor) - cleans up the level
 **
 ** Parameters: None 
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Level::~Level(void)
{
   if (special_req != NULL)
      delete_special_holder(special_req);
   if (when_awarded != NULL)
      delete_special_holder(when_awarded);
   delete lvl_flags;
}


/***********************************************************************
 ** set_chain_name - sets the name of this level chain
 **
 ** Parameters: new_val - the new value to set to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Level::set_chain_name(char *new_name)
{
   chain_name = new_name;
}


/***********************************************************************
 ** get_chain_name - gets the name of this level chain
 **
 ** Parameters: None
 **
 ***********************************************************************/

char *Level::get_chain_name(void)
{
   return chain_name.str_show();
}


/***********************************************************************
 ** set_award_string - sets the string that appears when the player
 **                    is awarded this level
 **
 ** Parameters: new_str - the new string to set this to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Level::set_award_string(char *new_str)
{
   award_string = new_str;
}


/***********************************************************************
 ** get_award_string - gets the string displayed when the level is awarded
 **
 ** Parameters: None
 **
 ***********************************************************************/

char *Level::get_award_string(void)
{
   return award_string.str_show();
}


/***********************************************************************
 ** set_level_str - set the string that says what level you are
 **
 ** Parameters: new_str - the new string to set this to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Level::set_level_str(char *new_str)
{
   level_str = new_str;
}


/***********************************************************************
 ** get_level_str - gets the string that says what level you are
 **
 ** Parameters: None
 **
 ***********************************************************************/

char *Level::get_level_str(void)
{
   return level_str.str_show();
}


/***********************************************************************
 ** set_special_req_str - assign the special name for the special
 **                       requirements
 **
 ** Parameters: new_str - the new string to set this to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Level::set_special_req_str(char *new_str)
{
   special_req_str = new_str;
}


/***********************************************************************
 ** get_special_req_str - gets the name of the special to assign to the
 **                       special requirements
 **
 ** Parameters: None
 **
 ***********************************************************************/

char *Level::get_special_req_str(void)
{
   return special_req_str.str_show();
}


/***********************************************************************
 ** set_req_abilities - assign the required abilities string
 **
 ** Parameters: new_str - the new string to set this to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Level::set_req_abilities(char *new_str)
{
   req_abilities = new_str;
}


/***********************************************************************
 ** get_req_abilities - gets the required abilities string
 **
 ** Parameters: None
 **
 ***********************************************************************/

char *Level::get_req_abilities(void)
{
   return req_abilities.str_show();
}


/***********************************************************************
 ** set_when_awarded_str - assign the special name for the special run when
 **                        the level is awarded (used to assign or set attrib)
 **
 ** Parameters: new_str - the new string to set this to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Level::set_when_awarded_str(char *new_str)
{
   when_awarded_str = new_str;
}


/***********************************************************************
 ** get_when_awarded_str - gets the name of the special that is executed
 **                        when the level is awarded
 **
 ** Parameters: None
 **
 ***********************************************************************/

char *Level::get_when_awarded_str(void)
{
   return when_awarded_str.str_show();
}


/***********************************************************************
 ** set_when_awarded - assign the special that is run when the level
 **                    is awarded
 **
 ** Parameters: the_special - the new special to set this to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Level::set_when_awarded(Specials *the_special)
{
   when_awarded = new_special_holder();
   when_awarded->the_special = the_special;
   when_awarded->environment.exec_vars = NULL;
   when_awarded->next_special = NULL;
}


/***********************************************************************
 ** set_special_req - assign the special that allows special levelling
 **                   requirements to be laid down
 **
 ** Parameters: the_special - the new special to set this to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Level::set_special_req(Specials *the_special)
{
   special_req = new_special_holder();
   special_req->the_special = the_special;
   special_req->environment.exec_vars = NULL;
   special_req->next_special = NULL;
}

/***********************************************************************
 ** get_when_awarded - gets the special that is executed when the
 **                    level is awarded
 **
 ***********************************************************************/

special_holder *Level::get_when_awarded(void)
{
   return when_awarded;
}


/***********************************************************************
 ** get_special_req - gets the special that is executed to determine any
 **                   special requirements for levelling
 **
 ***********************************************************************/

special_holder *Level::get_special_req(void)
{
   return special_req;
}


/***********************************************************************
 ** set_lvl_num - sets the level number of this level in this chain
 **
 ** Parameters: new_val - the new value to set to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Level::set_lvl_num(int new_val)
{
   if (new_val < 0)
      return;

   lvl_num = new_val;
}


/***********************************************************************
 ** get_lvl_num - gets the level number of this level in this chain
 **
 ** Parameters: None
 **
 ***********************************************************************/

int Level::get_lvl_num(void)
{
   return lvl_num;
}


/***********************************************************************
 ** set_min_str - sets the minimum strength required to level
 **
 ** Parameters: new_val - the min strength required to level
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Level::set_min_str(int new_val)
{
   if (new_val < 0)
      return;

   min_str = new_val;
}


/***********************************************************************
 ** get_min_str - gets the minimum strength rqquired to level
 **
 ** Parameters: None
 **
 ***********************************************************************/

int Level::get_min_str(void)
{
   return min_str;
}


/***********************************************************************
 ** set_min_dex - sets the minimum dexterity required to level
 **
 ** Parameters: new_val - the min dex required to level
 **
 **
 ***********************************************************************/

void Level::set_min_dex(int new_val)
{
   if (new_val < 0)
      return;

   min_dex = new_val;
}


/***********************************************************************
 ** get_min_dex - gets the minimum dexterity required to level
 **
 ** Parameters: None
 **
 ***********************************************************************/

int Level::get_min_dex(void)
{
   return min_dex;
}


/***********************************************************************
 ** set_min_intel - sets the minimum intelligence required to level
 **
 ** Parameters: new_val - the min intel required to level
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Level::set_min_intel(int new_val)
{
   if (new_val < 0)
      return;

   min_intel = new_val;
}


/***********************************************************************
 ** get_min_intel - gets the minimum intelligence required to level
 **
 ** Parameters: None
 **
 ***********************************************************************/

int Level::get_min_intel(void)
{
   return min_intel;
}


/***********************************************************************
 ** set_min_con - sets the minimum constitution required to level
 **
 ** Parameters: new_val - the min con required to level
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Level::set_min_con(int new_val)
{
   if (new_val < 0)
      return;

   min_con = new_val;
}


/***********************************************************************
 ** get_min_con - gets the minimum constitution required to level
 **
 ** Parameters: None
 **
 ***********************************************************************/

int Level::get_min_con(void)
{
   return min_con;
}


/***********************************************************************
 ** set_min_wisdom - sets the minimum wisdom required to level
 **
 ** Parameters: new_val - the min wisdom required to level
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Level::set_min_wisdom(int new_val)
{
   if (new_val < 0)
      return;

   min_wisdom = new_val;
}


/***********************************************************************
 ** get_min_wisdom - gets the minimum wisdom required to level
 **
 ** Parameters: None
 **
 ***********************************************************************/

int Level::get_min_wisdom(void)
{
   return min_wisdom;
}


/***********************************************************************
 ** set_min_cha - sets the minimum charisma required to level
 **
 ** Parameters: new_val - the min charisma required to level
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Level::set_min_cha(int new_val)
{
   if (new_val < 0)
      return;

   min_cha = new_val;
}


/***********************************************************************
 ** get_min_cha - gets the minimum charisma required to level
 **
 ** Parameters: None
 **
 ***********************************************************************/

int Level::get_min_cha(void)
{
   return min_cha;
}


/***********************************************************************
 ** set_min_exp - sets the minimum experience required to level
 **
 ** Parameters: new_val - the min experience required to level
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Level::set_min_exp(int new_val)
{
   if (new_val < 0)
      return;

   min_exp = new_val;
}


/***********************************************************************
 ** get_min_exp - gets the minimum experience required to level
 **
 ** Parameters: None
 **
 ***********************************************************************/

int Level::get_min_exp(void)
{
   return min_exp;
}


/***********************************************************************
 ** load_level - loads a level from a file into memory
 **
 ** Parameters: the_file - where we are getting the action from
 **
 ** Returns: pointer to the next level
 **
 ***********************************************************************/

int Level::load_level(FILE *the_file, ErrLog *error_log, int is_builder)
{
   token_record *the_token;
   char         *tmp_charholder;
   Strings      holder;

   if (is_builder)
   {
       the_token = get_token(the_file,'\0');
 
       if (the_token->token_type != T_NUMERICAL)
       {
           error_log->log_err("Invalid format in levels file", "load_level");
           return -1;
       }
       set_modified(atoi(the_token->the_string));
   }

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format for attribute chain_name in level %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_level");
      return -1;
   }

   /* get the chain name of this location */
   the_token = get_token(the_file, '^');
   set_chain_name(the_token->the_string);   

   /* get the award string */
   tmp_charholder = read_desc_type(the_file, error_log, NULL); 
   set_award_string(tmp_charholder);
   delete tmp_charholder;

   /* get the level string */
   tmp_charholder = read_desc_type(the_file, error_log, NULL); 
   set_level_str(tmp_charholder);
   delete tmp_charholder;

   /* Set lvl_num value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute lvl_num for level %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_level");
      return -1;
   }
   set_lvl_num(atoi(the_token->the_string));


   /* Set min_str value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute min_str for level %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_level");
      return -1;
   }
   set_min_str(atoi(the_token->the_string));

   /* Set min_dex value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute min_dex for level %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_level");
      return -1;
   }
   set_min_dex(atoi(the_token->the_string));

   /* Set min_intel value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute min_intel for level %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_level");
      return -1;
   }
   set_min_intel(atoi(the_token->the_string));

   /* Set min_exp value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute min_exp for level %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_level");
      return -1;
   }
   set_min_exp(atoi(the_token->the_string));

   /* Set min_con value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute min_con for level %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_level");
      return -1;
   }
   set_min_con(atoi(the_token->the_string));

   /* Set min_wisdom value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute min_wisdom for level %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_level");
      return -1;
   }
   set_min_wisdom(atoi(the_token->the_string));

   /* Set min_cha value */
   the_token = get_token(the_file, '\0');     
   if (the_token->token_type != T_NUMERICAL)
   {
      holder.sprintf("Invalid format for attribute min_cha for level %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_level");
      return -1;
   }
   set_min_cha(atoi(the_token->the_string));

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format for attribute req_abilities in level %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_level");
      return -1;
   }

   /* get the req_abilities name of this location */
   the_token = get_token(the_file, '^');
   set_req_abilities(the_token->the_string);   

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf(
         "Invalid format for attribute special_req_str in level %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_level");
      return -1;
   }

   /* get the special_req_str name of this location */
   the_token = get_token(the_file, '^');
   set_special_req_str(the_token->the_string);   


   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf(
         "Invalid format for attribute when_awarded_str in level %s", 
                                                               get_name());
      error_log->log_err(holder.str_show(), "load_level");
      return -1;
   }

   /* get the when_awarded_str name of this location */
   the_token = get_token(the_file, '^');
   set_when_awarded_str(the_token->the_string);   

   /* read in the level flags */
   if (lvl_flags->read_flags(the_file, error_log) <= 0)
   {
      holder.sprintf("Error reading lvlflags for level '%s'", get_name());
      error_log->log_err(holder.str_show(), "load_level");

      return -1;
   }   

   return 1;
}


/***********************************************************************
 ** describe - describes the level to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Level::describe(Builder *the_builder)
{
   the_builder->send_bldr("\n&+GLevel: \t\t&+M%s&*\n", get_name());
   the_builder->send_bldr("&+GChainName: \t&+w%s&*\n", get_chain_name());
   the_builder->send_bldr("&+GAwardStr: \n&+w%s&*\n", get_award_string());
   the_builder->send_bldr("&+GLevelStr: \n&+w%s&*\n", get_level_str());
   the_builder->send_bldr("&+GLvlNum: \t&+Y%d\n", get_lvl_num());
   the_builder->send_bldr("&+GMinStr: \t&+W%d\n", get_min_str());
   the_builder->send_bldr("&+GMinDex: \t&+W%d\n", get_min_dex());
   the_builder->send_bldr("&+GMinIntel: \t&+W%d\n", get_min_intel());
   the_builder->send_bldr("&+GMinExp: \t&+W%d\n", get_min_exp());
   the_builder->send_bldr("&+GMinCon: \t&+W%d\n", get_min_con());
   the_builder->send_bldr("&+GMinWisdom: \t&+W%d\n", get_min_wisdom());
   the_builder->send_bldr("&+GMinCha: \t&+W%d\n", get_min_cha());
   the_builder->send_bldr("&+GReqAbilities: \t&+w%s\n", get_req_abilities());
   the_builder->send_bldr("&+GSpecialReq: \t&+w%s\n", get_special_req_str());
   the_builder->send_bldr("&+GWhenAwarded: \t&+w%s\n", get_when_awarded_str());
   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the level to a player
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Level::describe(Player *the_player)
{
   the_player->send_plr("\n&+GLevel: \t\t&+M%s&*\n", get_name());
   the_player->send_plr("&+GChainName: \t&+w%s&*\n", get_chain_name());
   the_player->send_plr("&+GAwardStr: \n&+w%s&*\n", get_award_string());
   the_player->send_plr("&+GLevelStr: \n&+w%s&*\n", get_level_str());
   the_player->send_plr("&+GLvlNum: \t&+Y%d\n", get_lvl_num());
   the_player->send_plr("&+GMinStr: \t&+W%d\n", get_min_str());
   the_player->send_plr("&+GMinDex: \t&+W%d\n", get_min_dex());
   the_player->send_plr("&+GMinIntel: \t&+W%d\n", get_min_intel());
   the_player->send_plr("&+GMinExp: \t&+W%d\n", get_min_exp());
   the_player->send_plr("&+GMinCon: \t&+W%d\n", get_min_con());
   the_player->send_plr("&+GMinWisdom: \t&+W%d\n", get_min_wisdom());
   the_player->send_plr("&+GMinCha: \t&+W%d\n", get_min_cha());
   the_player->send_plr("&+GReqAbilities: \t&+w%s\n", get_req_abilities());
   the_player->send_plr("&+GSpecialReq: \t&+w%s\n", get_special_req_str());
   the_player->send_plr("&+GWhenAwarded: \t&+w%s\n", get_when_awarded_str());
   the_player->send_plr("\n");
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **             the_parsed - the parsed structure for this
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Level::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
        send_bldr("You can set the following attributes on a level.\n"
               "   ChainName, AwardStr, LvlNum, MinStr, MinDex, \n"
               "   MinIntel, MinExp, MinCon, MinWisdom, MinCha, \n"
	       "   ReqAbilities, SpecialReq, WhenAwarded,  and lvlflags\n");
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "awardstr",
                               strlen(the_parsed->get_target1())))
   {
      if (the_builder->get_long_input(&award_string) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "levelstr",
                               strlen(the_parsed->get_target1())))
   {
      if (the_builder->get_long_input(&level_str) < 0)
      {
         the_builder->send_bldr("Error reading in input, failed!\n");
         return -1;
      }
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "chainname",
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->
               send_bldr("You need to specify a chainname to set to.\n");
         return -1;
      }
      holder.assign_word(the_parsed->get_speech(), 1);
      set_chain_name(holder.str_show());

      the_builder->send_bldr("ChainName on %s set to: %s\n", get_name(), 
                                                     get_chain_name());
      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "reqabilities",
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;
      
      if (the_parsed->get_speech() != NULL)
	  {
	      holder.assign_word(the_parsed->get_speech(), 1);
	      set_req_abilities(holder.str_show());
	  }
	  else
		  req_abilities.truncate(0);


      the_builder->send_bldr("ReqAbilities on %s set to: %s\n", get_name(), 
                                                     get_req_abilities());
      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "specialreq",
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;
      
      if (the_parsed->get_speech() != NULL)
	  {
	      holder.assign_word(the_parsed->get_speech(), 1);
	      set_special_req_str(holder.str_show());
	  }
	  else
		  special_req_str.truncate(0);

      the_builder->send_bldr("SpecialReq on %s set to: %s\n", get_name(), 
                                                     get_special_req_str());
      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "whenawarded",
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;
      
      if (the_parsed->get_speech() != NULL)
	  {
	      holder.assign_word(the_parsed->get_speech(), 1);
	      set_when_awarded_str(holder.str_show());
	  }
	  else
		  when_awarded_str.truncate(0);

      the_builder->send_bldr("WhenAwarded on %s set to: %s\n", get_name(), 
                                                     get_when_awarded_str());
      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "lvlnum",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as LvlNum.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if (value <= 0)
      {
         the_builder->send_bldr("Value must be greater than 0\n");
         return -1;
      }

      set_lvl_num(value);
      the_builder->send_bldr("LvlNum set to %d on level object %s.\n",
                                          get_lvl_num(), get_name());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "minstr",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as MinStr.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if (value < 0)
      {
         the_builder->send_bldr("Value must be greater than or equal to 0\n");
         return -1;
      }

      set_min_str(value);
      the_builder->send_bldr("MinStr set to %d on level object %s.\n",
                                          get_min_str(), get_name());
      return 1;
   }


   if (!STRNCASECMP(the_parsed->get_target1(), "mindex",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as MinDex.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if (value < 0)
      {
         the_builder->send_bldr("Value must be greater than or equal to 0\n");
         return -1;
      }

      set_min_dex(value);
      the_builder->send_bldr("MinDex set to %d on level object %s.\n",
                                          get_min_dex(), get_name());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "minintel",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as MinIntel.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if (value < 0)
      {
         the_builder->send_bldr("Value must be greater than or equal to 0\n");
         return -1;
      }

      set_min_intel(value);
      the_builder->send_bldr("MinIntel set to %d on level object %s.\n",
                                          get_min_str(), get_name());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "minexp",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as MinExp.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if (value < 0)
      {
         the_builder->send_bldr("Value must be greater than or equal to 0\n");
         return -1;
      }

      set_min_exp(value);
      the_builder->send_bldr("MinExp set to %d on level object %s.\n",
                                          get_min_exp(), get_name());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "mincon",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as MinCon.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if (value < 0)
      {
         the_builder->send_bldr("Value must be greater than or equal to 0\n");
         return -1;
      }

      set_min_con(value);
      the_builder->send_bldr("MinCon set to %d on level object %s.\n",
                                          get_min_con(), get_name());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "minwisdom",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as MinWisdom.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if (value < 0)
      {
         the_builder->send_bldr("Value must be greater than or equal to 0\n");
         return -1;
      }

      set_min_wisdom(value);
      the_builder->send_bldr("MinWisdom set to %d on level object %s.\n",
                                          get_min_wisdom(), get_name());
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "mincha",
                               strlen(the_parsed->get_target1())))
   {
      int value;
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->send_bldr("You need to specify a number as well.\n");
         return -1;
      }
 
      if (!isdigit(*(the_parsed->get_speech()))) 
      {
         the_builder->send_bldr("You need to specify a number as ChaExp.\n");
         return -1;
      }

      value = atoi(the_parsed->get_speech());

      if (value < 0)
      {
         the_builder->send_bldr("Value must be greater than or equal to 0\n");
         return -1;
      }

      set_min_cha(value);
      the_builder->send_bldr("MinCha set to %d on level object %s.\n",
                                          get_min_cha(), get_name());
      return 1;
   }

   the_builder->send_bldr("The attribute '%s' is not a level attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** write_object - writes the level to a specified file in specified
 **                format
 **
 ** Parameters: the_file - the file to write to
 **             build_format - shall we use builder format or not
 **
 ** Returns:  Nothing
 **
 ***********************************************************************/

void Level::write_object(FILE *the_file, int build_format)
{
   fprintf(the_file, "\nlevel %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());

   fprintf(the_file, "^%s^\n", (get_chain_name() == NULL) ? "" : 
                                                        get_chain_name());
   fprintf(the_file, "^%s^\n", (get_award_string() == NULL) ? "" : 
                                                        get_award_string());
   fprintf(the_file, "^%s^\n", (get_level_str() == NULL) ? "" : 
                                                        get_level_str());
   fprintf(the_file, "%d\n", get_lvl_num());
   fprintf(the_file, "%d\n", get_min_str());
   fprintf(the_file, "%d\n", get_min_dex());
   fprintf(the_file, "%d\n", get_min_intel());
   fprintf(the_file, "%d\n", get_min_exp());
   fprintf(the_file, "%d\n", get_min_con());
   fprintf(the_file, "%d\n", get_min_wisdom());
   fprintf(the_file, "%d\n", get_min_cha());
   fprintf(the_file, "^%s^\n", (get_req_abilities() == NULL) ? "" : 
                                                        get_req_abilities());
   fprintf(the_file, "^%s^\n", (get_special_req_str() == NULL) ? "" : 
                                                    get_special_req_str());
   fprintf(the_file, "^%s^\n", (get_when_awarded_str() == NULL) ? "" : 
                                                    get_when_awarded_str());
   lvl_flags->write_flag(the_file);

}


/***********************************************************************
 ** is_modified - has this quest been modified?
 **
 ** Parameters: None
 **
 ** Returns:  1 for yes, 0 for no
 **
 ***********************************************************************/
   
int Level::is_modified(void)
{
   return modified;
}



/***********************************************************************
 ** set_modified - shall we set this?
 **
 ** Parameters: the_num - the number to set it to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/
   
void Level::set_modified(int the_num)
{
   modified = the_num;
}


/***********************************************************************
 ** get_lvl_flags - gets the level flags for this level
 **
 **
 ***********************************************************************/
   
Flags *Level::get_lvl_flags(void)
{
   return lvl_flags;
}


/***********************************************************************
 ** copy_object - copies the level to a level of a different name
 **
 ** Parameters: copy_obj - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Level::copy_object(Entity *copy_obj)
{
   Level *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_LEVEL)
      return 0;

   copy_from = (Level *) copy_obj;


   /******* set the action attributes *****/
   set_chain_name(copy_from->get_chain_name());
   set_award_string(copy_from->get_award_string());
   set_level_str(copy_from->get_level_str());
   set_lvl_num(copy_from->get_lvl_num());
   set_min_str(copy_from->get_min_str());
   set_min_dex(copy_from->get_min_dex());
   set_min_intel(copy_from->get_min_intel());
   set_min_exp(copy_from->get_min_exp());
   set_min_con(copy_from->get_min_con());
   set_min_wisdom(copy_from->get_min_wisdom());
   set_min_cha(copy_from->get_min_cha());
   set_req_abilities(copy_from->get_req_abilities());
   set_special_req_str(copy_from->get_special_req_str());
   set_when_awarded_str(copy_from->get_when_awarded_str());
   lvl_flags->copy_flags(copy_from->get_lvl_flags());

   return 1;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Level::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Level::get_mem_size_dynamic()
{
   int  size = 0;
   
   size += chain_name.get_mem_size_dynamic();
   size += award_string.get_mem_size_dynamic();
   size += level_str.get_mem_size_dynamic();
   size += req_abilities.get_mem_size_dynamic();
   size += special_req_str.get_mem_size_dynamic();
   size += when_awarded_str.get_mem_size_dynamic();

   size += lvl_flags->get_mem_size();
   size += sizeof(*special_req);

   size += get_mem_size_entity();
   size += sizeof(*when_awarded);

   return size;
}

#endif
