/**********************************************************************
 ** Inventory class: subclass of the LinkedList class, has a few other
 **                  methods useful for finding objects in an inventory
 **
 ** Last reviewed:
 **
 **
 ** Copyright (C) 2001 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudobject.h"
#include "linkedlist.h"
#include "timespan.h"
#include "merger.h"
#include "player.h"
#include "newfuncts.h"
#include "global.h"


/*********************************************************************
 * CONSTRUCTOR - initializes the inventory
 *********************************************************************/
Inventory::Inventory()
{
}

/*********************************************************************
 * DESTRUCTOR - Cleans up the linked list and clears all allocated
 *              memory.
 *********************************************************************/
Inventory::~Inventory()
{
}


/*********************************************************************
 * find - finds an entry in the linked list by searching for its name
 *
 * Parameters:
 * INPUT : a name
 * OUTPUT: a pointer to the first mudobject that satisfies to that
 *         criteria or NULL if not found.
 *********************************************************************/
MudObject *Inventory::find(char *thename, int *the_num)
{  MudObject *tmpObject;
   Strings   the_keywords;

   tmpObject = get_first();
   while (tmpObject != NULL)
   {
      the_keywords = tmpObject->get_keywords();
      if (the_keywords.find_in_str(thename))
      {
         if (*the_num <= 1)
            return tmpObject;
         else
            *the_num = *the_num - 1;
      }
      tmpObject = get_next(); 
   }

   return NULL;
}


/*********************************************************************
 * find_by_objname - like find but uses the object's name instead of
 *                   the object's title
 *
 * Parameters:
 * INPUT : a name, the_area
 * OUTPUT: a pointer to the first mudobject that satisfies to that
 *         criteria or NULL if not found.
 *********************************************************************/

MudObject *Inventory::find_by_objname(char *the_name, char *the_area)
{  MudObject *tmpObject;

   tmpObject = get_first();
   while (tmpObject != NULL)
   {
        if (STRCASECMP(tmpObject->get_name(),the_name) == 0)
        { 
            if (!STRCASECMP(tmpObject->get_area(),the_area))
               return tmpObject;
        }
        tmpObject = get_next(); 
   }

   return NULL;
}


/*********************************************************************
 * find_altname
 *
 * Parameters:
 * INPUT : a name
 * OUTPUT: a pointer to the first mudobject that satisfies to that
 *         criteria or NULL if not found.
 *********************************************************************/
MudObject *Inventory::find_altname(char *the_name)
{  MudObject *tmpObject;
   Strings   the_altnames;

   tmpObject = get_first();

   while (tmpObject != NULL)
   {
      the_altnames = tmpObject->get_altname();
      if (the_altnames.find_in_str(the_name) > 0)
      {
          return tmpObject;
      }
      tmpObject = get_next(); 
   }

   return NULL;
}



/*********************************************************************
 * get_same_merger - gets a merger with the same parent specified
 *
 * Parameters:
 * INPUT : the_parent - the parent string we are looking for
 * OUTPUT: a pointer to the first mudobject that satisfies to that
 *         criteria or NULL if not found.
 *********************************************************************/
Merger *Inventory::get_same_merger(char *the_parent)
{  MudObject *tmpObject;
   Merger    *the_merger;

   tmpObject = get_first();
   while (tmpObject != NULL)
   {
      if (tmpObject->is_merger())
      { 
         the_merger = (Merger *) tmpObject;
         if ((the_merger->get_parent() != NULL) && 
             (STRCASECMP(the_merger->get_parent(), the_parent) == 0))
            return the_merger;
      }
      tmpObject = get_next(); 
   }
   return NULL;
}

/***********************************************************************
 ** display_list - displays a linked list in inventory format
 **
 ** Parameters: the_player - the player we send the data to
 **
 ** Returns: num objects displayed for success, -1 for failure
 **
 ***********************************************************************/

int Inventory::display_list(Player *the_player)
{
   MudObject  *the_obj;
   int        count = 0;
   int        line_len = 0;
   Strings    send_str;
   Strings    holder;

   reset_current();
   the_obj = get_next();

   while (the_obj != NULL)
   {  
      if (the_obj->is_an_item())
      {
         Flags *tmp_itemflags;

         tmp_itemflags = ((Item *) the_obj)->get_itemflags();

         if (the_obj->is_merger())
	    holder.sprintf("%s%s(%ld)%s", ((line_len == 0) ? "\n" : ", "), 
                            the_obj->get_title(), 
                            ((Merger *) the_obj)->get_number_of(), 
                 tmp_itemflags->get_flag(ITEMFLAG_LIT) ? " &+Y(lit)&*" : "");
         else 
	 {
            if (tmp_itemflags->get_flag(ITEMFLAG_WORN))
	    {
               the_obj = get_next();
	       count++;
               continue;
	    }

            holder.sprintf("%s%s%s", ((line_len == 0) ? "\n" : ", "), 
                                                    the_obj->get_title(),
               tmp_itemflags->get_flag(ITEMFLAG_LIT) ? " &+Y(lit)&*" : "");
	 }
      }
      else
         holder.sprintf("%s%s", ((line_len == 0) ? "\n" : ", "), 
                                                    the_obj->get_title());
      send_str.str_cat(holder.str_show());

      count++;
      the_obj = get_next();
      if ((line_len = send_str.str_len()) >= the_config.max_line_len)
      {
         the_player->send_plr(send_str.str_show());
         send_str.truncate(0);
         line_len = 0;
      }
   }
   if (send_str.str_show() != NULL)
      the_player->send_plr(send_str.str_show());
   the_player->send_plr("\n");

   return count;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Inventory::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Inventory::get_mem_size_dynamic()
{
   int  size = 0;

   size += get_mem_size_linkedlist();

   return size;
}





