;;;***************************************************************
;;;An ACL2 Library of Floating Point Arithmetic

;;;David M. Russinoff
;;;Advanced Micro Devices, Inc.
;;;February, 1998
;;;***************************************************************

(in-package "ACL2")

(include-book "trunc")

(defun away (x n)
  (* (sgn x) (cg (* (expt 2 (1- n)) (sig x))) (expt 2 (- (1+ (expo x)) n))))



(defthm away-minus
  (= (away (* -1 x) n) (* -1 (away x n))))

(defthm away-pos
  (implies (and (> x 0)
                (rationalp x)
                (integerp n))
           (> (away x n) 0))
  :rule-classes :linear
  :hints (("Goal" :in-theory (disable expo sig fl-weakly-monotonic)
           :use ((:instance sig-lower-bound)
                 (:instance pos* 
                            (x (cg (* (sig x) (expt 2 (1- n))))) 
                            (y (expt 2 (- (1+ (expo x)) n))))
                 (:instance sgn+1)
                 (:instance expo-monotone (x 1) (y (1- n)))
                 (:instance cg-def-linear (x (sig x)))))))

(defthm away-neg
    (implies (and (< x 0)
                  (rationalp x)
		  (integerp n))
	     (< (away x n) 0))
  :rule-classes :linear
  :hints (("Goal" :in-theory (disable expo sig)
		  :use ((:instance sig-lower-bound)
			(:instance pos* 
				   (x (cg (* (sig x) (expt 2 (1- n))))) 
				   (y (expt 2 (- (1+ (expo x)) n))))
			(:instance sgn-1)
			(:instance expo-monotone (x 1) (y (1- n)))
			(:instance cg-def-linear (x (sig x)))))))

(defthm away-0
  (equal (away 0 n) 0)
  :hints (("Goal" :in-theory (enable away))))

(defthm away-0-0
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (iff (= (away x n) 0)
		  (= x 0)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable away)
		  :use ((:instance away-pos)
			(:instance away-neg)
			(:instance away-0)))))

(defthm sgn-away
  (implies (and (rationalp x)
                (integerp n))
           (equal (sgn (away x n))
                  (sgn x)))
  :hints (("Goal" :in-theory (disable away)
           :use ((:instance away-pos)
                 (:instance away-neg)
                 (:instance away-0)))))


(defthm abs-away
  (implies (and (rationalp x)
                (integerp n))
           (equal (abs (away x n)) (* (cg (* (expt 2 (1- n)) (sig x))) (expt 2 (- (1+ (expo x)) n)))))
  :hints (("Goal" :in-theory (disable expo sig)
           :use ((:instance sig-lower-bound)
                 (:instance pos* 
                            (x (cg (* (sig x) (expt 2 (1- n))))) 
                            (y (expt 2 (- (1+ (expo x)) n))))
                 (:instance sgn-1)
                 (:instance sgn+1)
                 (:instance expo-monotone (x 1) (y (1- n)))
                 (:instance cg-def-linear (x (sig x)))))))

(in-theory (disable cg))


(defthm away-to-0-or-fewer-bits
  (implies (and (<= n 0)
                (rationalp x)
                (integerp n)
                )
           (equal (away x n)
                  (* (sgn x) (expt 2 (+ 1 (expo x) (- n))))))
  :hints (("Goal" :in-theory (set-difference-theories
                              (enable away)
                              '(expo sig))
           :use ((:instance cg-unique
                            (x (* 1/2 (SIG X) (EXPT 2 N)))
                            (n 1))
                 sig-upper-bound
                 sig-lower-bound
                 (:instance expt-weak-monotone
                            (n n)
                            (m 0))
                 (:instance expt-strong-monotone
                            (n 0)
                            (m n))))))



(defthm away-lower-bound
    (implies (and (rationalp x)
		  (integerp n))
	     (>= (abs (away x n)) (abs x)))
  :rule-classes :linear
  :hints (("Goal" :in-theory (disable sig away)
		  :use (away-to-0-or-fewer-bits
                        expo-upper-bound
                        away-neg
                        (:instance away-neg (x (* (SIG X) (EXPT 2 (EXPO X)))))
                        (:instance cg-def-linear (x (* (expt 2 (1- n)) (sig x))))
			(:instance sig-lower-bound)
			(:instance fp-abs)
			(:instance expo+ (m (1- n)) (n (- (1+ (expo x)) n)))))))



#|
;subsumed by type-prescription?
(defthm rationalp-away
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (rationalp (away x n))))
|#


(defthm away-lower-pos
    (implies (and (>= x 0)
                  (rationalp x)
		  (integerp n))
	     (>= (away x n) x))
  :rule-classes :linear
  :hints (("Goal" :in-theory (disable abs-away away)
		  :use ((:instance away-lower-bound)
			(:instance away-pos)
			(:instance away-0-0)))))

;elim?
(defthm expo-away-lower-bound
  (implies (and (rationalp x)
                (integerp n)
                (> n 0))
           (>= (expo (away x n)) (expo x)))
  :rule-classes :linear
  :hints (("Goal" :in-theory (disable expo sig away)
           :use ((:instance away-lower-bound)
                 (:instance away-0-0)
                 (:instance expo-monotone (y (away x n)))))))

(defthm away-upper-1
  (implies (and (rationalp x)
                (integerp n)
                (> n 0))
           (< (abs (away x n)) (+ (abs x) (expt 2 (- (1+ (expo x)) n)))))
  :rule-classes :linear
  :hints (("Goal" :in-theory (disable abs expo sig away a15)
           :use ((:instance trunc-lower-1-1)
                 (:instance trunc-lower-1-3
                            (u (* (sig x) (expt 2 (1- n))))
                            (v (fl (* (sig x) (expt 2 (1- n)))))
                            (r (expt 2 (- (1+ (expo x)) n))))
                 (:instance cg-def-linear (x (* (expt 2 (1- n)) (sig x))))))))

(defthm away-upper-2
  (implies (and (rationalp x)
                (not (= x 0))
                (integerp n)
                (> n 0))
           (< (abs (away x n)) (* (abs x) (+ 1 (expt 2 (- 1 n))))))
  :rule-classes :linear
  :hints (("Goal" :in-theory (disable abs expo sig away a15)
           :use ((:instance away-upper-1)
                 (:instance trunc-lower-2-1)))))

(defthm away-upper-pos
    (implies (and (> x 0)
                  (rationalp x)
		  (integerp n)
		  (> n 0))
	     (< (away x n) (* x (+ 1 (expt 2 (- 1 n))))))
  :rule-classes :linear
  :hints (("Goal" :in-theory (disable abs-pos abs-away expo sig away a15)
		  :use ((:instance away-upper-2)
			(:instance away-pos)))))

(defthm away-upper-3
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (<= (abs (away x n)) (* (abs x) (+ 1 (expt 2 (- 1 n))))))
  :rule-classes :linear
  :hints (("Goal" :in-theory (disable abs expo sig away a15)
		  :use ((:instance away-upper-1)
			(:instance away-0-0)
			(:instance trunc-lower-2-1)))))

(defthm away-diff
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (< (abs (- (away x n) x)) (expt 2 (- (1+ (expo x)) n))))
  :rule-classes :linear
  :hints (("Goal" :in-theory (disable away expo abs-away)
		  :use ((:instance trunc-diff-1 (y (away x n)))
			(:instance away-neg)
			(:instance away-pos)
			(:instance away-0-0)
			(:instance away-lower-bound)
			(:instance away-upper-1)))))

(defthm away-diff-pos
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (> n 0))
	     (< (- (away x n) x) (expt 2 (- (1+ (expo x)) n))))
  :rule-classes :linear
  :hints (("Goal" :in-theory (disable away expo abs-away)
		  :use ((:instance away-diff)
			(:instance away-pos)
			(:instance away-lower-bound)))))

(defthm away-diff-expo-1
    (implies (and (rationalp x)
		  (not (= x (away x n)))
		  (integerp n)
		  (> n 0))
	     (<= (expo (- (away x n) x)) (- (expo x) n)))
  :rule-classes :linear
  :hints (("Goal" :in-theory (disable away abs expo abs-away)
		  :use ((:instance away-diff)
			(:instance expo-lower-bound (x (- (away x n) x)))
			(:instance expt-strong-monotone 
				   (n (expo (- (away x n) x)))
				   (m (- (1+ (expo x)) n)))))))

(defthm away-rewrite
  (implies (and (rationalp x)
                (integerp n)
                (> n 0))
           (equal (away x n)
                  (* (sgn x) 
                     (cg (* (expt 2 (- (1- n) (expo x))) (abs x))) 
                     (expt 2 (- (1+ (expo x)) n))))))

(in-theory (disable away))

(local
 (defthm away-exactp-1
    (implies (and (rationalp x)
		  (integerp n))
	     (= x (* (sgn x) (* (expt 2 (- (1- n) (expo x))) (abs x)) (expt 2 (- (1+ (expo x)) n)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo)
		  :use ((:instance expo+ (n (- (1- n) (expo x))) (m (- (1+ (expo x)) n))))))))

(local
 (defthm away-exactp-2
    (implies (and (rationalp x)
		  (rationalp y)
		  (rationalp z)
		  (not (= x 0))
		  (not (= z 0)))
	     (iff (= (* x y z) (* x (cg y) z))
		  (integerp y)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable cg-int cg-int-2 fl)
		  :use ((:instance cg-int-2 (x y))
			(:instance *cancell (x (cg y)) (z (* x z))))))))

(local
 (defthm away-exactp-3
    (implies (integerp x) (integerp (- x)))
  :rule-classes ()))

(local
 (defthm away-exactp-4
    (implies (rationalp x)
	     (equal (- (- x)) x))))

(local
 (defthm away-exactp-5
    (implies (rationalp x)
	     (iff (integerp x) (integerp (- x))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable a2)
		  :use ((:instance away-exactp-3)
			(:instance away-exactp-3 (x (- x))))))))

(local (in-theory (enable exactp2)))

(local
 (defthm away-exactp-6
    (implies (and (rationalp x)
		  (integerp n))
	     (iff (exactp x n)
		  (integerp (* (abs x) (expt 2 (- (1- n) (expo x)))))))
  :rule-classes ()
  :hints (("Goal" :use ((:instance away-exactp-5 (x (* x (expt 2 (- (1- n) (expo x)))))))))))


(defthm away-exactp-a
    (implies (and (rationalp x)
		  (integerp n) 
		  (> n 0))
	     (iff (= x (away x n))
		  (exactp x n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo)
		  :use ((:instance away-exactp-1)
			(:instance away-exactp-6)
			(:instance away-exactp-2
				   (x (sgn x))
				   (y (* (expt 2 (- (1- n) (expo x))) (abs x)))
				   (z (expt 2 (- (1+ (expo x)) n))))))))

(defthm away-diff-expo
    (implies (and (rationalp x)
		  (not (exactp x n))
		  (integerp n)
		  (> n 0))
	     (<= (expo (- (away x n) x)) (- (expo x) n)))
  :rule-classes :linear
  :hints (("Goal" :in-theory (disable away abs exactp2 expo abs-away)
		  :use ((:instance away-diff-expo-1)
			(:instance away-exactp-a)))))
(local
 (defthm away-exactp-b-1    
    (implies (and (rationalp x)
		  (rationalp y)
		  (integerp n)
		  (> n 0))
	     (integerp (* (* (sgn x) (cg y) (expt 2 (- (1- n) (expo x)))) (expt 2 (- (1+ (expo x)) n)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo)
		  :use ((:instance integerp-x-y 
				   (x (sgn x))
				   (y (cg (* (expt 2 (- (1- n) (expo x))) (abs x)))))
			(:instance expo+ (n (- (1- n) (expo x))) (m (- (1+ (expo x)) n))))))))

(local
 (defthm away-exactp-b-2
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (integerp (* (away x n) (expt 2 (- (1- n) (expo x))))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo sgn)
		  :use ((:instance away-exactp-b-1 (y (* (expt 2 (- (1- n) (expo x))) (abs x)))))))))

(local
 (defthm away-exactp-b-3
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0))
	     (<= (* (expt 2 (1- n)) (sig x)) (expt 2 n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo sig abs)
		  :use ((:instance sig-upper-bound)
			(:instance expo+ (n (1- n)) (m 1)))))))

(local
 (defthm away-exactp-b-4
    (implies (and (rationalp c)
		  (integerp n)
		  (integerp m)
		  (<= c (expt 2 n)))
	     (<= (* c (expt 2 (- m n))) (expt 2 m)))
  :rule-classes ()))

(local
 (defthm away-exactp-b-5
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0))
	     (<= (abs (away x n)) (expt 2 (1+ (expo x)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable a15 expo sig abs away-rewrite)
		  :use ((:instance away-exactp-b-3)
			(:instance away-exactp-b-4 (c (cg (* (sig x) (expt 2 (1- n))))) (m (1+ (expo x))))
			(:instance n>=cg-linear (n (expt 2 n)) (x (* (expt 2 (1- n)) (sig x)))))))))

(local
 (defthm away-exactp-b-6
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0)
		  (not (= (abs (away x n)) (expt 2 (1+ (expo x))))))
	     (<= (expo (away x n)) (expo x)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo abs away-rewrite)
		  :use ((:instance away-exactp-b-5)
			(:instance expo-lower-bound (x (away x n)))
			(:instance away-0-0)
			(:instance expt-strong-monotone (n (expo (away x n))) (m (1+ (expo x)))))))))

(local
 (defthm away-exactp-b-7
   (implies (and (rationalp x)
                 (not (= x 0))
                 (integerp n)
                 (> n 0)
                 (not (= (abs (away x n)) (expt 2 (1+ (expo x))))))
            (exactp (away x n) n))
   :rule-classes ()
   :hints (("Goal" :in-theory (disable expo abs away-rewrite)
            :use ((:instance away-exactp-b-2)
                  (:instance away-exactp-b-6)
                  (:instance away-0-0)
                  (:instance exactp->=-expo (x (away x n)) (e (expo x))))))))

(local
 (defthm away-exactp-b-8
    (implies (rationalp x)
	     (= (expo x) (expo (- x))))
  :rule-classes ()))

(local
 (defthm away-exactp-b-9
    (implies (and (rationalp x)
		  (integerp n)
		  (integerp m)
		  (> m 0)
		  (= (abs x) (expt 2 n)))
	     (exactp x m))
  :rule-classes ()
  :hints (("Goal" :use ((:instance away-exactp-b-8)
			(:instance exactp-2**n)
			(:instance trunc-exactp-5 (x (* x (expt 2 (- (1- m) (expo x)))))))))))

(local
 (defthm away-exactp-b-10    
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0))
	     (exactp (away x n) n))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo abs abs-away away-rewrite)
		  :use ((:instance away-exactp-b-7)
			(:instance away-exactp-b-9 (x (away x n)) (m n) (n (1+ (expo x)))))))))

(defthm away-exactp-b
    (implies (and (rationalp x)
		  (integerp n)
		  (> n 0))
	     (exactp (away x n) n))
  :hints (("Goal" :in-theory (disable expo away-rewrite)
		  :use ((:instance away-exactp-b-10)
			(:instance away-0-0)))))

(local
 (defthm away-exactp-c-1
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (rationalp a)
		  (exactp a n)
		  (>= a x)
		  (< a (away x n)))
	     (>= (away x n) (+ x (expt 2 (- (1+ (expo x)) n)))))
  :hints (("Goal" :in-theory (disable expo exactp2 abs-away away-rewrite)
		  :use ((:instance away-exactp-b)
			(:instance fp+1 (x a) (y (away x n)))
			(:instance expo-monotone (y a))
			(:instance expt-monotone (n (- (1+ (expo x)) n)) (m (- (1+ (expo a)) n))))))))

(defthm away-exactp-c
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (rationalp a)
		  (exactp a n)
		  (>= a x))
	     (>= a (away x n)))
  :hints (("Goal" :in-theory (disable expo exactp2 abs-away away-rewrite)
		  :use ((:instance away-exactp-c-1)
			(:instance away-upper-1)
			(:instance away-pos)))))

(defthm away-monotone-old
    (implies (and (rationalp x)
		  (rationalp y)
		  (integerp n)
		  (> x 0)
		  (> n 0)
		  (<= x y))
	     (<= (away x n) (away y n)))
  :rule-classes :linear
  :hints (("Goal" :in-theory (disable expo exactp2 abs-away away-rewrite)
		  :use ((:instance away-exactp-b (x y))
			(:instance away-lower-pos (x y))
			(:instance away-exactp-c (a (away y n)))))))

(defthm away-monotone
  (implies (and (rationalp x)
                (rationalp y)
                (integerp n)
                (<= x y))
           (<= (away x n) (away y n)))
  :hints (("Goal" :in-theory (disable away away-rewrite away-upper-pos
                                      away-pos away-neg
                                      away-to-0-or-fewer-bits expo-monotone
                                      away-monotone-old)
           :cases ((> n 0)))
          ("subgoal 2" 
           :use (away-pos
                 away-neg
                 (:instance away-pos (x y))
                 (:instance away-neg (x y))
                 (:instance expt-weak-monotone
                            (n (+ 1 (EXPO X) (* -1 N)))
                            (m (+ 1 (EXPO y) (* -1 N))))
                 away-to-0-or-fewer-bits
                 expo-monotone
                 (:instance expo-monotone (x y) (y x))
                 (:instance away-to-0-or-fewer-bits (x y))))
          ("subgoal 1" 
           :use (away-monotone-old
                 away-pos
                 away-neg
                 (:instance away-pos (x y))
                 (:instance away-neg (x y))
                 (:instance away-monotone-old (x (- y))
                            (y (- x))))))
  :rule-classes :linear)

(in-theory (disable away-monotone-old))

(defthm away-exactp-d
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0))
	     (<= (abs (away x n)) (expt 2 (1+ (expo x)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo abs away-rewrite)
		  :use ((:instance away-exactp-b-5)))))

(defthm away-pos-rewrite
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (> n 0))
	     (equal (away x n)
		    (* (cg (* (expt 2 (- (1- n) (expo x))) x))
		       (expt 2 (- (1+ (expo x)) n))))))

(in-theory (disable away-rewrite))

(defthm expo-away
    (implies (and (rationalp x)
		  (not (= x 0))
		  (integerp n)
		  (> n 0)
		  (not (= (abs (away x n)) (expt 2 (1+ (expo x))))))
	     (= (expo (away x n)) (expo x)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo abs abs-away away-pos-rewrite)
		  :use ((:instance away-exactp-b-6)
			(:instance expo-monotone (y (away x n)))
			(:instance away-lower-bound)))))

(local
 (defthm away-away-1
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (not (= (away x n) (expt 2 (1+ (expo x)))))
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (= (away (away x n) m)
		(* (cg (* (expt 2 (- (1- m) (expo x)))
			  (* (cg (* (expt 2 (- (1- n) (expo x))) x))
			     (expt 2 (- (1+ (expo x)) n)))))
		   (expt 2 (- (1+ (expo x)) m)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo expo-shift)
		  :use ((:instance away-pos)
			(:instance expo-away))))))

(local
 (defthm away-away-2
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (not (= (away x n) (expt 2 (1+ (expo x)))))
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (= (away (away x n) m)
		(* (cg (* (cg (* (expt 2 (- (1- n) (expo x))) x)) (expt 2 (- m n)))) 
		   (expt 2 (- (1+ (expo x)) m)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo away-pos-rewrite away-rewrite)
		  :use ((:instance away-away-1)
			(:instance expo+ (n (- (1- m) (expo x))) (m (- (1+
                                                                        (expo
                                                                         x))
                                                                       n))))))))


(local
 (defthm away-away-3
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (not (= (away x n) (expt 2 (1+ (expo x)))))
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (= (away (away x n) m)
		(* (cg (/ (cg (* (expt 2 (- (1- n) (expo x))) x)) (expt 2 (- n m)))) 
		   (expt 2 (- (1+ (expo x)) m)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo away-pos-rewrite away-rewrite)
		  :use ((:instance away-away-2))))))

(local
 (defthm away-away-4
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (not (= (away x n) (expt 2 (1+ (expo x)))))
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (= (away (away x n) m)
		(* (cg (/ (* (expt 2 (- (1- n) (expo x))) x) (expt 2 (- n m)))) 
		   (expt 2 (- (1+ (expo x)) m)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable cg/int-rewrite expo away-pos-rewrite away-rewrite)
		  :use ((:instance away-away-3)
			(:instance cg/int-rewrite 
				   (x (* (expt 2 (- (1- n) (expo x))) x))
				   (n (expt 2 (- n m)))))))))

(local
 (defthm away-away-5
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (not (= (away x n) (expt 2 (1+ (expo x)))))
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (= (away (away x n) m)
		(* (cg (* (expt 2 (- (1- m) (expo x))) x))
		   (expt 2 (- (1+ (expo x)) m)))))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo away-pos-rewrite away-rewrite)
		  :use ((:instance away-away-4))))))

(local
 (defthm away-away-6
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (not (= (away x n) (expt 2 (1+ (expo x)))))
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (equal (away (away x n) m)
		    (away x m)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo)
		  :use ((:instance away-away-5))))))

(local
 (defthm away-away-7
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (>= (away x m) (away x n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo exactp2 away-rewrite)
		  :use ((:instance away-exactp-c (a (away x m)))
			(:instance away-exactp-b (n m))
			(:instance away-lower-pos (n m))
			(:instance exactp-<= (x (away x m))))))))
(local
 (defthm away-away-8
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (>= (away x m) (away x n)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo exactp2 away-rewrite)
		  :use ((:instance away-away-7)
			(:instance away-0-0)
			(:instance away-0-0 (n m)))))))

(local
 (defthm away-away-9
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (= (away x n) (expt 2 (1+ (expo x))))
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (equal (away (away x n) m)
		    (away x m)))
  :rule-classes ()
  :hints (("Goal" :in-theory (disable expo away-rewrite exactp2)
		  :use ((:instance away-away-8)
			(:instance exactp-2**n (n (1+ (expo x))))
			(:instance away-exactp-a (x (expt 2 (1+ (expo x)))) (n m))
			(:instance away-exactp-d (n m)))))))

(defthm away-away
    (implies (and (rationalp x)
		  (>= x 0)
		  (integerp n)
		  (integerp m)
		  (> m 0)
		  (>= n m))
	     (equal (away (away x n) m)
		    (away x m)))
  :hints (("Goal" :in-theory (disable expo away-rewrite exactp2)
		  :use ((:instance away-away-9)
			(:instance away-away-6)))))

(in-theory (disable away-away))

(defthm away-shift
  (implies (integerp n)
           (= (away (* x (expt 2 k)) n)
              (* (away x n) (expt 2 k))))
  :instructions
  (:promote (:dv 1)
            :expand :top (:dv 2 1)
            :expand :top (:casesplit (rationalp x))
            (:change-goal nil t)
            :prove (:casesplit (= x 0))
            :prove (:casesplit (integerp k))
            (:change-goal nil t)
            :prove (:dv 1 1)
            (:= (sgn (* x (expt 2 k)))
                (sgn x)
                :hints (("goal" :use sgn-shift)))
            :top (:dv 1 2 1 2)
            (:= (sig (* x (expt 2 k)))
                (sig x)
                :hints
                (("goal" :use (:instance sig-shift (n k)))))
            :top (:dv 1 3 2 1 2)
            (:= (expo (* x (expt 2 k)))
                (+ k (expo x))
                :hints
                (("goal" :use (:instance expo-shift (n k)))))
            :top (:dv 1 3)
            (:= (expt 2 (+ (+ 1 k (expo x)) (- n)))
                (* (expt 2 (+ (+ 1 (expo x)) (- n)))
                   (expt 2 k))
                :hints
                (("goal" :use
                  (:instance a15 (i 2)
                             (j1 k)
                             (j2 (+ (+ 1 (expo x)) (- n)))))))
            :top :prove))

(in-theory (disable away-away))

;from div sqrt

(local
(defthm trunc-away-1
    (implies (and (integerp n) (> n 0)
		  (rationalp x) (> x 0)
		  (not (exactp x n)))
	     (> x (expt 2 (expo x))))
  :rule-classes ()
  :hints (("goal" 
           :in-theory (disable exactp2 exactp2-lemma)
           :use ((:instance expo-lower-bound)
			(:instance exactp-2**n (n (expo x)) (m n)))))))


(local
(defthm trunc-away-2
    (implies (and (integerp n) (> n 0)
		  (rationalp x) (> x 0)
		  (exactp x (1+ n))
		  (not (exactp x n)))
	     (>= (- x (expt 2 (- (expo x) n)))
		 (expt 2 (expo x))))
  :rule-classes ()
  :hints (("goal" :use ((:instance trunc-away-1)
			(:instance exactp-2**n (n (expo x)) (m (1+ n)))
			(:instance fp+1 (x (expt 2 (expo x))) (n (1+ n)) (y x)))))))

(local
(defthm trunc-away-3
    (implies (and (integerp n) (> n 0)
		  (rationalp x) (> x 0)
		  (exactp x (1+ n))
		  (not (exactp x n)))
	     (= (expo (- x (expt 2 (- (expo x) n))))
		(expo x)))
  :rule-classes ()
  :hints (("goal" :in-theory (disable ;expt-pos
                                      )
		  :use ((:instance trunc-away-2)
			(:instance expo-unique (x (- x (expt 2 (- (expo x) n)))) (n (expo x)))
			(:instance exactp-2**n (n (- (expo x) n)) (m n))
;			(:instance expt-pos (x (- (expo x) n)))
			(:instance expo-lower-bound)
			(:instance expt-monotone (n (- (expo x) n)) (m (expo x)))
			(:instance expo-upper-bound))))))

(local
(defthm hack-83
    (implies (and (integerp n)
		  (< 0 n))
	     (= (* 1/2 (expt 2 (+ n (* -1 (expo x)))))
		(expt 2 (+ -1 n (* -1 (expo x))))))
  :rule-classes ()
  :hints (("goal" :use ((:instance expo+ (m (- n (expo x))) (n -1)))))))

(local
(defthm hack-84
    (implies (and (rationalp x)
		  (rationalp a)
		  (rationalp b)
		  (= a b))
	     (= (* x a) (* x b)))
  :rule-classes ()))

(local
 (defthm hack-85
   (implies (and (integerp n)
                 (< 0 n)
                 (rationalp x))
            (equal (* 1/2 x (expt 2 (+ n (* -1 (expo x)))))
                   (* x (expt 2 (+ -1 n (* -1 (expo x)))))))
   :hints (("goal" :use ((:instance hack-83)
                         (:instance hack-84 
                                    (a (* 1/2 (expt 2 (+ n (* -1 (expo x))))))
                                    (b (expt 2 (+ -1 n (* -1 (expo x)))))))))))

(local
 (defthm trunc-away-4
   (implies (and (integerp n) (> n 0)
                 (rationalp x) (> x 0)
                 (exactp x (1+ n))
                 (not (exactp x n)))
            (= (* x (expt 2 (- n (expo x))))
               (1+ (* 2 (fl-half (* x (expt 2 (- n (expo x)))))))))
   :rule-classes ()
   :hints (("goal" :in-theory (enable exactp2-lemma)
            :use ((:instance fl-half-lemma (x (* x (expt 2 (- n (expo x)))))))))))

(local
 (defthm hack-86
   (implies (integerp k)
            (= (- (/ (1+ (* 2 k)) 2) 1/2) k))
   :rule-classes ()))

(local
 (defthm trunc-away-5
   (implies (and (integerp n) (> n 0)
                 (rationalp x) (> x 0)
                 (exactp x (1+ n))
                 (not (exactp x n)))
            (= (* (- x (expt 2 (- (expo x) n)))
                  (expt 2 (- (1- n) (expo x))))
               (fl-half (* x (expt 2 (- n (expo x)))))))
   :rule-classes ()
   :hints (("goal" :use ((:instance trunc-away-4)
                         (:instance hack-86 (k (fl-half (* x (expt 2 (- n (expo x))))))) 
                         (:instance expo+ (m (- (expo x) n)) (n (- (1- n) (expo x))))
                         (:instance expo+ (m 1) (n (- (1- n) (expo x)))))))))

(local
 (defthm trunc-away-6
   (implies (and (integerp n) (> n 0)
                 (rationalp x) (> x 0)
                 (exactp x (1+ n))
                 (not (exactp x n)))
            (integerp (* (- x (expt 2 (- (expo x) n)))
                         (expt 2 (- (1- n) (expo x))))))
   :rule-classes ()
   :hints (("goal" :use ((:instance trunc-away-5))))))

(local
 (defthm trunc-away-7
   (implies (and (integerp n) (> n 0)
                 (rationalp x) (> x 0)
                 (exactp x (1+ n))
                 (not (exactp x n)))
            (integerp (* (- x (expt 2 (- (expo x) n)))
                         (expt 2 (- (1- n) (expo (- x (expt 2 (- (expo x) n)))))))))
   :rule-classes ()
   :hints (("goal" :use ((:instance trunc-away-6)
                         (:instance trunc-away-3))))))

(local
 (defthm trunc-away-8
   (implies (and (integerp n) (> n 0)
                 (rationalp x) (> x 0)
                 (exactp x (1+ n))
                 (>= (- x (expt 2 (- (expo x) n))) 0)
                 (not (exactp x n)))
            (exactp (- x (expt 2 (- (expo x) n)))
                    n))
   :rule-classes ()
   :hints (("goal" :in-theory (enable exactp2-lemma)
            :use ((:instance trunc-away-7))))))

(local
 (defthm trunc-away-9
   (implies (and (integerp n) (> n 0)
                 (rationalp x) (> x 0)
                 (exactp x (1+ n))
                 (not (exactp x n)))
            (exactp (- x (expt 2 (- (expo x) n)))
                    n))
   :rule-classes ()
   :hints (("goal" :use ((:instance trunc-away-8)
                         (:instance expo-lower-bound)
                         (:instance expt-monotone (n (- (expo x) n)) (m (expo x))))))))

(local
 (defthm trunc-away-10
   (implies (and (integerp n) (> n 0)
                 (rationalp x) (> x 0)
                 (exactp x (1+ n))
                 (not (exactp x n)))
            (<= (- x (expt 2 (- (expo x) n)))
                (trunc x n)))
   :rule-classes ()
   :hints (("goal" :in-theory (disable; expt-pos
                               trunc-exactp-c)
            :use ((:instance trunc-away-9)
                  (:instance expo-lower-bound)
                  (:instance expt-monotone (n (- (expo x) n)) (m (expo x)))
;                  (:instance expt-pos (x (- (expo x) n)))
                  (:instance trunc-exactp-c (a (- x (expt 2 (- (expo x) n))))))))))

(local
(defthm trunc-away-11
    (implies (and (integerp n) (> n 0)
		  (rationalp x) (> x 0)
		  (exactp x (1+ n))
		  (not (exactp x n))
		  (< (- x (expt 2 (- (expo x) n)))
		     (trunc x n)))
	     (<= x (trunc x n)))
  :rule-classes ()
  :hints (("goal" :in-theory (disable 
                              ;expt-pos
                                      exactp2 exactp2-lemma trunc-pos-rewrite)
		  :use ((:instance trunc-away-8)
			(:instance trunc-away-3)
			(:instance expo+ (m 1) (n (- n (expo x))))
;			(:instance expt-pos (x (- (expo x) n)))
			(:instance fp+1 (x (- x (expt 2 (- (expo x) n)))) (y (trunc x n)))
			(:instance expo-lower-bound)
			(:instance expt-strong-monotone (n (- (expo x) n)) (m (expo x)))
			(:instance expt-strong-monotone (n (- (expo x) n)) (m (- (1+ (expo x)) n))))))))

(defthm trunc-away-a
    (implies (and (integerp n) (> n 0)
		  (rationalp x) (> x 0)
		  (exactp x (1+ n))
		  (not (exactp x n)))
	     (= (- x (expt 2 (- (expo x) n)))
		(trunc x n)))
  :rule-classes ()
  :hints (("goal" :in-theory (disable trunc-exactp-b exactp2 exactp2-lemma trunc-pos-rewrite)
		  :use ((:instance trunc-away-10)
			(:instance trunc-away-11)
			(:instance trunc-upper-pos)
			(:instance trunc-exactp-b)))))
(local
(defthm hack-87
    (implies (and (rationalp x)
		  (integerp n)
		  (= (expo (- x (expt 2 (- (expo x) n))))
		     (expo x)))
	     (equal (+ x (* -1 (expt 2 (+ (expo x) (* -1 n))))
		       (expt 2
			 (+ 1 (* -1 n)
			    (expo (+ x
				     (* -1 (expt 2 (+ (expo x) (* -1 n)))))))))
		    (+ x (expt 2 (+ (expo x) (* -1 n))))))
  :rule-classes ()
  :hints (("goal" :use ((:instance expo+ (m (- (expo x) n)) (n 1)))))))

(local
(defthm hack-88
    (implies (equal x y) (equal (exactp x n) (exactp y n)))
  :rule-classes ()))

(local
(defthm hack-89
    (implies (and (rationalp x)
		  (integerp n)
		  (= (expo (- x (expt 2 (- (expo x) n))))
		     (expo x)))
	     (equal (exactp (+ x (* -1 (expt 2 (+ (expo x) (* -1 n))))
			       (expt 2
				     (+ 1 (* -1 n)
					(expo (+ x
						 (* -1 (expt 2 (+ (expo x) (* -1 n)))))))))
			    n)
		    (exactp (+ x (expt 2 (+ (expo x) (* -1 n)))) n)))
  :rule-classes ()
  :hints (("goal" :use ((:instance hack-87)
			(:instance hack-88 
				   (x (+ x (* -1 (expt 2 (+ (expo x) (* -1 n))))
					 (expt 2
					       (+ 1 (* -1 n)
						  (expo (+ x
							   (* -1 (expt 2 (+ (expo x) (* -1 n))))))))))
				   (y (+ x (expt 2 (+ (expo x) (* -1 n)))))))))))

(local (in-theory (disable expo-monotone)))

(local
(defthm trunc-away-12
    (implies (and (integerp n) (> n 0)
		  (rationalp x) (> x 0)
		  (exactp x (1+ n))
		  (not (exactp x n)))
	     (exactp (+ x (expt 2 (- (expo x) n)))
		     n))
  :rule-classes ()
  :hints (("goal" :use ((:instance trunc-away-9)
			(:instance fp+2 (x (- x (expt 2 (- (expo x) n)))))
			(:instance expo-lower-bound)
			(:instance expt-strong-monotone (n (- (expo x) n)) (m (expo x)))
			(:instance expo+ (m (- (expo x) n)) (n 1))
			(:instance trunc-away-3)))
	  ("subgoal 1" :use ((:instance hack-89))))))

(local
(defthm trunc-away-13
    (implies (and (integerp n) (> n 0)
		  (rationalp x) (> x 0)
		  (exactp x (1+ n))
		  (not (exactp x n)))
	     (>= (+ x (expt 2 (- (expo x) n)))
		 (away x n)))
  :rule-classes ()
  :hints (("goal" :in-theory (disable ;expt-pos
                                      away-exactp-c)
		  :use ((:instance trunc-away-12)
			(:instance expo-lower-bound)
			(:instance expt-monotone (n (- (expo x) n)) (m (expo x)))
;			(:instance expt-pos (x (- (expo x) n)))
			(:instance away-exactp-c (a (+ x (expt 2 (- (expo x) n))))))))))

(local
(defthm trunc-away-14
    (implies (and (integerp n) (> n 0)
		  (rationalp x) (> x 0)
		  (exactp x (1+ n))
		  (not (exactp x n)))
	     (> (away x n)
		(- x (expt 2 (- (expo x) n)))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable ;expt-pos
                              )
		  :use ((:instance away-lower-pos)
;			(:instance expt-pos (x (- (expo x) n)))
                        )))))

(local
(defthm trunc-away-15
    (implies (and (integerp n) (> n 0)
		  (rationalp x) (> x 0)
		  (exactp x (1+ n))
		  (not (exactp x n)))
	     (>= (away x n)
		 (+ (- x (expt 2 (- (expo x) n)))
		    (expt 2 (- (1+ (expo x)) n)))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable away-exactp-b)
		  :use ((:instance trunc-away-8)
			(:instance trunc-away-3)
			(:instance expo-lower-bound)
			(:instance expt-strong-monotone (n (- (expo x) n)) (m (expo x)))
			(:instance trunc-away-14)
			(:instance away-exactp-b)
			(:instance fp+1 (x (- x (expt 2 (- (expo x) n)))) (y (away x n))))))))

(local
(defthm trunc-away-16
    (implies (and (integerp n) (> n 0)
		  (rationalp x) (> x 0)
		  (exactp x (1+ n))
		  (not (exactp x n)))
	     (>= (away x n)
		 (+ x (expt 2 (- (expo x) n)))))
  :rule-classes ()
  :hints (("goal" :use ((:instance trunc-away-15)
			(:instance expo+ (m 1) (n (- (expo x) n))))))))

(defthm trunc-away-b
    (implies (and (integerp n) (> n 0)
		  (rationalp x) (> x 0)
		  (exactp x (1+ n))
		  (not (exactp x n)))
	     (= (away x n)
		(+ x (expt 2 (- (expo x) n)))))
  :rule-classes ()
  :hints (("goal" :use ((:instance trunc-away-16)
			(:instance trunc-away-13)))))





(local (defthm away-imp-1
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (integerp m)
		  (>= m n)
		  (exactp x m))
	     (< (trunc (+ x
			  (expt 2 (- (1+ (expo x)) n))
			  (- (expt 2 (- (1+ (expo x)) m))))
		       n)
		(+ x (expt 2 (- (1+ (expo x)) n)))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable ;expt-pos
                              )
		  :use ((:instance trunc-upper-pos
				   (x (+ x
					 (expt 2 (- (1+ (expo x)) n))
					 (- (expt 2 (- (1+ (expo x)) m))))))
;			(:instance expt-pos (x (- (1+ (expo x)) m)))
			(:instance expt-monotone (n (- (1+ (expo x)) m)) (m (- (1+ (expo x)) n))))))))

(local (in-theory (disable abs-pos abs-away)))

(local (defthm away-imp-2
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0))
	     (<= (+ x (expt 2 (- (1+ (expo x)) n)))
		 (+ (away x n)
		    (expt 2 (- (1+ (expo (away x n))) n)))))
  :rule-classes ()
  :hints (("goal" :use ((:instance away-lower-pos)
			(:instance expo-monotone (y (away x n)))
			(:instance expt-monotone
				   (n (- (1+ (expo x)) n)) (m (- (1+ (expo (away x n))) n))))))))

(local (defthm away-imp-3
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (integerp m)
		  (>= m n)
		  (exactp x m))
	     (< (trunc (+ x
			  (expt 2 (- (1+ (expo x)) n))
			  (- (expt 2 (- (1+ (expo x)) m))))
		       n)
		(+ (away x n)
		   (expt 2 (- (1+ (expo (away x n))) n)))))
  :rule-classes ()
  :hints (("goal" :use (away-imp-1 away-imp-2)))))

(local 
 (defthm away-imp-4
   (implies (and (rationalp x)
                 (> x 0)
                 (integerp n)
                 (> n 0)
                 (integerp m)
                 (>= m n)
                 (exactp x m))
            (<= (trunc (+ x
                          (expt 2 (- (1+ (expo x)) n))
                          (- (expt 2 (- (1+ (expo x)) m))))
                       n)
                (away x n)))
   :rule-classes ()
   :hints (("goal" :in-theory (disable away-exactp-b trunc-exactp-b)
            :use (away-imp-3
                  (:instance fp+1
                             (x (away x n))
                             (y (trunc (+ x
                                          (expt 2 (- (1+ (expo x)) n))
                                          (- (expt 2 (- (1+ (expo x)) m))))
                                       n)))
                  (:instance away-pos)
                  (:instance away-exactp-b)
                  (:instance trunc-exactp-b
                             (x (+ x
                                   (expt 2 (- (1+ (expo x)) n))
                                   (- (expt 2 (- (1+ (expo x)) m)))))))))))

(local (defthm away-imp-5
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (integerp m)
		  (>= m n)
		  (exactp x m)
		  (exactp x n))
	     (>= (trunc (+ x
			   (expt 2 (- (1+ (expo x)) n))
			   (- (expt 2 (- (1+ (expo x)) m))))
			n)
		 (away x n)))
  :rule-classes ()
  :hints (("goal" :use ((:instance trunc-monotone
				   (y (+ x
					 (expt 2 (- (1+ (expo x)) n))
					 (- (expt 2 (- (1+ (expo x)) m))))))
			(:instance expt-monotone (n (- (1+ (expo x)) m)) (m (- (1+ (expo x)) n)))
			(:instance trunc-exactp-a)
			(:instance away-exactp-a))))))

(local (defthm away-imp-6
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (integerp m)
		  (>= m n)
		  (exactp x m)
		  (not (exactp x n)))
	     (>= x
		 (+ (trunc x n)
		    (expt 2 (- (1+ (expo x)) m)))))
  :rule-classes ()
  :hints (("goal" :in-theory (disable trunc-exactp-b exactp2 exactp2-lemma trunc-pos-rewrite)
           :use (trunc-exactp-a
			trunc-pos
			trunc-upper-pos
			trunc-exactp-b
			(:instance exactp-<= (x (trunc x n)) (n m) (m n))
			(:instance fp+1 (x (trunc x n)) (y x) (n m)))))))

(local (defthm away-imp-7
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (integerp m)
		  (>= m n)
		  (exactp x m)
		  (not (exactp x n)))
	     (>= (+ x
		    (expt 2 (- (1+ (expo x)) n))
		    (- (expt 2 (- (1+ (expo x)) m))))
		 (+ (trunc x n)
		    (expt 2 (- (1+ (expo x)) n)))))
  :rule-classes ()
  :hints (("goal" :use (away-imp-6)))))

;(local (in-theory (disable expt-pos)))

(local (defthm away-imp-8
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (not (exactp x n)))
	     (> (+ (trunc x n)
		   (expt 2 (- (1+ (expo x)) n)))
		x))
  :rule-classes ()
  :hints (("goal" :in-theory (disable trunc-exactp-c exactp2
                                      exactp2-lemma trunc-exactp-b trunc-pos-rewrite)

           :use ((:instance fp+2 (x (trunc x n)))
			(:instance trunc-exactp-b)
			(:instance trunc-pos)
;			(:instance expt-pos (x (- (1+ (expo x)) n)))
			(:instance trunc-exactp-c 
				   (a (+ (trunc x n) (expt 2 (- (1+ (expo x)) n))))))))))

(local (defthm away-imp-9
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (not (exactp x n)))
	     (>= (+ (trunc x n)
		   (expt 2 (- (1+ (expo x)) n)))
		(away x n)))
  :rule-classes ()
  :hints (("goal"  :in-theory (disable trunc-exactp-b exactp2 exactp2-lemma
                                       trunc-pos-rewrite away-exactp-c)
           :use (away-imp-8
			(:instance fp+2 (x (trunc x n)))
			(:instance trunc-exactp-b)
			(:instance trunc-pos)
			(:instance away-exactp-c 
				   (a (+ (trunc x n) (expt 2 (- (1+ (expo x)) n))))))))))

(local (defthm away-imp-10
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (integerp m)
		  (>= m n)
		  (exactp x m)
		  (not (exactp x n)))
	     (>= (+ x
		    (expt 2 (- (1+ (expo x)) n))
		    (- (expt 2 (- (1+ (expo x)) m))))
		 (away x n)))
  :rule-classes ()
  :hints (("goal" :use (away-imp-7 away-imp-9)))))

(local (defthm away-imp-11
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (integerp m)
		  (>= m n)
		  (exactp x m)
		  (not (exactp x n)))
	     (>= (trunc (+ x
			   (expt 2 (- (1+ (expo x)) n))
			   (- (expt 2 (- (1+ (expo x)) m))))
			n)
		 (away x n)))
  :rule-classes ()
  :hints (("goal" :in-theory (disable away-exactp-b trunc-exactp-c)
           :use (away-imp-10
			away-exactp-b
			(:instance expt-monotone (n (- (1+ (expo x)) m)) (m (- (1+ (expo x)) n)))
			(:instance trunc-exactp-c
				   (a (away x n))
				   (x (+ x
					 (expt 2 (- (1+ (expo x)) n))
					 (- (expt 2 (- (1+ (expo x)) m)))))))))))

(defthm away-imp
    (implies (and (rationalp x)
		  (> x 0)
		  (integerp n)
		  (> n 0)
		  (integerp m)
		  (>= m n)
		  (exactp x m))
	     (= (away x n)
		(trunc (+ x
			  (expt 2 (- (1+ (expo x)) n))
			  (- (expt 2 (- (1+ (expo x)) m))))
		       n)))
  :rule-classes ()
  :hints (("goal" :use (away-imp-11 away-imp-5 away-imp-4))))

