/* 
 * xwave - an interactive audio player, recorder, editor 
 * for the XWindow System
 * 
 * Copyright (C) 1996 Kai Kollmorgen
 * (kkollmor@informatik.uni-rostock.de)
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <ctype.h>
#include <dirent.h>
#include <math.h>
#include <X11/Intrinsic.h>	/* Intrinsics Definitions */
#include <X11/StringDefs.h>	/* Standard Name-String definitions */
#include <X11/Shell.h>     	/* Shell Definitions */

#include <X11/Xaw/Command.h>	/* Athena Command Widget */
#include <X11/Xaw/Form.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Repeater.h>

#include <Xfwf/RadioGrp.h>

#include "types.h"
#include "xwave.h"
#include "xwave_widget.h"
#include "misc.h"
#include "filebrowse.h"
#include "audio_file.h"
#include "sample_settings.h"

/* MD, home ,app_resources are global in xwave.c */

extern Main_Data *MD;
extern AppResources app_resources;
extern Pixmap upBitmap;
extern Pixmap downBitmap;

static void create_filetype_dialog (Widget w);
static void create_absvol_dialog (Widget w);
static void create_envvol_dialog (Widget w);
static void create_echo_dialog (Widget w);
static void ca_CB(Widget w, XtPointer client_data, XtPointer call_data);
static void ss_ok_CB(Widget w, XtPointer client_data, XtPointer call_data);
static void ft_ok_CB(Widget w, XtPointer client_data, XtPointer call_data);
static void type_CB(Widget w, XtPointer client_data,XtPointer call_data);
static void comp_CB(Widget w, XtPointer client_data,XtPointer call_data);
static void res_CB(Widget w, XtPointer client_data,XtPointer call_data);
static void chn_CB(Widget w, XtPointer client_data,XtPointer call_data);
static void frq_CB(Widget w, XtPointer client_data,XtPointer call_data);
static void up_CB(Widget w, XtPointer client_data,XtPointer call_data);
static void down_CB(Widget w, XtPointer client_data,XtPointer call_data);
static Boolean check_freq();
static void (*sample_return)(Sample_Return *sr);
static void (*ft_return)(int type,int comp);
static void av_ok_CB(Widget w, XtPointer client_data, XtPointer call_data);
static void ev_ok_CB(Widget w, XtPointer client_data, XtPointer call_data);
static void ec_ok_CB(Widget w, XtPointer client_data, XtPointer call_data);
static void (*av_return)(float mult1,float mult2);
static void (*ev_return)(float fo_time[MAX_CHANNELS],
			 float fo_factor[MAX_CHANNELS],
			 float sup_amp[MAX_CHANNELS]);
static void (*ec_return)(float fo_time[MAX_CHANNELS],
			 float fo_factor[MAX_CHANNELS]);

/* we need some globals */

static Widget popup,userinput,up,down,rgroup1,rgroup2,rgroup3;
static Widget ft_popup,ft_rgroup1,ft_rgroup2;
static char userfreq[MAX_NUMLENGTH];
static int ft_type,ft_comp;
static Widget av_popup,av_label,av1_text,av2_text;
static char av1_string[MAX_NUMLENGTH],av2_string[MAX_NUMLENGTH];
static Widget ev_popup,ev1_text,ev2_text,ev3_text,ev4_text,ev5_text,ev6_text;
static char ev_fo_time_str[MAX_CHANNELS][MAX_NUMLENGTH],
ev_fo_factor_str[MAX_CHANNELS][MAX_NUMLENGTH],
ev_sup_amp_str[MAX_CHANNELS][MAX_NUMLENGTH];
static Widget ec_popup,ec1_text,ec2_text,ec3_text,ec4_text;
static char ec_time_str[MAX_CHANNELS][MAX_NUMLENGTH],
ec_fo_str[MAX_CHANNELS][MAX_NUMLENGTH];


Boolean check_freq()
{
   int freq,i;
   
   /* prepare error message */
   sprintf(MD->mw->messages,"%s %i - %i",
	   app_resources.err_input,MINFREQ,MAXFREQ);
   
   if (strcmp(userfreq,"")==0) {
      warn_popup(userinput,MD->mw->messages);
      return(False);
   }

   freq=atoi(userfreq);

   i=strlen(userfreq)-1;
   while (i>=0) if (!isdigit(userfreq[i--])) freq=-1;
   
   if (freq<MINFREQ) {
      warn_popup(userinput,MD->mw->messages);
      return(False);
   }
   if (freq>MAXFREQ) {
      warn_popup(userinput,MD->mw->messages);
      return(False);
   }
   return(True);
}

void ca_CB(Widget w, XtPointer client_data, XtPointer call_data)
{
   XtPopdown((Widget) client_data);
}

void ss_ok_CB(Widget w, XtPointer client_data, XtPointer call_data)
{
   long choice;
   Display *dpy=XtDisplay(w);
   Widget popup=(Widget) client_data;
   Sample_Return *sr;
   
   XtVaGetValues(rgroup3, XtNselection, &choice, NULL);
   if (choice==3 /* user defined */) 
     if (!check_freq()) return;

   XtPopdown(popup);
   XFlush(dpy);
   
   sr=XtNew(Sample_Return);

   switch (choice)  {
    case 0: sr->freq=LOWFREQ;break;
    case 1: sr->freq=MIDFREQ;break;
    case 2: sr->freq=HIFREQ;break;
    case 3: sr->freq=atoi(userfreq);break;
   }
   XtVaGetValues(rgroup1, XtNselection, &choice, NULL);
   sr->res= (choice == 0) ? 8 : 16;
   XtVaGetValues(rgroup2, XtNselection, &choice, NULL);
   sr->channels= (choice == 0) ? MONO : STEREO;

   sample_return(sr);
}

void res_CB(Widget w, XtPointer client_data,XtPointer call_data)
{
   long choice;
   XtVaGetValues(w, XtNselection, &choice, NULL);
   if (choice && (!MD->mb->canplay16bit))  {
      warn_popup(w,app_resources.err_16bit);
      XtVaSetValues(w, XtNselection, 0 /* 8bit */ , NULL);
   }
}

void chn_CB(Widget w, XtPointer client_data,XtPointer call_data)
{
   long choice1,choice2;
   Widget frq=(Widget) client_data;
   
   XtVaGetValues(w, XtNselection, &choice1, NULL);
   XtVaGetValues(frq, XtNselection, &choice2, NULL);
   
   if (choice1 /* stereo */) {
      if (!MD->mb->canplaystereo) {
	 warn_popup(w,app_resources.err_stereo);
	 XtVaSetValues(w, XtNselection, 0 /* mono */, NULL);
      }
      if ((choice2==2 /* 44100 hz */) && (!MD->mb->canplay44khzstereo))  {
	 warn_popup(w,app_resources.err_44khzstereo);
	 XtVaSetValues(w, XtNselection, 0 /* mono */, NULL);
      }
   }
}

void frq_CB(Widget w, XtPointer client_data,XtPointer call_data)
{
   long choice1,choice2;
   Widget chn=(Widget) client_data;

   XtVaGetValues(w, XtNselection, &choice1, NULL);
   XtVaGetValues(chn, XtNselection, &choice2, NULL);
   
   if (choice1==2 /* 44100 hz */)  {
      if (!MD->mb->canplay44khz)  {
	 warn_popup(w,app_resources.err_44khz);
	 XtVaSetValues(w, XtNselection, 1 /* 22050 hz */ , NULL);
      }
      if (choice2 /* stereo */ && (!MD->mb->canplay44khzstereo))  {
	 warn_popup(w,app_resources.err_44khzstereo);
	 XtVaSetValues(w, XtNselection, 1 /* 22050 hz */, NULL);
      }
   }    
   if (choice1==3 /* user defined */)  {
      XtVaSetValues(userinput,XtNsensitive,True,NULL);
      XtVaSetValues(up,XtNsensitive,True,NULL);
      XtVaSetValues(down,XtNsensitive,True,NULL);
   } else  {
      XtVaSetValues(userinput,XtNsensitive,False,NULL);
      XtVaSetValues(up,XtNsensitive,False,NULL);
      XtVaSetValues(down,XtNsensitive,False,NULL);
   }
}

void up_CB(Widget w, XtPointer client_data,XtPointer call_data)
{
   int freq;

   if (strcmp(userfreq,"")==0) freq=0;
   else freq=atoi(userfreq);
   
   if (freq<MINFREQ) freq=MINFREQ;
   else if (freq>=MAXFREQ) { freq=MAXFREQ;return; }

   freq++;
   sprintf(userfreq,"%i",freq);
   XtVaSetValues(userinput,XtNstring,userfreq,NULL);
}

void down_CB(Widget w, XtPointer client_data,XtPointer call_data)
{
   int freq;

   if (strcmp(userfreq,"")==0) freq=0;
   else freq=atoi(userfreq);
   
   if (freq<=MINFREQ) { freq=MINFREQ;return; }
   else if (freq>MAXFREQ) freq=MAXFREQ;

   freq--;
   sprintf(userfreq,"%i",freq);
   XtVaSetValues(userinput,XtNstring,userfreq,NULL);
}

void create_sample_dialog (Widget w)
{
   Widget form,label,form1,button;
   static XtAccelerators accel;
 
   
   accel = XtParseAcceleratorTable("#override\n\
                                    <Key>Return: set() notify() unset()\n");
 
   popup=XtVaCreatePopupShell("ss_shell",transientShellWidgetClass,w,
			      NULL);   

   form  = MW ("ss_main_form",formWidgetClass,popup,
		XtNresizable,True,
		XtNaccelerators, accel,
		XtNborderWidth,0,
   		NULL);

   rgroup1 = MW ("ss_res_rgroup",xfwfRadioGroupWidgetClass, form, NULL);
   XtAddCallback (rgroup1, XtNactivate, res_CB , (XtPointer) rgroup1);

   rgroup2 = MW ("ss_chn_rgroup",xfwfRadioGroupWidgetClass, form,
		   XtNfromHoriz,rgroup1,NULL);

   rgroup3 = MW ("ss_freq_rgroup",xfwfRadioGroupWidgetClass, form, 
		 XtNfromVert,rgroup1,NULL);
   
   XtAddCallback (rgroup2, XtNactivate, chn_CB , (XtPointer) rgroup3);
   XtAddCallback (rgroup3, XtNactivate, frq_CB , (XtPointer) rgroup2);

   label = MW ("ss_ufreq_label",labelWidgetClass,form,
	       XtNborderWidth,0,
	       XtNfromVert,rgroup1,XtNfromHoriz,rgroup3,
	       XtNbottom, XtChainTop,XtNtop, XtChainTop,
	       XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);
	       
   strcpy(userfreq,"");
   userinput = MW ("ss_ufreq_text",asciiTextWidgetClass,form,
		   XtNfromVert,label,XtNfromHoriz,rgroup3,
		   XtNsensitive, False,
		   XtNeditType, XawtextEdit,
		   XtNwrap, XawtextWrapNever,
		   XtNresize, XawtextResizeWidth,
		   XtNuseStringInPlace, True,
		   XtNlength,MAX_NUMLENGTH,XtNstring,&userfreq,
		   XtNbottom, XtChainTop,XtNtop, XtChainTop,
		   XtNleft,XawChainLeft,XtNright,XawChainLeft,
		   NULL);
   
   form1= MW ("ss_ufreq_form",formWidgetClass,form,
	      XtNfromVert,label,XtNfromHoriz,userinput,
	      XtNhorizDistance,0,
	      XtNborderWidth, 0,
	      XtNtop,XawChainTop,XtNbottom,XawChainTop,
	      XtNleft,XawChainLeft,XtNright,XawChainLeft,
	      NULL);
    
   up = MW ("ss_ufreq_up_btn",repeaterWidgetClass,form1,
	    XtNvertDistance,0,
	    XtNtop,XawChainTop,XtNbottom,XawChainTop,
	    XtNleft,XawChainLeft,XtNright,XawChainLeft,
	    XtNbitmap, upBitmap,
	    XtNsensitive, False,
	    NULL);
   
   XtAddCallback (up, XtNcallback, up_CB,(XtPointer) NULL); 

   down = MW ("ss_ufreq_down_btn",repeaterWidgetClass,form1,
	      XtNfromVert,up,
	      XtNvertDistance,0,
	      XtNtop,XawChainBottom,XtNbottom,XawChainBottom,
	      XtNleft,XawChainLeft,XtNright,XawChainLeft,
	      XtNbitmap, downBitmap,
	      XtNsensitive, False,
	      NULL);
   
   XtAddCallback (down, XtNcallback, down_CB,(XtPointer) NULL);

   label = MW ("ss_sep1_label",labelWidgetClass,form,
	       XtNfromVert,rgroup3,XtNlabel,"",NULL);

   button = MW ("ss_ok_btn",commandWidgetClass,form,
		XtNfromVert,label,
		XtNaccelerators, accel,
    		XtNtop,XawChainTop,XtNbottom,XawChainTop,
    		XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);

   XtAddCallback (button, XtNcallback, ss_ok_CB, (XtPointer) popup);    
   XtInstallAccelerators(userinput, button);
   XtInstallAccelerators(form, button);
 
   button = MW ("ss_ca_btn",commandWidgetClass,form,
		XtNfromVert,label,XtNfromHoriz,button,		
    		XtNtop,XawChainTop,XtNbottom,XawChainTop,
    		XtNleft,XawChainRight,XtNright,XawChainRight,NULL);

   XtAddCallback (button, XtNcallback, ca_CB, (XtPointer) popup);    

   label = MW ("ss_sep2_label",labelWidgetClass,form,
	       XtNfromVert,button,XtNlabel,"",NULL);

   /* create widgets for filetype dialog */
   create_filetype_dialog(w);

   /* create widgets for change absolute volume dialog */
   create_absvol_dialog(w);

   /* create widgets for dynamic compressor dialog */
   create_envvol_dialog(w);

   /* create widgets for echo dialog */
   create_echo_dialog(w);
}

void sample_dialog (void (*sample_callback)(Sample_Return *sr),Wave_Data *wd)
{
   long choice=0;
   
   sample_return=sample_callback;
   
   if (wd!=NULL) {
   
      switch (wd->res) {
       case 8:choice=0;break;
       case 16:choice=1;break;
      }
      XtVaSetValues(rgroup1, XtNselection, choice, NULL);
      XtVaSetValues(rgroup2, XtNselection, wd->channels-1, NULL);
      switch (wd->freq)  {
       case LOWFREQ: choice=0;break;
       case MIDFREQ: choice=1;break;
       case HIFREQ: choice=2;break;
       default: 
	 choice=3; 
	 sprintf(userfreq,"%i",wd->freq);
	 XtVaSetValues(userinput,XtNstring,userfreq,NULL);
      }
      XtVaSetValues(rgroup3, XtNselection, choice, NULL);
   }
   popup_centered(popup);
}

void ft_ok_CB(Widget w, XtPointer client_data, XtPointer call_data)
{
   XtPopdown((Widget) client_data);
   ft_return(ft_type,ft_comp);
}

void type_CB(Widget w, XtPointer client_data,XtPointer call_data)
{
   long choice1,choice2;
   
   XtVaGetValues(ft_rgroup1, XtNselection, &choice1, NULL);
   XtVaGetValues(ft_rgroup2, XtNselection, &choice2, NULL);
   ft_type=choice1;
   ft_comp=choice2;
}

void comp_CB(Widget w, XtPointer client_data,XtPointer call_data)
{
   long choice1,choice2;
   
   XtVaGetValues(ft_rgroup1, XtNselection, &choice1, NULL);
   XtVaGetValues(ft_rgroup2, XtNselection, &choice2, NULL);
   ft_type=choice1;
   ft_comp=choice2;
}

void filetype_dialog (void (*filetype_callback)(int type,int comp))
{
   long choice1,choice2;

   choice1=app_resources.default_ftype;
   if (choice1>AF_MAX_TYPE) choice1=0;
   choice2=app_resources.default_comp;
   if (choice2>AF_MAX_COMP) choice2=0;
   switch (choice1) {
    case AF_RIFF:
      break;
    case AF_AIFC:
      choice2=0;
      break;
    case AF_AU:
      switch (choice2) {
       case AF_PCM:
	 break;
       case AF_MULAW:
	 break;
       default:
	 choice2=0;
      }
      break;
    case AF_RAW:
      choice2=0;
      break;
   }
   XtVaSetValues(ft_rgroup1, XtNselection, choice1, NULL);
   XtVaSetValues(ft_rgroup2, XtNselection, choice2, NULL);

   ft_return=filetype_callback;
   popup_centered(ft_popup);
}


void create_filetype_dialog (Widget w)
{
   Widget form,label,button;
   
   ft_popup=XtVaCreatePopupShell("ft_shell",transientShellWidgetClass,w,NULL);

   form  = MW ("ft_main_form",formWidgetClass,ft_popup,
		XtNresizable,TRUE,
		XtNborderWidth,0,
   		NULL);

   ft_rgroup1 = MW ("ft_type_rgroup",xfwfRadioGroupWidgetClass, form, NULL);

   ft_rgroup2 = MW ("ft_comp_rgroup",xfwfRadioGroupWidgetClass, form,
		    XtNfromHoriz,ft_rgroup1,NULL);

   XtAddCallback (ft_rgroup1, XtNactivate, type_CB , (XtPointer) ft_rgroup2);
   XtAddCallback (ft_rgroup2, XtNactivate, comp_CB , (XtPointer) ft_rgroup1);


   button = MW ("ft_ok_btn",commandWidgetClass,form,
		XtNfromVert,ft_rgroup1,		
    		XtNtop,XawChainTop,XtNbottom,XawChainTop,
    		XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);

   XtAddCallback (button, XtNcallback, ft_ok_CB, (XtPointer) ft_popup);    

   button = MW ("ft_ca_btn",commandWidgetClass,form,
		XtNfromVert,ft_rgroup1,XtNfromHoriz,button,		
    		XtNtop,XawChainTop,XtNbottom,XawChainTop,
    		XtNleft,XawChainRight,XtNright,XawChainRight,NULL);

   XtAddCallback (button, XtNcallback, ca_CB, (XtPointer) ft_popup);    

   label = MW ("ft_sep_label",labelWidgetClass,form,
	       XtNfromVert,button,XtNlabel,"",NULL);

}

void av_ok_CB(Widget w, XtPointer client_data, XtPointer call_data)
{
   float mult1,mult2=0.0;
   
   mult1=atof(av1_string);

   if ((mult1==0.0)||(mult1>MAXVOL)||(mult1<MINVOL)) {
      sprintf(MD->mw->messages,"%s %.2f < 0.0 > %.2f",
	      app_resources.err_input,MINVOL,MAXVOL);
      warn_popup(av1_text,MD->mw->messages);
      return;
   }

   if (MD->wd->channels>1) {
      mult2=atof(av2_string);
      
      if ((mult2==0.0)||(mult2>MAXVOL)||(mult2<MINVOL)) {
	 sprintf(MD->mw->messages,"%s %.2f < 0.0 > %.2f",
		 app_resources.err_input,MINVOL,MAXVOL);
	 warn_popup(av2_text,MD->mw->messages);
	 return;
      }
   }
   
   XtPopdown((Widget) client_data);
   XFlush(XtDisplay(w));
   
   av_return(mult1,mult2);
}

void absvol_dialog (void (*absvol_callback)(float mult1,float mult2),
		    Main_Data *md)
{
   float max;
   av_return=absvol_callback;
   
   max=pow(2.0,(float)md->wd->res)/2.0;

   sprintf(md->mw->messages,"Ch.1=%i",md->wd->peak_l);
   sprintf(av1_string,"%.3f",max/(float)md->wd->peak_l);
   strcpy(av2_string,"");
   XtVaSetValues(av2_text,XtNsensitive,False,NULL);
   if (md->wd->channels==2) {
      sprintf(md->mw->messages,"Ch.1=%i  Ch.2=%i",
	      md->wd->peak_l,md->wd->peak_r);
      XtVaSetValues(av2_text,XtNsensitive,True,NULL);
      sprintf(av2_string,"%.3f",max/(float)md->wd->peak_r);
      XtVaSetValues(av2_text,XtNsensitive,True,NULL);
   }
   XtVaSetValues(av1_text,XtNstring,av1_string,NULL);
   XtVaSetValues(av2_text,XtNstring,av2_string,NULL);
   
   XtVaSetValues(av_label,XtNlabel,md->mw->messages,NULL);
   popup_centered(av_popup);
}

void create_absvol_dialog (Widget w)
{
   Widget form,label,button;
   static XtAccelerators accel;
 
   accel = XtParseAcceleratorTable("#override\n\
                                    <Key>Return: set() notify() unset()\n");

   av_popup=XtVaCreatePopupShell("av_shell",transientShellWidgetClass,
				 w,NULL);

   form  = MW ("av_main_form",formWidgetClass,av_popup,
		XtNborderWidth,0,
   		NULL);

   label = MW ("av_peak_label",labelWidgetClass,form,
	       XtNborderWidth,0,NULL);
   
   av_label = MW ("av_value_label",labelWidgetClass,form,
		  XtNfromVert,label,
		  XtNresizable,True,
		  XtNborderWidth,0,NULL);

   label = MW ("av_text_label",labelWidgetClass,form,
	       XtNfromVert,av_label,
	       XtNborderWidth,0,NULL);

   av1_text = MW ("av_abs1_text",asciiTextWidgetClass,form,
		   XtNfromVert,label,
		   XtNeditType, XawtextEdit,
		   XtNwrap, XawtextWrapNever,
		   XtNresize, XawtextResizeWidth,
		   XtNuseStringInPlace, True,
		   XtNlength,MAX_NUMLENGTH,XtNstring,&av1_string,
		   XtNbottom, XtChainTop,XtNtop, XtChainTop,
		   XtNleft,XawChainLeft,XtNright,XawChainLeft,
		   NULL);
   
   av2_text = MW ("av_abs2_text",asciiTextWidgetClass,form,
		   XtNfromVert,label,XtNfromHoriz,av1_text,
		   XtNeditType, XawtextEdit,
		   XtNwrap, XawtextWrapNever,
		   XtNresize, XawtextResizeWidth,
		   XtNuseStringInPlace, True,
		   XtNlength,MAX_NUMLENGTH,XtNstring,&av2_string,
		   XtNbottom, XtChainTop,XtNtop, XtChainTop,
		   XtNleft,XawChainLeft,XtNright,XawChainLeft,
		   NULL);
   
   button = MW ("av_ok_btn",commandWidgetClass,form,
		XtNfromVert,av2_text,		
		XtNaccelerators, accel,
    		XtNtop,XawChainTop,XtNbottom,XawChainTop,
    		XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);

   XtAddCallback (button, XtNcallback, av_ok_CB, (XtPointer) av_popup);    

   button = MW ("av_ca_btn",commandWidgetClass,form,
		XtNfromVert,av2_text,XtNfromHoriz,button,		
    		XtNtop,XawChainTop,XtNbottom,XawChainTop,
    		XtNleft,XawChainRight,XtNright,XawChainRight,NULL);

   XtAddCallback (button, XtNcallback, ca_CB, (XtPointer) av_popup);    

}

void ev_ok_CB(Widget w, XtPointer client_data, XtPointer call_data)
{
   float fo_time[MAX_CHANNELS];
   float fo_factor[MAX_CHANNELS];
   float sup_amp[MAX_CHANNELS];
   int i;
   
   for (i=0;i<MD->wd->channels;i++) {
      fo_time[i]=atof(ev_fo_time_str[i]);
      fo_factor[i]=atof(ev_fo_factor_str[i]);
      sup_amp[i]=atof(ev_sup_amp_str[i]);
   
      if (fo_time[i]<=IN_FO_TIME_MIN || fo_time[i]>IN_FO_TIME_MAX) {
	 sprintf(MD->mw->messages,"%s %.4f - %.4f",
		 app_resources.err_input,IN_FO_TIME_MIN,IN_FO_TIME_MAX);
	 warn_popup(ev_popup,MD->mw->messages);
	 return;
      }
      if (fo_factor[i]<=IN_FO_FACTOR_MIN || fo_factor[i]>IN_FO_FACTOR_MAX) {
      sprintf(MD->mw->messages,"%s %.4f - %.4f",
	      app_resources.err_input,IN_FO_FACTOR_MIN,IN_FO_FACTOR_MAX);
      warn_popup(ev_popup,MD->mw->messages);
      return;
      }
      if (sup_amp[i]<=IN_SUP_AMP_MIN || sup_amp[i]>IN_SUP_AMP_MAX) {
	 sprintf(MD->mw->messages,"%s %.4f - %.4f",
		 app_resources.err_input,IN_SUP_AMP_MIN,IN_SUP_AMP_MAX);
	 warn_popup(ev_popup,MD->mw->messages);
	 return;
      }
   }
   
   XtPopdown((Widget) client_data);
   XFlush(XtDisplay(w));
   
   ev_return(fo_time,fo_factor,sup_amp);
}

void envvol_dialog (void (*envvol_callback)(float fo_time[MAX_CHANNELS],
					    float fo_factor[MAX_CHANNELS],
					    float sup_amp[MAX_CHANNELS]),
		    Wave_Data *wd)
{
   Boolean set_sense;
   
   if (wd->channels==1) set_sense=False;
   else set_sense=True;
   
   XtVaSetValues(ev2_text,XtNsensitive,set_sense,NULL);
   XtVaSetValues(ev4_text,XtNsensitive,set_sense,NULL);
   XtVaSetValues(ev6_text,XtNsensitive,set_sense,NULL);
   
   ev_return=envvol_callback;
   popup_centered(ev_popup);
}

void create_envvol_dialog (Widget w)
{
   Widget form,label,button;
   static XtAccelerators accel;
   int i;
   
   accel = XtParseAcceleratorTable("#override\n\
                                    <Key>Return: set() notify() unset()\n");

   ev_popup=XtVaCreatePopupShell("ev_shell",transientShellWidgetClass,
				 w,NULL);

   form  = MW ("ev_main_form",formWidgetClass,ev_popup,
		XtNborderWidth,0,
   		NULL);

   /* copy default values to strings */
   for (i=0;i<2;i++) {
      sprintf(ev_fo_time_str[i],"%.4f",DESCENT_TIME);
      sprintf(ev_fo_factor_str[i],"%.4f",DESCENT_FACTOR);
      sprintf(ev_sup_amp_str[i],"%.4f",MIN_AMP);
   }
   
   label = MW ("ev_fo-time_label",labelWidgetClass,form,
	       XtNborderWidth,0,NULL);
   
   ev1_text = MW ("ev_fo-time1_text",asciiTextWidgetClass,form,
		  XtNfromVert,label,
		  XtNeditType, XawtextEdit,
		  XtNwrap, XawtextWrapNever,
		  XtNresize, XawtextResizeWidth,
		  XtNuseStringInPlace, True,
		  XtNlength,MAX_NUMLENGTH,XtNstring,&ev_fo_time_str[0],
		  XtNbottom, XtChainTop,XtNtop, XtChainTop,
		  XtNleft,XawChainLeft,XtNright,XawChainLeft,
		  NULL);

   ev2_text = MW ("ev_fo-time2_text",asciiTextWidgetClass,form,
		  XtNfromVert,label,XtNfromHoriz,ev1_text,
		  XtNeditType, XawtextEdit,
		  XtNwrap, XawtextWrapNever,
		  XtNresize, XawtextResizeWidth,
		  XtNuseStringInPlace, True,
		  XtNlength,MAX_NUMLENGTH,XtNstring,&ev_fo_time_str[1],
		  XtNbottom, XtChainTop,XtNtop, XtChainTop,
		  XtNleft,XawChainLeft,XtNright,XawChainLeft,
		  NULL);

   label = MW ("ev_fo-value_label",labelWidgetClass,form,
	       XtNfromVert,ev1_text,
	       XtNborderWidth,0,NULL);
   
   ev3_text = MW ("ev_fo-value1_text",asciiTextWidgetClass,form,
		  XtNfromVert,label,
		  XtNeditType, XawtextEdit,
		  XtNwrap, XawtextWrapNever,
		  XtNresize, XawtextResizeWidth,
		  XtNuseStringInPlace, True,
		  XtNlength,MAX_NUMLENGTH,XtNstring,&ev_fo_factor_str[0],
		  XtNbottom, XtChainTop,XtNtop, XtChainTop,
		  XtNleft,XawChainLeft,XtNright,XawChainLeft,
		  NULL);

   ev4_text = MW ("ev_fo-value2_text",asciiTextWidgetClass,form,
		  XtNfromVert,label,XtNfromHoriz,ev3_text,
		  XtNeditType, XawtextEdit,
		  XtNwrap, XawtextWrapNever,
		  XtNresize, XawtextResizeWidth,
		  XtNuseStringInPlace, True,
		  XtNlength,MAX_NUMLENGTH,XtNstring,&ev_fo_factor_str[1],
		  XtNbottom, XtChainTop,XtNtop, XtChainTop,
		  XtNleft,XawChainLeft,XtNright,XawChainLeft,
		  NULL);

   label = MW ("ev_sup_label",labelWidgetClass,form,
	       XtNfromVert,ev3_text,
	       XtNborderWidth,0,NULL);
   
   ev5_text = MW ("ev_sup1_text",asciiTextWidgetClass,form,
		  XtNfromVert,label,
		  XtNeditType, XawtextEdit,
		  XtNwrap, XawtextWrapNever,
		  XtNresize, XawtextResizeWidth,
		  XtNuseStringInPlace, True,
		  XtNlength,MAX_NUMLENGTH,XtNstring,&ev_sup_amp_str[0],
		  XtNbottom, XtChainTop,XtNtop, XtChainTop,
		  XtNleft,XawChainLeft,XtNright,XawChainLeft,
		  NULL);

   ev6_text = MW ("ev_sup2_text",asciiTextWidgetClass,form,
		  XtNfromVert,label,XtNfromHoriz,ev5_text,
		  XtNeditType, XawtextEdit,
		  XtNwrap, XawtextWrapNever,
		  XtNresize, XawtextResizeWidth,
		  XtNuseStringInPlace, True,
		  XtNlength,MAX_NUMLENGTH,XtNstring,&ev_sup_amp_str[1],
		  XtNbottom, XtChainTop,XtNtop, XtChainTop,
		  XtNleft,XawChainLeft,XtNright,XawChainLeft,
		  NULL);

   button = MW ("ev_ok_btn",commandWidgetClass,form,
		XtNfromVert,ev6_text,
		XtNaccelerators, accel,
    		XtNtop,XawChainTop,XtNbottom,XawChainTop,
    		XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);

   XtAddCallback (button, XtNcallback, ev_ok_CB, (XtPointer) ev_popup);    

   button = MW ("ev_ca_btn",commandWidgetClass,form,
		XtNfromVert,ev6_text,XtNfromHoriz,button,		
    		XtNtop,XawChainTop,XtNbottom,XawChainTop,
    		XtNleft,XawChainRight,XtNright,XawChainRight,NULL);

   XtAddCallback (button, XtNcallback, ca_CB, (XtPointer) ev_popup);    

}

void ec_ok_CB(Widget w, XtPointer client_data, XtPointer call_data)
{
   float fo_time[MAX_CHANNELS];
   float fo_factor[MAX_CHANNELS];
   int i;
   
   for (i=0;i<MD->wd->channels;i++) {
      fo_time[i]=atof(ec_time_str[i]);
      fo_factor[i]=atof(ec_fo_str[i]);
   
      if (fo_time[i]<=EC_TIME_MIN || fo_time[i]>EC_TIME_MAX) {
	 sprintf(MD->mw->messages,"%s %.4f - %.4f",
		 app_resources.err_input,EC_TIME_MIN,EC_TIME_MAX);
	 warn_popup(ec_popup,MD->mw->messages);
	 return;
      }
      if (fo_factor[i]<=EC_FO_MIN || fo_factor[i]>EC_FO_MAX) {
      sprintf(MD->mw->messages,"%s %.4f - %.4f",
	      app_resources.err_input,EC_FO_MIN,EC_FO_MAX);
	 warn_popup(ec_popup,MD->mw->messages);
	 return;
      }
   }
   
   XtPopdown((Widget) client_data);
   XFlush(XtDisplay(w));
   
   ec_return(fo_time,fo_factor);
}

void echo_dialog (void (*echo_callback)(float fo_time[MAX_CHANNELS],
					float fo_factor[MAX_CHANNELS]),
		  Wave_Data *wd)
{
   Boolean set_sense;
   
   if (wd->channels==1) set_sense=False;
   else set_sense=True;
   
   XtVaSetValues(ec2_text,XtNsensitive,set_sense,NULL);
   XtVaSetValues(ec4_text,XtNsensitive,set_sense,NULL);
   
   ec_return=echo_callback;
   popup_centered(ec_popup);
}

void create_echo_dialog (Widget w)
{
   Widget form,label,button;
   static XtAccelerators accel;
   int i;
   
   accel = XtParseAcceleratorTable("#override\n\
                                    <Key>Return: set() notify() unset()\n");

   ec_popup=XtVaCreatePopupShell("ec_shell",transientShellWidgetClass,
				 w,NULL);

   form  = MW ("ec_main_form",formWidgetClass,ec_popup,
	       XtNborderWidth,0,
	       NULL);

   /* copy default values to strings */
   for (i=0;i<2;i++) {
      sprintf(ec_time_str[i],"%.4f",ECHO_TIME);
      sprintf(ec_fo_str[i],"%.4f",ECHO_FO);
   }
   
   label = MW ("ec_time_label",labelWidgetClass,form,
	       XtNborderWidth,0,NULL);
   
   ec1_text = MW ("ec_time1_text",asciiTextWidgetClass,form,
		  XtNfromVert,label,
		  XtNeditType, XawtextEdit,
		  XtNwrap, XawtextWrapNever,
		  XtNresize, XawtextResizeWidth,
		  XtNuseStringInPlace, True,
		  XtNlength,MAX_NUMLENGTH,XtNstring,&ec_time_str[0],
		  XtNbottom, XtChainTop,XtNtop, XtChainTop,
		  XtNleft,XawChainLeft,XtNright,XawChainLeft,
		  NULL);

   ec2_text = MW ("ec_time2_text",asciiTextWidgetClass,form,
		  XtNfromVert,label,XtNfromHoriz,ec1_text,
		  XtNeditType, XawtextEdit,
		  XtNwrap, XawtextWrapNever,
		  XtNresize, XawtextResizeWidth,
		  XtNuseStringInPlace, True,
		  XtNlength,MAX_NUMLENGTH,XtNstring,&ec_time_str[1],
		  XtNbottom, XtChainTop,XtNtop, XtChainTop,
		  XtNleft,XawChainLeft,XtNright,XawChainLeft,
		  NULL);

   label = MW ("ec_fo_label",labelWidgetClass,form,
	       XtNfromVert,ec1_text,
	       XtNborderWidth,0,NULL);
   
   ec3_text = MW ("ec_fo1_text",asciiTextWidgetClass,form,
		  XtNfromVert,label,
		  XtNeditType, XawtextEdit,
		  XtNwrap, XawtextWrapNever,
		  XtNresize, XawtextResizeWidth,
		  XtNuseStringInPlace, True,
		  XtNlength,MAX_NUMLENGTH,XtNstring,&ec_fo_str[0],
		  XtNbottom, XtChainTop,XtNtop, XtChainTop,
		  XtNleft,XawChainLeft,XtNright,XawChainLeft,
		  NULL);

   ec4_text = MW ("ec_fo2_text",asciiTextWidgetClass,form,
		  XtNfromVert,label,XtNfromHoriz,ec3_text,
		  XtNeditType, XawtextEdit,
		  XtNwrap, XawtextWrapNever,
		  XtNresize, XawtextResizeWidth,
		  XtNuseStringInPlace, True,
		  XtNlength,MAX_NUMLENGTH,XtNstring,&ec_fo_str[1],
		  XtNbottom, XtChainTop,XtNtop, XtChainTop,
		  XtNleft,XawChainLeft,XtNright,XawChainLeft,
		  NULL);

   button = MW ("ec_ok_btn",commandWidgetClass,form,
		XtNfromVert,ec4_text,
		XtNaccelerators, accel,
    		XtNtop,XawChainTop,XtNbottom,XawChainTop,
    		XtNleft,XawChainLeft,XtNright,XawChainLeft,NULL);

   XtAddCallback (button, XtNcallback, ec_ok_CB, (XtPointer) ec_popup);    

   button = MW ("ec_ca_btn",commandWidgetClass,form,
		XtNfromVert,ec4_text,XtNfromHoriz,button,		
    		XtNtop,XawChainTop,XtNbottom,XawChainTop,
    		XtNleft,XawChainRight,XtNright,XawChainRight,NULL);

   XtAddCallback (button, XtNcallback, ca_CB, (XtPointer) ec_popup);    

}

