/*
 * QCameraPathPlayer.cpp
 * $Id: 
 *
 * Copyright (C) 2000, 2001 Thomas Woerner, Michael Meissner, Markus Janich
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */




// Qt
///////
#include <qfile.h>
#include <qfiledialog.h>
#include <qpushbutton.h>
#include <qfiledialog.h>
#include <qmenubar.h>
#include <qpopupmenu.h>
#include <qlayout.h>
#include <qlabel.h>

// QGLViewer
//////////////
#include "QCameraPathPlayer.h"
#include "CCameraPathInterpolator.h"
#include "QGLViewerIO.h"

#include "pictures/begin.xpm"
#include "pictures/bwd.xpm"
#include "pictures/stop.xpm"
#include "pictures/play.xpm"
#include "pictures/pause.xpm"
#include "pictures/fwd.xpm"
#include "pictures/end.xpm"
#include "pictures/loop.xpm"


#define STOP   0
#define PLAY   1
#define PAUSE  2


QCameraPathPlayer::QCameraPathPlayer(QGLViewer *pViewer,
				     QWidget* parent, const char* name, 
                                     bool modal, WFlags fl)
  : QDialog(parent, name, modal, fl),
    m_pViewer(pViewer)
/***********************************************************/
{
   QGridLayout *pLayout;
   QMenuBar    *pMenuBar;
   QLabel      *pLabel;

   resize(250,100);

   if (!name)
      setName("Viewer");
   setCaption(tr("QCameraPathPlayer"));
   setSizeGripEnabled(TRUE);
   setSizePolicy(QSizePolicy((QSizePolicy::SizeType)7,
                             (QSizePolicy::SizeType)7,
                             sizePolicy().hasHeightForWidth()));

   pLayout = new QGridLayout(this, 3, 8, 10, 10);
   pLayout->setSpacing(5);
   pLayout->setMargin(10);

   pMenuBar = new QMenuBar(this, "menu");
   Q_CHECK_PTR(pMenuBar);

   QPopupMenu *menu = new QPopupMenu();
   Q_CHECK_PTR(menu);
   menu->insertItem("Load camera path", this, SLOT(sltLoad()), CTRL+Key_L);
   pMenuBar->insertItem("&File", menu);

   pLayout->setMenuBar(pMenuBar);

   QPushButton *button;
    
   button = new QPushButton(this, "beginButton");
   button->setPixmap(QPixmap((const char **) begin_xpm));
   connect(button, SIGNAL(clicked()), this, SLOT(sltBegin()));
   pLayout->addWidget(button, 0, 0);

   button = new QPushButton(this, "bwdButton");
   button->setPixmap(QPixmap((const char **) bwd_xpm));
   connect(button, SIGNAL(clicked()), this, SLOT(sltBwd()));
   pLayout->addWidget(button, 0, 1);

   button = new QPushButton(this, "stopButton");
   button->setPixmap(QPixmap((const char **) stop_xpm));
   connect(button, SIGNAL(clicked()), this, SLOT(sltStop()));
   pLayout->addWidget(button, 0, 2);

   m_pPlayPauseButton = new QPushButton(this, "play_pauseButton");
   m_pPlayPauseButton->setPixmap(QPixmap((const char **) play_xpm));
   connect(m_pPlayPauseButton, SIGNAL(clicked()), this, SLOT(sltPlayPause()));
   pLayout->addWidget(m_pPlayPauseButton, 0, 3);

   button = new QPushButton(this, "fwdButton");
   button->setPixmap(QPixmap((const char **) fwd_xpm));
   connect(button, SIGNAL(clicked()), this, SLOT(sltFwd()));
   pLayout->addWidget(button, 0, 4);

   button = new QPushButton(this, "endButton");
   button->setPixmap(QPixmap((const char **) end_xpm));
   connect(button, SIGNAL(clicked()), this, SLOT(sltEnd()));
   pLayout->addWidget(button, 0, 5);

   pLabel = new QLabel(this, " ");
   pLayout->addWidget(pLabel, 0, 6);

   m_pLoopButton = new QPushButton(this, "loopButton");
   m_pLoopButton->setPixmap(QPixmap((const char **) loop_xpm));
   m_pLoopButton->setToggleButton(true);
   pLayout->addWidget(m_pLoopButton, 0, 7);

   m_pStatusLabel = new QLabel(this, "statusLabel");
   pLayout->addMultiCellWidget(m_pStatusLabel, 1, 1, 0, 6);

   m_pFrameSlider = new QSlider(1, 1, 1, 0, QSlider::Horizontal,
                                this, "frameSlider");
   m_pFrameSlider->setTickmarks(QSlider::Below);
   m_pFrameSlider->setTickInterval(1);
   connect(m_pFrameSlider, SIGNAL(valueChanged(int)),
	   this, SLOT(sltSliderChanged(int)));
   pLayout->addMultiCellWidget(m_pFrameSlider, 2, 2, 0, 5);

   m_pFpsCombo = new QComboBox(FALSE, this, "fpsCombo");
   m_pFpsCombo->insertItem(QString(" 5 fps"));
   m_pFpsCombo->insertItem(QString("10 fps"));
   m_pFpsCombo->insertItem(QString("15 fps"));
   m_pFpsCombo->insertItem(QString("20 fps"));
   m_pFpsCombo->insertItem(QString("25 fps"));
   m_pFpsCombo->insertItem(QString("30 fps"));
   connect(m_pFpsCombo, SIGNAL(activated(int)), this, 
           SLOT(sltFpsChanged(int)));
   m_pFpsCombo->setCurrentItem(3);
   m_nFps = 1000 / ((3+1) * 5);
   pLayout->addMultiCellWidget(m_pFpsCombo, 2, 2, 7, 8);

   m_pTimer = new QTimer(this);
   connect(m_pTimer, SIGNAL(timeout()), this, SLOT(sltTimer()));

   m_nStatus = STOP;

   setAcceptDrops(TRUE);
}



void QCameraPathPlayer::sltSliderChanged(int i) 
/***********************************************************/
{
   if (m_nStatus == PLAY)
     if (i > 0 && i <= m_CameraList.getNumObjects())
       m_pViewer->setCamera(m_CameraList[i-1]);    

   QString str, str2;
   if (m_nStatus == STOP)
      str = tr("Stopped");
   if (i == 1)
      str = tr("Begin");
   if (i == m_CameraList.getNumObjects())
      str = tr("End");
   if (m_nStatus == PLAY)
      str = tr("Playing");

   str2.sprintf(" (%d / %d)", m_pFrameSlider->value(), m_CameraList.getNumObjects());
   m_pStatusLabel->setText(str+str2);

   emit(sigPositionChanged(m_pStatusLabel->text()));
}



void QCameraPathPlayer::sltFpsChanged(int i) 
/***********************************************************/
{
   m_nFps = 1000 / ((i+1) * 5);
   if (m_nStatus == PLAY)
      m_pTimer->changeInterval(m_nFps);
}



void QCameraPathPlayer::sltTimer() 
/***********************************************************/
{
   if (m_nStatus == PLAY) {
      if (m_pFrameSlider->value() < m_CameraList.getNumObjects())
         setSlider(m_pFrameSlider->value() + 1);
      else {
         if (m_pLoopButton->isOn())
            setSlider(1);
         else {
            sltBegin();
	    m_pViewer->enableMouseEvents(true);

	    emit(sigAnimationStopped());
	 }
      }
   }
}



void QCameraPathPlayer::sltSetCameraPath(const CList<CCamera> &cCameraPath)
/***********************************************************/
{
  m_CameraList.clear(1);
  m_CameraList = *cCameraPath.getFullDuplicate();
  if (m_CameraList.getNumObjects() > 0) {
    m_pFrameSlider->setMaxValue(m_CameraList.getNumObjects());
    if (m_CameraList.getNumObjects() > 100)
      m_pFrameSlider->setTickInterval(5);
    if (m_CameraList.getNumObjects() > 1000)
      m_pFrameSlider->setTickInterval(25);
    sltBegin();
  }
  sigRedraw();
}


void QCameraPathPlayer::sltLoad()
/***********************************************************/
{
   CList<CCamera> tmpCameraList;

   // load XML file to keys
   QString fname = QFileDialog::getOpenFileName(QString::null, "*.xml", this);
   if (QFile::exists(fname)) {
      if (!fname.isEmpty()) {
         QFile file(fname);
	 
         if (!QGLViewerIO::read(file, tmpCameraList)) {
            cout << "Can not open file" << endl;
            return;
         } 
         else {
	   sltSetCameraPath(tmpCameraList);
         }
      }
   }
}


void QCameraPathPlayer::sltStop()
/***********************************************************/
{
   m_pTimer->stop();
   m_pViewer->enableMouseEvents(true);
   m_nStatus = STOP;
   setSlider(1);
   m_pPlayPauseButton->setPixmap(QPixmap(play_xpm));
   emit(sigAnimationStopped());
}


void QCameraPathPlayer::sltPlayPause()
/***********************************************************/
{
   if (m_nStatus == PLAY) {
      m_nStatus = PAUSE;
      m_pTimer->stop();
      m_pViewer->enableMouseEvents(true);
      m_pStatusLabel->setText(tr("Pause"));
      emit(sigPositionChanged(tr("Pause")));
      m_pPlayPauseButton->setPixmap(QPixmap(play_xpm));
      emit(sigAnimationStopped());
   }
   else {
      m_nStatus = PLAY;
      m_pViewer->enableMouseEvents(false);
      m_pTimer->start(m_nFps, FALSE);
      m_pPlayPauseButton->setPixmap(QPixmap(pause_xpm));
      emit(sigAnimationStarted());
   }
}


void QCameraPathPlayer::sltBegin()
/***********************************************************/
{
   m_nStatus = STOP;
   m_pTimer->stop();
   setSlider(1);
   sltSliderChanged(1);
   m_pPlayPauseButton->setPixmap(QPixmap(play_xpm));
}



void QCameraPathPlayer::sltBwd()
/***********************************************************/
{
   m_nStatus = STOP;
   m_pTimer->stop();
   m_pStatusLabel->setText(tr(""));
   if (m_pFrameSlider->value() > 1)
      setSlider(m_pFrameSlider->value() - 1);
}



void QCameraPathPlayer::sltFwd()
/***********************************************************/
{
   m_nStatus = STOP;
   m_pTimer->stop();
   m_pStatusLabel->setText(tr(""));
   if (m_pFrameSlider->value() < m_CameraList.getNumObjects())
      setSlider(m_pFrameSlider->value() + 1);
}



void QCameraPathPlayer::sltEnd()
/***********************************************************/
{
   m_nStatus = STOP;
   m_pTimer->stop();
   setSlider(m_CameraList.getNumObjects());
}



// Function  : dragEnterEvent
// Parameters: QDragEnterEvent *event
// Purpose   : handles QDragEnterEvents.
// Comments  : See docu for details.
void QCameraPathPlayer::dragEnterEvent(QDragEnterEvent *event)
/***********************************************************/
{
   if (QCameraPathDrag::canDecode(event))
      event->accept();
}



// Function  : dropEvent
// Parameters: QDropEvent *event
// Purpose   : handles QDropEvents.
// Comments  : See docu for details.
void QCameraPathPlayer::dropEvent(QDropEvent *event)
/***********************************************************/
{
   CList<CCamera> tmpCameraList;

   if (QCameraPathDrag::decode(event, tmpCameraList)) {
     m_CameraList.clear(1);
     m_CameraList = tmpCameraList;
     m_pFrameSlider->setMaxValue(m_CameraList.getNumObjects());
     if (m_CameraList.getNumObjects() > 100)
       m_pFrameSlider->setTickInterval(5);
     if (m_CameraList.getNumObjects() > 1000)
       m_pFrameSlider->setTickInterval(25);
     sltBegin();
   }
}
