 /*
 * QCameraKeyPathEditorImpl.h  
 * $Id: 
 *
 * Copyright (C) 2001 Alexander Buck, Markus Janich
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * As a special exception to the GPL, the QGLViewer authors (Markus
 * Janich, Michael Meissner, Richard Guenther, Alexander Buck and Thomas
 * Woerner) give permission to link this program with Qt (non-)commercial
 * edition, and distribute the resulting executable, without including
 * the source code for the Qt (non-)commercial edition in the source
 * distribution.
 *
 */

//  Description : Class QCameraKeyPathEditorImpl
//  Purpose     : Provides a tool to handle camerapaths
//

#ifndef QCAMERAKEYPATHEDITORIMPL_H
#define QCAMERAKEYPATHEDITORIMPL_H


// Qt
///////
#include <qstring.h>
#include <qmenubar.h>
#include <qvalidator.h>
#include <qgl.h>
#include <qdom.h>

// QGLViewer
//////////////
#include "CCamera.h"
#include "CP3D.h"
#include "CV3D.h"
#include "QGLViewer.h"
#include "QCameraDrag.h"
#include "QSceneTreeNode.h"


// Own
///////////
#include "CList.h"
#include "CCameraKeyPathAttributes.h"
#include "QCameraKeyPathEditor.h"
#include "CCameraKeyPathPoint.h"
#include "QMyListBox.h"
#include "QCameraKeyPathDrag.h"
#include "QCameraKeyPathDropSite.h"
#include "QIconOptions.h"

// Forwar declarations
////////////////////////
class QVBoxLayout; 
class QHBoxLayout; 
class QGridLayout; 
class QComboBox;
class QLabel;
class QLineEdit;
class QPushButton;
class QSlider;
class QCheckBox;
 
/** Tool to handle camerapaths
  *
  * @author Alexander Buck
  */

class QCameraKeyPathEditorImpl : public QCameraKeyPathEditor, public QSceneTreeDrawable
{
    Q_OBJECT

 public:

    //////////////////
    // CONSTRUCTORS //
    //////////////////

    /** Constructor using given QGLExaminerViewer. */
    QCameraKeyPathEditorImpl( QGLViewer *viewer, QWidget* parent = 0, const char* name = 0, WFlags fl = 0 );
    

    /** Default destructor. */
    ~QCameraKeyPathEditorImpl();
    
    ////////////////////
    // PUBLIC METHODS //
    ////////////////////

    /** Update bounding box. */
    void updateBoundingBox();

    /** Draws a path. */
    virtual void draw(); 
      
    
 signals:
    /////////////
    // SIGNALS //
    /////////////

    /** This signal is emitted when the scene should be redrawed. */
    void sigRedrawGL();

    /** Is emitted to submit the camerapath. */
    void sigCameraKeyPath(const CList<CCameraKeyPathPoint> &);

 public slots:
    //////////////////
    // PUBLIC SLOTS //
    //////////////////   

    /** Sets Cameralist to given List.
      * This slot is calles for when a camerapath is 
      * dropped on the tool or on the CameraKeyPathDropSite.
      */
    void sltSetCameraPath(const CList<CCameraKeyPathPoint> &);

    
 private slots:
    ///////////////////
    // PRIVATE SLOTS //
    ///////////////////

    /** Is called when the delete button is clicked. */   
    void sltDeleteCamera();

    /** Is called when the replace button is clicked. */
    void sltReplaceCamera();

    /** Saves the current iconsize in m_rfIconsize. */
    void sltSetIconSize(double);

    /** Saves the current frustumsize in m_rfFrustumsize. */
    void sltSetFrustumSize(double);

    /** Shows options. */
    void sltShowOptions();

    /** Save path in XML-file. */
    void sltSaveInXML();

    /** Holds the current filename for saving. */
    void sltSetSaveName(const QString &);

    /** Loads XML-file. */
    void sltLoadFile();
    
    /** Changes values of current camera. */
    void sltChangeCamValue();

    /** Adds a new camera to the path. */ 
    void sltAddNewCamera();

    /** Checks all input fields for correct input. */
    void sltCheckValues();

    /** Marks highlighted cameras and redraws scene. */
    void sltUpdateHighlights(int);

    /** Clears old selection, selects item nIndex and redraws scene. */
    void sltClearHighlights(int nIndex); 

    /** Updates tool with given nIndex. */
    void sltUpdateTool(int nIndex); 

    /** Handles the attach/detach/ feature. */
    void sltAttachCamera();

    /** Shows "About" box. */
    void sltAbout();

    /** Moves selected items upwards. */
    void sltMoveUp();

    /** Moves selected items downwards. */
    void sltMoveDown();

    /** Updates name in given index. */
    void sltUpdateName(int);

    /** Emits the current path by sending a 'sigCameraPath(...)'-
      * signal and closes the window. */
    void sltOk() {
      sltApply();
      hide();
    };

    /** Emits the current path by sending a 'sigCameraPath(...)'-
      * signal but stays the window open. */
    void sltApply() {
      emit(sigCameraKeyPath(*m_pCameraKeyPath));
    };

 protected:
    ///////////////////////
    // PROTECTED METHODS //
    ///////////////////////

    /** Method to handle drag 'n drop. */
    void dragEnterEvent(QDragEnterEvent*); 
 
    /** Method to handle drag 'n drop. */
    void dragLeaveEvent(QDragLeaveEvent*);

    /** Method to handle drag 'n drop. */
    void dropEvent(QDropEvent*);  
          
private:

    /////////////////////
    // PRIVATE METHODS //
    /////////////////////

    /** Filters events. */
    bool eventFilter(QObject *o, QEvent *e);
    
    /** Adds given camera to path.
      *	This function is called when a camera is dropped on the tool.
      */
    void setCamera(const CCamera &); 
    
    /** Add current camera (m_myCamera) to path with given name. */
    void saveInList(QString qname); 
    
    /** Write given ttributes in proper fields. */
    void writeAttributes(CCameraKeyPathAttributes);

    /** Sets slider to given value.
      *	Also updates range of slider.
      */
    void setSlider(int);

    /** Write given camera values in proper fields. */
    void writeCameraText(CCamera *);
    
    /** Writes the names into the camera box. */
    void formatCamBox();

    /** Returns current EyePos. */
    CP3D getCurrentEyePos();

    /** Returns current RefPoint. */
    CP3D getCurrentRefPoint();

    /** Returns current ViewUp. */
    CV3D getCurrentViewUp();

    /** Returns current fovy. */
    int getCurrentFovy(); 

    /** Returns current frames. */
    int getCurrentFrames(); 

    /** Returns current tension. */
    float getCurrentTension(); 

    /** Returns current continuity. */
    float getCurrentContinuity(); 

    /** Returns current bias. */
    float getCurrentBias(); 

    /** Provides a menu bar, which is added to given layout. */
    void setMenuBar(QLayout *);

    /** Builds the OpenGl-List to draw an arrow.*/
    void makeArrowDispList();  

    /** Builds the OpenGl-List to draw the camera-key-path.*/
    void makePathDispList();

    /** Shows a Dialog with the given error.*/
    void showError(QString qError);  

    /** Returns pointer to the cameralist. */
    CList<CCameraKeyPathPoint>* getCamerapathPtr()
      {
	return m_pCameraKeyPath;
      }

    /** Updates current highlights. */
    void updateHighlights();

    /** Redraws the scene. */
    void redraw();

    
    /////////////////////
    // PRIVATE MEMBERS //
    /////////////////////

    /** The menu bar. */
    QMenuBar *m_pqMenuBar;                     

    /** Holds current given viewer. */
    QGLViewer *m_pMyViewer; 

    /** Current camera path point. */
    CCameraKeyPathPoint m_camPoint; 

    /** Current attributes. */
    CCameraKeyPathAttributes m_attributes;   

    /** Current camera. */
    CCamera m_myCamera;  

    /** Holds camera to detach view. */
    CCamera m_detachCamera;           

    /** Current path. */
    CList<CCameraKeyPathPoint> *m_pCameraKeyPath; 

    /** Saves current highlighted cameras. */
    CList<bool> m_highlightedCams;          

    /** The slider's current index. */
    int m_nSliderIndex;              

    /** Current iconsize. */
    float m_rfIconSize;           

    /** Current frustumsize. */
    float m_rfFrustumSize;              

    /** Validator for the EyePos. */
    QDoubleValidator *m_pDValEye;  

    /** Validator for the RefPoint. */
    QDoubleValidator *m_pDValRef;    

    /** Validator for the ViewUp. */
    QDoubleValidator *m_pDValUp;   

    /** Validator for the fovy. */
    QDoubleValidator *m_pDValVAngle;

    /** Validator for the iconsize. */
    QDoubleValidator *m_pDValIcon;   

    /** Validator for the frustumsize. */
    QDoubleValidator *m_pDValFrustum;   

    /** Validator for the framerate. */
    QIntValidator *m_pIValFrame;      

    /** Validator for the tension. */
    QDoubleValidator *m_pDValTension;

    /** Validator for the continuity. */
    QDoubleValidator *m_pDValCon;      

    /** Validator for the bias. */
    QDoubleValidator *m_pDValBias;

    /** Open-GL display list. */
    GLuint m_glPathDispList;

    /** Open-GL display list. */
    GLuint m_glArrowDispList;          

    /** Current name for saving . */
    QString m_qSaveName;           

    /** Clears selection if necessary. */
    bool m_fClearSelectedCams;          

    /** Attach cameras?. */
    bool m_fAttachCamera; 

    /** Values correct?. */
    bool m_fCorrectValues;

    /** Clear selection?. */
    bool m_fClearSelection;

    /** Update slider while deleting?. */
    bool m_fDeleteWithSelection;        

    /** Menu for options. */
    QIconOptions *m_pqOptions;
};

#endif // QCAMERAKEYPATHEDITORIMPL_H
