package PSP::SideMenu;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: SideMenu.pm,v 1.1 2000/11/23 23:36:19 muaddib Exp $

use strict;

=head1 NAME

B<PSP::SideMenu> - Element of a menu, and the menu itself.

=head1 SYNOPSIS

 #more later.

=head1 DESCRIPTION

=head2 "SPECIAL" FEATURES of ELEMENTS:

 1 - only display when the current node is a sibling.
 2 - only display siblings upto and including current node.
 3 - display children when element is current node.
 4 - display children when the current node is a sibling.

=cut

use Exporter;
@PSP::SideMenu::ISA = qw(Exporter);

use HTMLIO::Utils qw(html_tag);

# is there a better way to manage these style variables?
use vars qw(@STYLE_VARS @EXPORT_SUBS);
use vars qw(@ISA @EXPORT_OK %EXPORT_TAGS);
BEGIN {
  @STYLE_VARS = 
    qw($BODY_FGCOLOR
       $BODY_BGCOLOR
       $BODY_SIZE
       $BODY_FACE
       $HEADER_FGCOLOR
       $HEADER_BGCOLOR
       $HEADER_SIZE
       $HEADER_FACE
       $SELECTED_FGCOLOR
       $SELECTED_BGCOLOR
       $SELECTED_SIZE
       $SELECTED_FACE
       $ICON_DIR
       %TREE_IMAGE
       $USE_JAVASCRIPT);

  @EXPORT_SUBS = 
    qw(&tree_logic_js 
       &display_tree_js);
  # export the style variables.
  @EXPORT_OK = (@STYLE_VARS, @EXPORT_SUBS);
  $EXPORT_TAGS{all} = \@EXPORT_OK;
}

use vars @STYLE_VARS;

%TREE_IMAGE =
  (# xplrr-style
   blank	=> "blank.gif",
   doc		=> "doc.gif",
   folderclosed	=> "folderclosed.gif",
   folderopen	=> "folderopen.gif",
   lastnode	=> "lastnode.gif",
   link		=> "link.gif",
   mlastnode	=> "mlastnode.gif",
   mnode	=> "mnode.gif",
   node		=> "node.gif",
   plastnode	=> "plastnode.gif",
   pnode	=> "pnode.gif",
   vertline	=> "vertline.gif",
   # fxim-style
   blkarw       => "arw_blk.gif",
   redarw       => "arw_red.gif",
   dot          => "dot.gif"
  );

#to be included by fxim/pile-src/control.pl  ?
#
#use PSP::SideMenu qw(:all);
#BEGIN {
#  $BODY_FGCOLOR     = $FXIM_BODY_FGCOLOR;
#  $BODY_BGCOLOR     = $FXIM_BODY_BGCOLOR;
#  $BODY_SIZE        = $FXIM_BODY_SIZE;
#  $BODY_FACE        = $FXIM_BODY_FACE;
#  $PHEADER_FGCOLOR  = $FXIM_PHEADER_FGCOLOR;
#  $PHEADER_BGCOLOR  = $FXIM_PHEADER_BGCOLOR;
#  $PHEADER_SIZE     = $FXIM_PHEADER_SIZE;
#  $PHEADER_FACE     = $FXIM_PHEADER_FACE;
#  $SELECTED_FGCOLOR = $FXIM_MESSAGE_FGCOLOR;
#  $SELECTED_BGCOLOR = $FXIM_MESSAGE_BGCOLOR;
#  $SELECTED_SIZE    = $FXIM_MESSAGE_SIZE;
#  $SELECTED_FACE    = $FXIM_MESSAGE_FACE;
#  $ICON_DIR         = "/images/fx/fxim/$FXIM_IMG_SCHEME/";
#}

# here are some style defaults.
BEGIN {
  $BODY_FGCOLOR     or $BODY_FGCOLOR     = '#000000'; 
  $BODY_BGCOLOR     or $BODY_BGCOLOR     = '#ffffff';
  #$BODY_SIZE        or $BODY_SIZE        = 3;
  #$BODY_FACE        or $BODY_FACE        = 'Arial, Helvetica, sans-serif';

  $HEADER_FGCOLOR   or $HEADER_FGCOLOR   = $BODY_BGCOLOR;
  $HEADER_BGCOLOR   or $HEADER_BGCOLOR   = $BODY_FGCOLOR;
  #$HEADER_SIZE      or $HEADER_SIZE      = 3;
  #$HEADER_FACE      or $HEADER_FACE      = 'Arial, Helvetica, sans-serif';

  $SELECTED_FGCOLOR or $SELECTED_FGCOLOR = $BODY_FGCOLOR;
  $SELECTED_BGCOLOR or $SELECTED_BGCOLOR = '#eeeeee';
  #$SELECTED_SIZE    or $SELECTED_SIZE    = 3;
  #$SELECTED_FACE    or $SELECTED_FACE    = 'Arial, Helvetica, sans-serif';

  $ICON_DIR         or $ICON_DIR         = '/images/icons/';

  $USE_JAVASCRIPT = 1 unless defined $USE_JAVASCRIPT;
};

=head2 new

class
(PSP::SideMenu $this) new (string $class)

DESCRIPTION:

This public function creates a new SideMenu object, initializing
all instance vars.

=cut

sub new {
  my ($proto, $name, $href, $children, $special, $desc, $target) = @_;
  
  $children = [] unless $children;
  
  my $this = { 'name'     => $name, 
	       'desc'     => $desc, 
	       'target'   => $target, 
	       'href'     => $href,
	       'children' => $children,
	       'special'  => $special
	     };

  bless $this, ref($proto) || $proto;

  $this->{children} ||= [];

  return $this; 
}#new

=head2 mutator_accessor

private
instance
(value) mutator_accessor (Name[, Value])

DESCRIPTION:

Will optionally set the data member name to a passed in value.
Will return the value of this data member.

=cut

sub mutator_accessor {
  my ($this,$name,$value) = @_;
  defined $value and $this->{$name} = $value;
  return $this->{$name};
}#mutator_accessor

=head2 name

public
instance
(String value) name (string $value)

DESCRIPTION:

Will set the name to the passed name if passed.
Always returns either the value or passed value.

=head2 desc

public
instance
(String value) desc (string $value)

DESCRIPTION:

Will set the description to the passed value if passed.
Always returns either the value or passed value.

=head2 target

public
instance
(String value) target (string $value)

DESCRIPTION:

Will set the target of the link to the passed value if passed.
Always returns either the value or passed value.

=head2 href

public
instance
(String value) href (string $value)

DESCRIPTION:

Will set the href to the passed href if passed.
Always returns either the value or passed value.

=head2 children

public
instance
(SideMenu @children) children (SideMenu @children)

DESCRIPTION:

Will set the children to the passed children if passed.
Always returns either the array of children passed
or already implemented..

=head2 special

public
instance
(Num special) special (Num special)

DESCRIPTION:

Will set the special to num to the one passed. Returns
the passed or already set value in any case.

=cut

sub name     { shift->mutator_accessor('name',     @_) }
sub desc     { shift->mutator_accessor('desc',     @_) }
sub target   { shift->mutator_accessor('target',   @_) }
sub href     { shift->mutator_accessor('href',     @_) }
sub special  { shift->mutator_accessor('special',  @_) }
sub children { shift->mutator_accessor('children', @_) }

sub add_child {
  my ($this,$child) = @_;
  push @{$this->{children}}, $child;
}

=head2 menu_html

public
instance
(string) menu_html (SideMenu the_menu, String \where[])

DESCRIPTION:

Take the_menu passed and builds the appropriate table to display
in html.  

=cut

sub menu_html {
  my ($this, $where) = @_;

  my %table_attr = (valign=>"top", height=>17, colspan=>11,
		    cellpadding=>0, cellspacing=>0);
  $table_attr{bgcolor} = $BODY_BGCOLOR if $BODY_BGCOLOR;

  my %tr_attr = (align=>"center");

  my %td_attr = (valign=>"top", height=>17, colspan=>11);
  $td_attr{bgcolor} = $HEADER_BGCOLOR if $HEADER_BGCOLOR;

  my %font_attr;
  $font_attr{size}  = $HEADER_SIZE if $HEADER_SIZE;
  $font_attr{face}  = $HEADER_FACE if $HEADER_FACE;
  $font_attr{color} = $HEADER_FGCOLOR if $HEADER_FGCOLOR;

  my $return_string = 
    (html_tag('table', \%table_attr)."\n".
     (html_tag('tr', \%tr_attr)."\n".
      (html_tag('td', \%td_attr)."\n  ".
       (html_tag('font', \%font_attr).
	"<b>&nbsp;&nbsp;".$this->name()." Menu&nbsp;&nbsp;</b></font>")."\n".
       "  </td>").
      ("<td>&nbsp;</td>\n").
      "</tr>\n"));

  $return_string .= $this->branch_html( $where, 0 );

  $return_string .= "</table>\n";
  
  return $return_string;
}#menu_html

=head2 branch_html

private
instance
(string) branch_html (SideMenu menu_def, String \where[], $level)

DESCRIPTION:

Takes the side menu passed and starts building the appropriate html string
for the side menu.  This function will recursively call itself.

=cut

my $indent="";

sub branch_html {
  my ($this, $where, $level) = @_; 

  my $return_string = '';

  ($this and $where and $#$where >= $level) or return "";

  #First, cycle through this level's children until we end 
  #or find the named element that matches our current level.

#$indent .= " ";
  foreach my $child ( @{$this->children()} )
  {
#print STDERR $indent."here1: ".$child->name().": ".$child->special()."\n";
    #Check if we match our level of where.
    if ($child->name() eq $where->[$level])
    {
      #matched.  We need to put this string out,
      #and evaluate the rest of this branch.
      if ($level < $#$where)
      {
	#We know there are more wheres to go, so this
	#deserves the down black arrow.
        $return_string .= render_line($child, $level, 1);
      }
      elsif ($level == $#$where and 
	     ($child->special() == 3 or $child->special() == 4)) 
      {
	#Now that we are at or beyond the correct level,
	#traverse one level deeper.
	$return_string .= render_line($child, $level, 2);
	push @$where, "";
      }
      else
      {
	#We must be on the right level, give them the red arrow.
	$return_string .= render_line($child, $level, 2);
	last if $child->special() == 2;
	next;
      }
      #another where to go.
      $return_string .= $child->branch_html( $where, $level+1 );
    }
    elsif ($child->special() == 1)
    {
      #This means, if this element is special,
      #we only display it if we're on on the
      #last where, or the end of the branch.
      #We do this check by seeing if $level
      #equals the size of the where array.
      if($level == $#$where)
      {
        $return_string .= render_line($child, $level, 0);
      }
    }
    else
    {
      if ($child->special() == 4)
      {
	#Normal case.  We just put out the string.
	$return_string .= render_line($child, $level, 1);
	my @tmp = @$where; push @tmp,"";
	$return_string .= $child->branch_html(\@tmp,$level+1)
      }
      else
      {
	#Normal case.  We just put out the string.
	$return_string .= render_line($child, $level, 0);
      }
    }
  }
#$indent =~ s/.//;
  return $return_string;
}#branch_html

=head2 render_line

private
instance
(string) render_line (SideMenu the_menu, $level, $where_flag)

DESCRIPTION:

Takes the side menu passed and starts building the appropriate html line
for a table row.  We use the $level to determine how much horizontal
spacing we need.  We use the where flag to determine what kind of graphic
is needed at the begining of the line.

=cut

sub render_line {
  my ($this, $level, $where_flag) = @_;

  defined $level or return "";

  # We can have up to 11 cols,
  my $colspan = 10 - $level;

  # initialize the anchor tag.
  my $atag = "";
  if (my $href = $this->href()) {
    my %attrs = ( href=>$href );
    $this->target() and $attrs{target} = $this->target();
    $atag = html_tag('a',\%attrs,['href']);
  }

  # initialize the text tag.
  my $text = "";
  if (my $name = $this->name()) {

    my %font_attr;
    if ($where_flag == 2) {
      $font_attr{size}  = $SELECTED_SIZE if $SELECTED_SIZE;
      $font_attr{face}  = $SELECTED_FACE if $SELECTED_FACE;
      $font_attr{color} = $SELECTED_FGCOLOR if $SELECTED_FGCOLOR;
    } else {
      $font_attr{size}  = $BODY_SIZE if $BODY_SIZE;
      $font_attr{face}  = $BODY_FACE if $BODY_FACE;
      $font_attr{color} = $BODY_FGCOLOR if $BODY_FGCOLOR;
    }
    my $font = html_tag('font', \%font_attr);
    
    #  break it up into 13 char chunks and indent it.
    my @out = ("");
    my $sep_string = "<br>&nbsp;&nbsp;";
    $sep_string = "</a>$sep_string$atag" if $atag;
    for my $word (split /\s+/, $name) {
      if (length($out[-1]) and (length($out[-1])+length($word)) > 13) {
	push @out, $word;
      } else {
	$out[-1] and $out[-1] .= " ";
	$out[-1] .= $word;
      }
    }
    grep s/ /&nbsp;/g, @out;

    $text = $font.join($sep_string, @out)."</font>";
  }

  # initialize the image tag.
  my $icon_tag;
  if ($where_flag == 1) {
    #We put out the arrow down image
    $icon_tag = 
      html_tag('img', { src=>"$ICON_DIR/$TREE_IMAGE{blkarw}", border=>0,
			width=>8, height=>10, alt=>"black arrow right" });
  } elsif ($where_flag == 2) {
    #We put out the red arrow right image
    $icon_tag = 
      html_tag('img', { src=>"$ICON_DIR/$TREE_IMAGE{redarw}", border=>0,
			width=>8, height=>10, alt=>"red arrow down" });
  } else {
    #We put out the dot image.
    $icon_tag = 
      html_tag('img', { src=>"$ICON_DIR/$TREE_IMAGE{dot}", border=>0,
			width=>8, height=>8, alt=>"dot" });
  }

  # determine the spacer cell we will use to denote space at front of row.
  my $spacer = "";
  if ($level) {
    my %attrs = ( width=>"8%", height=>13 );
    if ($where_flag == 2 and $SELECTED_BGCOLOR) {
      $attrs{bgcolor} = $SELECTED_BGCOLOR;
    }
    $spacer = "  ".html_tag('td', \%attrs)."&nbsp;</td>\n";
  }

  # determine the attributes to first non-space table data tag.
  my %td_attrs = ( colspan=>$colspan, valign=>"top", height=>13 );
  if ($where_flag == 2 and $SELECTED_BGCOLOR) {
    $td_attrs{bgcolor} = $SELECTED_BGCOLOR;
  }

  my %table_attr = ( border=>0, cellspacing=>0, cellpadding=>0 );
  my %td_attr    = ( align=>"right", valign=>"top" );

  #So that we can support some nice image/text formatting:
  my $return_string = 
    ("<tr>\n".
     # display empty spaces at front of row.
     ($level ? ($spacer x $level) : "").
     # start opening table data.
     html_tag('td', \%td_attrs)."\n".
     # format the icon and text inside another table.. ?
     (html_tag('table', \%table_attr)."\n".
      (html_tag('tr',{}).
       # the possibly hyperlinked icon goes in its own cell.
       (html_tag('td', \%td_attr).
	($atag ? $atag : "").$icon_tag.($atag ? $atag : "").
	"</td>\n").
       # next comes the possibly hyperlinked display text.
       ("<td>".
	($atag ? $atag : "").$text.($atag ? "</a>" : "").
	"</td>\n").
       "</tr>\n").
      "</table>\n").
     "</td>\n</tr>\n");

  return $return_string;
}#render_line

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<AtomicData>, L<HTMLIO>, L<Field>.

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
