package PSP::Parser::Control;

# Copyright (c) 2000, FundsXpress Financial Network, Inc.
# This library is free software released under the GNU Lesser General
# Public License, Version 2.1.  Please read the important licensing and
# disclaimer information included below.

# $Id: Control.pm,v 1.8 2001/01/23 05:41:01 muaddib Exp $

use strict;

=head1 NAME

 PSP::Parser::Control - Tag extension for control structures.

=head1 SYNOPSIS

=head1 DESCRIPTION

=cut

use Error qw(:try);
use PSP::Parser;
use PSP::Utils;

BEGIN {
  @PSP::Parser::Control::ISA = qw(PSP::Parser);
}

use vars qw(@handled @stacks @current @propagatable);
BEGIN {
  @handled = qw(script declare comment nop if else list while include block);
};

=head1 TAG HANDLERS

=head2 begin_pspscript

 [private] instance
 () begin_pspscript (string $tag, \%attributes, \@atrseq, string $orig);

DESCRIPTION:

See PSP specification.

psp:script allows text to pass through to output code after HTML-decoding.

=cut
  
sub begin_pspscript {
  my ($this, $tag, $attr, $attrseq, $orig) = @_;
  $this->debug_line($orig);
  $this->script_mode();
  $this->push_handlers( { 'psp:include' => \&begin_pspinclude },
			{ 'psp:script'  => \&end_pspscript    });
  			    
  $this->{is_cdata}++;
}

=head2 end_pspscript

 [private] instance
 () end_pspscript (string $tag);

DESCRIPTION:

Resets the tag handlers of the parser it had before the call to 
begin_pspscript().

=cut

sub end_pspscript {
  my ($this, $tag) = @_;
  $this->debug_line($tag);
  undef $this->{is_cdata};
  $this->pop_handlers();
  $this->script_mode(0);
  $this->code("\n");
}

=head2 begin_pspdeclare

 [private] instance
 () begin_pspdeclare (string $tag, \%attributes, \@atrseq, string $orig);

DESCRIPTION:

See PSP specification.

psp:declare allows text to pass through to file-scoped output code 
after HTML-decoding. 

=cut
  
sub begin_pspdeclare {
  my ($this, $tag, $attr, $attrseq, $orig) = @_;
  $this->debug_line($orig);
  $this->script_mode();
  $this->push_code_sub(\&code_pspdeclare);
  $this->push_handlers( { 'psp:include' => \&begin_pspinclude },
			{ 'psp:declare' => \&end_pspdeclare   });
  $this->{is_cdata}++;
}

=head2 code_pspdeclare

 [private] instance
 () code_pspdeclare (string $text);

DESCRIPTION:

This function decodes the text between handled tags.

=cut
  
sub code_pspdeclare {
  my ($this,$text) = @_;
  $this->debug_line($text);
  $this->decl($text);
}

=head2 end_pspdeclare

 [private] instance
 () end_pspdeclare (string $tag);

DESCRIPTION:

Resets the tag handlers of the parser it had before the call to 
begin_pspdeclare().

=cut

sub end_pspdeclare {
  my ($this, $tag) = @_;
  $this->debug_line($tag);
  undef $this->{is_cdata};
  $this->pop_handlers();
  $this->pop_code_sub();
  $this->script_mode(0);
  $this->code("\n");
}

=head2 begin_pspcomment

 [private] instance
 () begin_pspcomment (string $tag, \%attributes);

DESCRIPTION:

See PSP specification.

psp:comment allows all contained text to be ignored.

=cut

sub begin_pspcomment {
  my ($this, $tag, $attr) = @_;
  $this->push_text_sub(\&text_pspcomment);
  $this->push_handlers( { 'psp:include' => \&begin_pspinclude },
			{ 'psp:comment' => \&end_pspcomment   });
  $this->{is_cdata}++;
}

=head2 text_pspcomment

 [private] instance
 () text_pspcomment ();

DESCRIPTION:

See PSP specification.

Ignore all text within a psp:comment element.

=cut

sub text_pspcomment {
  # do not do anything with commented text.
}

=head2 end_pspcomment

 [private] instance
 () end_pspcomment (string $tag)

DESCRIPTION:

Resets the tag handlers of the parser it had before the call to 
begin_pspcomment().

=cut

sub end_pspcomment {
  my ($this, $orig_text) = @_;
  undef $this->{is_cdata};
  $this->pop_handlers();
  $this->pop_text_sub();
}

=head2 begin_pspnop

 [private] instance
 () begin_pspnop (string $tag, \%attributes);

DESCRIPTION:

See PSP specification.

psp:nop does nothing.

=cut

sub begin_pspnop {
  my ($this, $tag, $attr) = @_;
}

=head2 end_pspnop

 [private] instance
 () end_pspnop (string $tag)

DESCRIPTION:

/psp:nop does nothing.

=cut

sub end_pspnop {
  my ($this, $orig_txt) = @_;
}

=head1 IF, ELSIF, ELSE, WHILE, LIST

=head2 begin_pspif

 [private] instance
 () begin_pspif (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspif {
  my ($this, $tag, $attr, $tagseq, $orig) = @_;
  $this->debug_line($orig);
  defined $attr->{test} or throw
    Error::Simple("<$tag> requires a TEST tag.");
  $this->debug_line($attr->{test});
  $this->code("if ($attr->{test})");
  $this->begin_pspblock("if");
}
sub end_pspif {
  my ($this, $orig_txt) = @_;
  return $this->end_pspblock("if");
}

=head2 begin_pspelsif

 [private] instance
 () begin_pspelsif (string $tag, \%attributes);

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspelsif {
  my ($this, $tag, $attr) = @_;
  defined $attr->{test} or throw
    Error::Simple("<$tag> requires a TEST attribute.");
  $this->debug_line($attr->{test});
  $this->prev_context() eq "else" and throw
    Error::Simple("<$tag> called after an ELSE context.");

#  this hack would have allowed a "psp:else" without a "psp:if" or "psp:elsif"
#  (($this->context() eq "if") or ($this->context() eq "elsif")) and
#    $this->end_pspblock($this->context());

  $this->{text_to_flush} =~ s/^\s+//s;
  $this->code("elsif ($attr->{test})");
  $this->begin_pspblock("elsif");
}
sub end_pspelsif {
  my ($this, $orig_txt) = @_;
  return $this->end_pspblock("elsif");
}

=head2 begin_pspelse

 [private] instance
 () begin_pspelse (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspelse {
  my ($this, $tag, $attr) = @_;
  defined $attr->{test} and
    return $this->begin_pspelsif($tag,$attr);
  $this->debug_line("");

  $this->prev_context() eq "else" and throw
    Error::Simple("<$tag> called in a <$tag> context.");

#   this hack allows a "psp:else" without a "/psp:if"
#  ($this->context() eq "if") and
#    $this->end_pspblock($this->context());

  $this->{text_to_flush} =~ s/^\s+//s;
  $this->code("else");
  $this->begin_pspblock("else");
}
sub end_pspelse {
  my ($this, $orig_txt) = @_;
  return $this->end_pspblock("else");
}

=head2 begin_psplist

 [private] instance
 () begin_psplist (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_psplist {
  my ($this, $tag, $attr) = @_;
  defined $attr->{list} or throw
    Error::Simple("<$tag> requires a LIST attribute.");
  if (defined $attr->{iterator}) {
    $this->code("for my $attr->{iterator} ($attr->{list})");
  } else {
    $this->code("for ($attr->{list})");
  }
  $this->begin_pspblock("list");
}
sub end_psplist {
  my ($this, $orig_txt) = @_;
  return $this->end_pspblock("list");
}

=head2 begin_pspwhile

 [private] instance
 () begin_pspwhile (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

=cut

sub begin_pspwhile {
  my ($this, $tag, $attr) = @_;
  $attr->{test} or throw
    Error::Simple("<$tag> requires a TEST attribute.");
  $this->code("while ($attr->{test})");
  $this->begin_pspblock("while");
}
sub end_pspwhile {
  my ($this, $orig_txt) = @_;
  return $this->end_pspblock("while");
}

=head2 begin_pspinclude

 [private] instance
 () begin_pspinclude (string $tag, \%attributes)

DESCRIPTION:

See PSP specification.

This function will start a new parser of its own and hand the specified
file off to it.  The file will be searched for in the following paths:

 1) current directory of parent parser.
 2) directories specified on command line via -I argument.

=cut

sub begin_pspinclude {
  my ($this, $tag, $attr, $tagseq, $orig) = @_;
  $this->debug_line($orig);

  #validate input.
  defined(my $src = $attr->{src}) or throw
    Error::Simple("<$tag> requires a SRC attribute.");

  return $this->include_file($src);
}
sub end_pspinclude {
  my ($this, $orig_txt) = @_;
}

=head2 begin_pspblock

 [private] instance
 () begin_pspblock  (string $tag)

DESCRIPTION:

A general purpose block beginner.

=cut

sub begin_pspblock {
  my ($this, $tag) = @_;
  $this->debug_line($tag);
  $this->code("{ # $tag\n");
  $this->code_add_indent("  ");
  $this->push_context($tag);
}

=head2 end_pspblock

 [private] instance
 () end_pspblock  (string $tag)

DESCRIPTION:

A general purpose block ender.

=cut

sub end_pspblock {
  my ($this, $tag) = @_;
  $this->debug_line($tag);
  $this->pop_context() or throw
    Error::Simple("Stack underflow. (</$tag> called without <$tag>?");
  $this->code_del_indent("  ");
  $this->code("} # $tag\n");
}

1;
__END__

=head1 BUGS

No known bugs, but this does not mean no bugs exist.

=head1 SEE ALSO

L<HTML::Parser>, L<HTMLIO::Utils>

=head1 COPYRIGHT

 PSP - Perl Server Pages
 Copyright (c) 2000, FundsXpress Financial Network, Inc.

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2 of the License, or (at your option) any later version.

 BECAUSE THIS LIBRARY IS LICENSED FREE OF CHARGE, THIS LIBRARY IS
 BEING PROVIDED "AS IS WITH ALL FAULTS," WITHOUT ANY WARRANTIES
 OF ANY KIND, EITHER EXPRESS OR IMPLIED, INCLUDING, WITHOUT
 LIMITATION, ANY IMPLIED WARRANTIES OF TITLE, NONINFRINGEMENT,
 MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE, AND THE
 ENTIRE RISK AS TO SATISFACTORY QUALITY, PERFORMANCE, ACCURACY,
 AND EFFORT IS WITH THE YOU.  See the GNU Lesser General Public
 License for more details.

 You should have received a copy of the GNU Lesser General Public
 License along with this library; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307  USA

=cut
