/*
  log.c

  $Id: log.c,v 1.7 2002/12/27 21:48:25 bears Exp $
*/

#if HAVE_CONFIG_H
#include <config.h>
#endif

#include <syslog.h>
#include <stdarg.h>
#include "common.h"
#include "lock.h"
#include "log.h"
#include "portable.h"

#define MAXLENGTH 240

#define	DEFAULT_DBG_MASK	LOG_DBG_NONE

static struct
{
    Bool interactive;
    unsigned debugMask;
} log = { FALSE, DEFAULT_DBG_MASK };

void
Log_init( const char *name, Bool interactive, int facility )
{
    int option = LOG_PID | LOG_CONS;

    log.interactive = interactive;
    openlog( name, option, facility );
}

#define DO_LOG( LEVEL )               \
    va_list ap;                       \
    Str t;                            \
                                      \
    va_start( ap, fmt );              \
    vsnprintf( t, MAXCHAR, fmt, ap ); \
    if ( MAXLENGTH < MAXCHAR )        \
        t[ MAXLENGTH ] = '\0';        \
    syslog( LEVEL, "%s", t );         \
    if ( log.interactive )            \
        fprintf( stderr, "%s\n", t );   \
    va_end( ap );

void
Log_inf( const char *fmt, ... )
{
    DO_LOG( LOG_INFO );
}

void
Log_err( const char *fmt, ... )
{
    DO_LOG( LOG_ERR );
}

/* Ensure the condition "cond" is true; otherwise log an error and return 1 */
int 
Log_check(int cond, const char *fmt, ... )
{
  if (!cond) {
    DO_LOG( LOG_ERR );
    return 1;
  }
  return 0;
}

void
Log_ntc( const char *fmt, ... )
{
    DO_LOG( LOG_NOTICE );
}

void
Log_dbg( unsigned subsystem, const char *fmt, ... )
{
    if ( ( subsystem & log.debugMask ) != 0 ) {
	DO_LOG( LOG_DEBUG );
    }
}

void
Log_setDbgMask( unsigned mask )
{
    /* A non-zero mask always include Noffle logs */
    if ( mask != 0 )
	mask |= LOG_DBG_NOFFLE;

    log.debugMask = mask;
}

/*
 * A fatal error. Log it, close down as much as possible and
 * exit with EXIT_FAILURE.
 */
void
Log_fatal( const char *fmt, ... )
{
    DO_LOG( LOG_ERR );
    exit( EXIT_FAILURE );
    /* NOTREACHED */
}

/* Fatal error function for gdbm */
void
Log_gdbm_fatal( const char *msg )
{
    Log_fatal( "gdbm: %s", msg );
    /* NOTREACHED */
}


