/* Copyright (C) 2000-2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   There are special exceptions to the terms and conditions of the GPL as it
   is applied to this software. View the full text of the exception in file
   EXCEPTIONS in the directory of this software distribution.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

#include "MYODBCSetup.h"

/*!
    \internal
    \brief      Adds a new DSN.

    \note       This function uses the current SQLSetConfigMode().
*/    
BOOL MYODBCSetupConfigDSNAdd( HWND hWnd, LPCSTR pszDriver, MYODBC_NAMEVALUE **aNameValues )
{
    int nDataSourceName;

    /*!
        ODBC RULE

        If a data source name is passed to ConfigDSN in lpszAttributes, ConfigDSN 
        checks that the name is valid.
    */    
    nDataSourceName = MYODBCSetupGetAttributeIndex( aNameValues, "DSN" );
    if ( nDataSourceName >=0 )
    {
        /*!
            ODBC RULE

            ConfigDSN should call SQLValidDSN to check the length of the data source 
            name and to verify that no invalid characters are included in the name.
        */    
        /*!
            MYODBC RULE
             
            Assumption is that this also checks to ensure we are not trying to create 
            a DSN using an odbc.ini reserved section name. 
        */
        if ( !SQLValidDSN( aNameValues[nDataSourceName]->pszValue ) )
        {
            SQLPostInstallerError( ODBC_ERROR_REQUEST_FAILED, "DSN contains illegal characters or length does not make sense." );
            return FALSE;
        }
    }

    /*!
        ODBC RULE

        If lpszAttributes contains enough information to connect to a data source, 
        ConfigDSN can add the data source or display a dialog box with which the user 
        can change the connection information. If lpszAttributes does not contain 
        enough information to connect to a data source, ConfigDSN must determine the 
        necessary information; if hwndParent is not null, it displays a dialog box to 
        retrieve the information from the user.
    */
    /*!
        MYODBC RULE

        We always show the dialog if given hWnd.
    */    
    if ( hWnd )
        return MYODBCSetupDataSourceConfig( hWnd, ODBC_ADD_DSN, pszDriver, aNameValues );

    /*!
        ODBC RULE
        
        If ConfigDSN cannot get complete connection information for a data source, it 
        returns FALSE.
    */
    /*!
        MYODBC RULE

        We want pszDriver and a DSN attribute - we can default the rest.
    */    
    if ( !pszDriver || !(*pszDriver) )
    {
        SQLPostInstallerError( ODBC_ERROR_INVALID_NAME, "Must be given a viable driver reference." );
        return FALSE;
    }

    if ( nDataSourceName < 0 )
    {
        SQLPostInstallerError( ODBC_ERROR_INVALID_KEYWORD_VALUE, "Missing DSN attribute." );
        return FALSE;
    }

    if ( aNameValues[nDataSourceName]->pszValue == 0 )
    {
        SQLPostInstallerError( ODBC_ERROR_INVALID_KEYWORD_VALUE, "Missing DSN attribute value." );
        return FALSE;
    }

    /*!
        ODBC RULE

        SQLWriteDSNToIni adds the data source name to the [ODBC Data Sources] section, 
        creates the data source specification section, and adds the DRIVER keyword with 
        the driver description as its value.
    */    
    if ( !SQLWriteDSNToIni( aNameValues[nDataSourceName]->pszValue, pszDriver ) )
        return FALSE;

    /*!
        ODBC RULE

        ConfigDSN calls SQLWritePrivateProfileString in the installer DLL to add any 
        additional keywords and values used by the driver.
    */
    return MYODBCSetupWriteAttributes( aNameValues, nDataSourceName );
}



