/* Copyright (C) 2000-2004 MySQL AB

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   There are special exceptions to the terms and conditions of the GPL as it
   is applied to this software. View the full text of the exception in file
   EXCEPTIONS in the directory of this software distribution.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA */

/*!
  \file   myodbcinst.c
  \author Peter Harvey <peterh@mysql.com>
  \brief  This program will aid installers when installing/uninstalling
          MyODBC.

          This program can; register/deregister a myodbc driver and create
          a sample dsn. The key thing here is that it does this with 
          cross-platform code - thanks to the ODBC installer API. This is
          most useful to those creating installers (apps using myodbc or 
          myodbc itself).

          For example; this program is used in the postinstall script
          of the MyODBC for OSX installer package. 
*/

#include <stdio.h>

#ifdef _WIN_
    #include <windows.h>
    #include <odbcinst.h>
#endif

#ifdef _OSX_
    #include <iodbcinst.h>
    #include <ltdl.h>
#endif

#ifdef _UNIX_
    #include <odbcinst.h>
    #include <ltdl.h>
#endif

#ifndef TRUE
    #define TRUE 1
    #define FALSE 0
#endif

/*! Our syntax. This is likely to expand over time. */
char *szSyntax =
"\n" \
"myodbcinst\n" \
"----------\n" \
"\n" \
"Purpose:\n" \
"\n" \
"  Use this program to manage the MyODBC driver and data sources from the\n" \
"  command-line. This is particularly useful for installation processes.\n" \
"\n" \
"Usage:\n" \
"\n" \
"  $ myodbcinst <action> <option>\n" \
"\n" \
"Actions:\n" \
"\n" \
"  -q    query\n" \
"  -i    register Driver or increase usage count\n" \
"  -u    deregister Driver or decrease usage count\n" \
"  -s    create sample Data Source Name\n" \
"  -e    edit Data Source Name\n" \
"  -r    remove Data Source Name\n" \
"\n" \
"Options:\n" \
"\n" \
"  -d    query is for drivers (default is data source names)\n" \
"  -g    use GUI as needed (provided by ConfigDSN)\n" \
"  -n    driver name or data source name (optional)\n" \
"\n" \
"Example Session:\n" \
"\n" \
"  1. <copy driver and setup libs to /usr/lib or system32>\n" \
"  2. # myodbcinst -i\n" \
"  3. $ myodbcinst -s -ntest\n" \
"  4. $ myodbcinst -e -g -ntest\n" \
"\n" \
"Enjoy\n" \
"Peter Harvey\n";

char *  pszDriverName                   = "MySQL ODBC 3.51 Driver";
#ifdef _WIN_
/* We do not want to use Qt based GUI in our setup on Windows - not now and perhaps never.
char *  pszDriverAttributes             = "DRIVER=myodbc3.dll\0SETUP=myodbc3S.dll\0\0";
int     nDriverAttributesLength         = 39; 
*/
char *  pszDriverAttributes             = "DRIVER=myodbc3.dll\0SETUP=myodbc3.dll\0\0";
int     nDriverAttributesLength         = 38; 
#endif
#ifdef _OSX_
/*!
    \note OSX

          At least some of the functions dealing with the
          odbc system information are case sensitive
          (contrary to the ODBC spec.). They appear to like
          leading caps.
*/
char *  pszDriverAttributes             = "Driver=/usr/lib/libmyodbc3.dylib\0Setup=/usr/lib/libmyodbc3S.dylib\0\0";
int     nDriverAttributesLength         = 67; 
#endif
#ifdef _UNIX_
char *  pszDriverAttributes             = "DRIVER=/usr/lib/libmyodbc3.so\0SETUP=/usr/lib/libmyodbc3S.so\0\0";
int     nDriverAttributesLength         = 61; 
#endif

char *  pszDataSourceName               = "myodbc";
char *  pszDataSourceNameAttributes     = "SERVER=localhost\0USER=\0PASSWORD=\0DATABASE=test\0PORT=\0OPTION=3\0SOCKET=\0STMT=\0\0";
int     nDataSourceNameAttributesLength = 77;

char    cAction                         = '_';
int     bGUI                            = 0;
int     bQueryDrivers                   = 1;

#if defined(_WIN_)
/*!
    \brief  Get the last error and display string ver of it.
*/
void printLastErrorString() 
{
    LPVOID pszMsg;

    FormatMessage(FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
                  NULL,
                  GetLastError(),
                  MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),
                  (LPTSTR) &pszMsg,
                  0, 
                  NULL );
    fprintf( stderr, pszMsg );
    LocalFree( pszMsg );
}
#endif

/*!
    \brief  Prints any installer errors sitting in the installers
            error queue.

            General purpose error dump function for installer 
            errors. Hopefully; it provides useful information about
            any failed installer call.

    \note   Does not try to process all 8 possible records from the queue.
*/            
void printInstallerError()
{
    WORD      nRecord = 1;
    DWORD     nError;
    char      szError[SQL_MAX_MESSAGE_LENGTH];
    RETCODE   nReturn;

    nReturn = SQLInstallerError( nRecord, &nError, szError, SQL_MAX_MESSAGE_LENGTH - 1, 0 );
    if ( SQL_SUCCEEDED( nReturn ) )
        fprintf( stderr, "[%s][%d][ERROR] ODBC Installer error %d: %s\n", __FILE__, __LINE__, nError, szError );
    else
        fprintf( stderr, "[%s][%d][ERROR] ODBC Installer error (unknown)\n", __FILE__, __LINE__ );
}

/*!
    \brief  Show a list of drivers or data source names.

    \note   OSX

            SQLGetPrivateProfileString() is broken but fortunately the
            old call, GetPrivateProfileString() is available.

    \note   XP

            SQLGetPrivateProfileString() with a NULL 1st arg does not
            return anything - ever. To return a list of drivers we can
            use SQLGetInstalledDrivers() but no alternative in ODBC
            installer for getting a list of data source names.
*/
int doQuery( char *pszSource )
{
    char    szResults[32000];
    char *  ptr;
    int     nChars = 0;

#if defined(_OSX_)
    nChars = GetPrivateProfileString( NULL, NULL, "", szResults, sizeof( szResults ) - 1, pszSource );
#elif defined(_WIN_)
    if ( strcmp( pszSource, "ODBCINST.INI" ) == 0 )
        nChars = ( SQLGetInstalledDrivers( szResults, sizeof(szResults), NULL ) ? 1 : 0 );
    else
        nChars = SQLGetPrivateProfileString( pszDriverName, NULL, "", szResults, sizeof( szResults ) - 1, pszSource );
#else
    nChars = SQLGetPrivateProfileString( NULL, NULL, "", szResults, sizeof( szResults ) - 1, pszSource );
#endif
    if ( nChars < 1 )
    {
        printInstallerError();
        fprintf( stderr, "[%s][%d][INFO] Call returned no data. Could be an error or just no data to return.\n", __FILE__, __LINE__ );
        return 0;
    }
    
    ptr = szResults;
    while ( *ptr )
    {
        printf( "[%s]\n", ptr );
        ptr += strlen( ptr ) + 1;
    }

    return 1;
}

/*!
    \brief  Register the driver (or just increase usage count).

    \note   XP
    
            On MS Windows (XP for example) the driver is registered in two places; 
              1) \windows\system32\odbcinst.ini
              2) registry
              Fortunately the installer calls will ensure they are both updated.

            All ODBC drivers *should* be installed in the standard location (\windows\system32) and this call
            reflects this as no path is given for the driver file.

    \note   OSX
    
            On OSX there are many odbcinst.ini files - each account has one in ~/Library/ODBC and there
            is a system wide one in /Library/ODBC. This function will register the driver in ~/Library/ODBC.

            There are at least two notable complicating factors;
              - the files are read-ony for average user so one should use sudo when doing this
              - the files do not exist until someone manually creates entries using the ODBC Administrator AND
                they are not created by iodbc installer lib when we execute this code (we get error)

            ODBC spec says that "Driver" should NOT include path 
            but path seems needed for iodbc. The implication is that
            the driver *must* be installed in /usr/lib for this to work.

            Usage Count is not returned on OSX and returned location does not seem to reflect reality.

    \note   Linux/UNIX

            ODBC spec says that "Driver" should NOT include path 
            but path seems needed for unixODBC. The implication is that
            the driver *must* be installed in /usr/lib for this to work.

            Location returned does not seem to reflect reality.
*/
int doRegisterDriver()
{
    char  *pszAttributes = (char *)malloc( nDriverAttributesLength + strlen(pszDriverName) + 1 );
    char  szLoc[FILENAME_MAX];
    WORD  nLocLen;
    DWORD nUsageCount = -1;

    strcpy( pszAttributes, pszDriverName );
    memcpy( &(pszAttributes[strlen(pszAttributes) + 1]), pszDriverAttributes, nDriverAttributesLength ); 
    if ( !SQLInstallDriverEx( pszAttributes, 0, szLoc, FILENAME_MAX, &nLocLen, ODBC_INSTALL_COMPLETE, &nUsageCount ) )
    {
        printInstallerError();
        fprintf( stderr, "[%s][%d][ERROR] Failed to register driver\n", __FILE__, __LINE__ );
        free( pszAttributes );
        return 0;
    }

    printf( "[%s][%d][INFO] Driver registered. Usage count is %d. Location \"%s\" \n", __FILE__, __LINE__, nUsageCount, szLoc );
    free( pszAttributes );
    return 1;
}

/*!
  \brief  Deregister the driver.

      Simply removes driver from the list of known ODBC 
      drivers - does not remove any files.
*/
int doDeregisterDriver()  
{
    DWORD nUsageCount;
    BOOL  bRemoveDSNs = FALSE;

    if ( !SQLRemoveDriver( pszDriverName, bRemoveDSNs, &nUsageCount ) )
    {
        printInstallerError();
        fprintf( stderr, "[%s][%d][ERROR] Failed to deregister driver.\n", __FILE__, __LINE__ );
        return 0;
    }

    printf( "[%s][%d][INFO] Driver deregistered. Usage count is %d.\n", __FILE__, __LINE__, nUsageCount );
    return 1;
}

#if defined(_DIRECT_)
/*!
    \brief  Call drivers ConfigDSN when we are linked directly to driver.
*/
int doConfigDataSource( WORD nRequest )
{
    char  szAttributes[nDataSourceNameAttributesLength + strlen(pszDataSourceName) + 5];

    sprintf( szAttributes, "DSN=%s", pszDataSourceName );
    memcpy( &(szAttributes[strlen(szAttributes) + 1]), pszDataSourceNameAttributes, nDataSourceNameAttributesLength ); 
    if ( !ConfigDSN( (HWND)bGUI /* fake window handle */, nRequest, pszDriverName, szAttributes ) )
    {
        printInstallerError();
        return 0;
    }

    return 1;
}
#elif defined(_WIN_)
/*!
    \brief  Call drivers ConfigDSN without going through the DM.
*/
int doConfigDataSource( WORD nRequest )
{
    char  *     pszAttributes = NULL;
    BOOL        (*pFunc)( HWND, WORD, LPCSTR, LPCSTR );
    HINSTANCE   hLib = 0;
    char        szDriverSetup[FILENAME_MAX + 1];

    /* get the setup library from odbcinst */
    if ( !SQLGetPrivateProfileString( pszDriverName, "SETUP", "", szDriverSetup, FILENAME_MAX, "ODBCINST.INI" ) )
    {
        fprintf( stderr, "[%s][%d][ERROR] Could not determine the path/filename of driver setup library for driver (%s).\n", __FILE__, __LINE__, pszDriverName );
        return 0;
    }

    /* load it */
    hLib = LoadLibrary( szDriverSetup );
    if ( !hLib )
    {
        printLastErrorString();
        fprintf( stderr, "[%s][%d][ERROR] Could not load driver setup library (%s).\n", __FILE__, __LINE__, szDriverSetup );
        return 0;
    }

    /* lookup ConfigDSN */
    pFunc = (BOOL (*)(HWND, WORD, LPCSTR, LPCSTR )) GetProcAddress( hLib, "ConfigDSN" );
    if ( !pFunc )
    {
        printLastErrorString();
        fprintf( stderr, "[%s][%d][ERROR] Could not find ConfigDSN in (%s).\n", __FILE__, __LINE__, szDriverSetup );
        FreeLibrary( hLib );
        return 0;
    }

    /* make call */
    pszAttributes = malloc( nDataSourceNameAttributesLength + strlen(pszDataSourceName) + 5 );
    sprintf( pszAttributes, "DSN=%s", pszDataSourceName );
    memcpy( &(pszAttributes[strlen(pszAttributes) + 1]), pszDataSourceNameAttributes, nDataSourceNameAttributesLength ); 
    if ( !pFunc( (HWND)bGUI, nRequest, pszDriverName, pszAttributes ) )
    {
        printInstallerError();
        free( pszAttributes );
        return 0;
    }
    free( pszAttributes );

    /* cleanup */
    FreeLibrary( hLib );

    return 1;
}
#else
/*!
    \brief  Call drivers ConfigDSN without going through the DM.
*/
int doConfigDataSource( WORD nRequest )
{
    char    szAttributes[nDataSourceNameAttributesLength + strlen(pszDataSourceName) + 5];
    BOOL    (*pFunc)( HWND, WORD, LPCSTR, LPCSTR );
    void *  hLib = 0;
    char    szDriverSetup[FILENAME_MAX + 1];

    /* get the setup library from odbcinst */
#ifdef _OSX_
    if ( !GetPrivateProfileString( pszDriverName, "SETUP", "", szDriverSetup, FILENAME_MAX, "ODBCINST.INI" ) )
#else
    if ( !SQLGetPrivateProfileString( pszDriverName, "SETUP", "", szDriverSetup, FILENAME_MAX, "ODBCINST.INI" ) )
#endif
    {
        fprintf( stderr, "[%s][%d][ERROR] Could not determine the path/filename of driver setup library for driver (%s).\n", __FILE__, __LINE__, pszDriverName );
        return 0;
    }

    /* load it */
    lt_dlinit();
    if ( !(hLib = lt_dlopen( szDriverSetup )) )
    {
        fprintf( stderr, "[%s][%d][ERROR] Could not load driver setup library (%s). Error is %s\n", __FILE__, __LINE__, szDriverSetup, lt_dlerror() );
        return 0;
    }

    /* lookup ConfigDSN */
    pFunc = (BOOL (*)(HWND, WORD, LPCSTR, LPCSTR )) lt_dlsym( hLib, "ConfigDSN" );
    if ( !pFunc )
    {
        fprintf( stderr, "[%s][%d][ERROR] Could not find ConfigDSN in (%s). Error is %s\n", __FILE__, __LINE__, szDriverSetup, lt_dlerror() );
        lt_dlclose( hLib );
        return 0;
    }

    /* make call */
    sprintf( szAttributes, "DSN=%s", pszDataSourceName );
    memcpy( &(szAttributes[strlen(szAttributes) + 1]), pszDataSourceNameAttributes, nDataSourceNameAttributesLength ); 
    if ( !pFunc( (HWND)bGUI /* fake window handle */, nRequest, pszDriverName, szAttributes ) )
    {
        printInstallerError();
        return 0;
    }

    /* cleanup */
    lt_dlclose( hLib );

    return 1;
}
#endif

/*!
    \brief  Remove data source name from ODBC system information.
*/
int doRemoveDataSource()
{
    if ( SQLRemoveDSNFromIni( pszDataSourceName ) == FALSE )
    {
        printInstallerError();
        fprintf( stderr, "[%s][%d][ERROR] Request failed.\n", __FILE__, __LINE__ );
        return 0;
    }

    return 1;
}

/*!
    \brief  This is the entry point to this program.
            
    \note   More features/args will probably be added in the future.
*/
int main( int argc, char *argv[] )
{
    int nArg;
    int nName = -1;

    if ( argc <= 1 )
    {
        printf( szSyntax );
        exit( 1 );
    }

    /* parse args */
    for ( nArg = 1; nArg < argc; nArg++ )
    {
        if ( argv[nArg][0] == '-' )
        {
            switch ( argv[nArg][1] )
            {
                /* actions (more args to be added) */
                case 'q':
                case 'i':
                case 'u':
                case 'e':
                case 's':
                case 'r':
                    cAction = argv[nArg][1];
                    break;
                case 'g':
#ifdef _WIN_
                    printf( "GUI option is not valid on Windows because Windows specific driver setup code does not know about the fake window handle.\n" );
                    exit( 1 );
#else
                    bGUI = 1;
#endif
                    break;
                case 'd':
                    bQueryDrivers = 0;
                    break;
                case 'n':
                    nName = nArg;
                    break;
                default:
                    {
                        printf( szSyntax );
                        exit( 1 );
                    }
            }
        }
    }

    /* */
    if ( nName >= 0 )
    {
        if ( cAction == 'i' || cAction == 'u' )
            pszDriverName = &(argv[nName][2]); 
        else if ( cAction == 's' || cAction == 'e' || cAction == 'r' )
            pszDataSourceName = &(argv[nName][2]);
    }

    if ( !(*pszDriverName) || !(*pszDataSourceName) )
    {
        printf( "[%s][%d][ERROR] Missing driver name or data source name.\n", __FILE__, __LINE__ );
        exit( 1 );
    }

    /* do it */
    switch ( cAction )
    {
        case 'q':
            if ( bQueryDrivers )
                doQuery( "ODBCINST.INI" );
            else
                doQuery( "ODBC.INI" );
            break;
        case 'i':
            doRegisterDriver();
            break;
        case 'u':
            doDeregisterDriver();
            break;
        case 's':
            doConfigDataSource( ODBC_ADD_DSN );
            break;
        case 'e':
            doConfigDataSource( ODBC_CONFIG_DSN );
            break;
        case 'r':
            doRemoveDataSource();
            break;
        default:  
            printf( "[%s][%d][ERROR] Invalid, or missing, action.\n", __FILE__, __LINE__ );
            exit( 1 );
    }

    return 0;
}

