/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <math.h>
#include <unistd.h>
#include <string.h>

#include "MRI.h"
#include "MRW_Loader.h"

char	*progname;

/* This program reads a RAW Photo Image from its standard input,
 * then writes a 2D map of the chromaticities in the input image to the
 * standard output.
 *
 * The approximate convex hull of the input chromaticities is also calculated
 * and indicated as colored pixels on the chromaticity map (and also to the
 * standard error.
 */
void
Usage()
{
	fprintf (stderr, "Usage: %s < mrwfile > pnmfile\n", progname);
	fprintf (stderr, "Command line options:\n");
	PrintOptions (stderr);
	exit (1);
}

struct range {
	int	valid;
	int	min;
	int	max;
};

struct vertex {
	int	x, y;
};

struct ChromaMapData {
	MRI_Link *next;
	int	width;
	int	height;
	int	freedata;
	int	xmin, xmax;
	struct range *yrange;
	struct MRI_ScanLine **sl;
	int	nVertices;
	struct vertex *vertices;
};

#define	CHROMAMAP_HEIGHT 800
#define	CHROMAMAP_WIDTH	800

void
ChromaMapStart (void *private, int width, int height, int freedata)
{
	int i, j;
	struct ChromaMapData *wd = private;
	wd->width = width;
	wd->height = height;
	wd->freedata = freedata;
	wd->sl = (struct MRI_ScanLine **)malloc(sizeof(struct MRI_ScanLine *) * CHROMAMAP_HEIGHT);
	for (i = 0; i < CHROMAMAP_HEIGHT; i++) {
		wd->sl[i] = MRI_NewScanLine (LINETYPE_SHORT, CHROMAMAP_WIDTH);
		for (j = 0; j < CHROMAMAP_WIDTH; j++)
			((unsigned short *)(wd->sl[i]->R))[j] = 0;
		for (j = 0; j < CHROMAMAP_WIDTH; j++)
			((unsigned short *)(wd->sl[i]->G))[j] = 0;
		for (j = 0; j < CHROMAMAP_WIDTH; j++)
			((unsigned short *)(wd->sl[i]->B))[j] = 0;
	}
	wd->xmin = CHROMAMAP_WIDTH-1;
	wd->xmax = 0;
	wd->yrange = (struct range *)malloc(sizeof(struct range) * CHROMAMAP_WIDTH);
	for (j = 0; j < CHROMAMAP_WIDTH; j++)
		wd->yrange[j].valid = 0;
	(*wd->next->start) (wd->next->private, CHROMAMAP_WIDTH, CHROMAMAP_HEIGHT, TRUE);
}

/* Chromaticities are clipped to this value.
 */
#define CHROMAMAX 8.0

void
ChromaMapRow (void *private, void *data)
{
	struct ChromaMapData *wd = private;
	struct MRI_ScanLine *sl = data;
	unsigned short *R, *G, *B;
	int x;

	if (sl->scanLineType != LINETYPE_SHORT) {
		fprintf (stderr, "ChromaMapRow: unexpected scanLineType %d\n", sl->scanLineType);
		exit (1);
	}
	R = (unsigned short *)sl->R;
	G = (unsigned short *)sl->G;
	B = (unsigned short *)sl->B;
	for (x = 0; x < wd->width; x++) {
		double	r, b;
		int px, py;

		if (G[x] != 0) {
			double scale = 1.0 / G[x];
			r = scale * R[x];
			b = scale * B[x];
		}
		else if (B[x] != 0) {
			r = R[x] / (B[x]*CHROMAMAX);
			b = CHROMAMAX;
		}
		else {
			r = CHROMAMAX;
			b = 0.0;
		}
		if (r > CHROMAMAX) {
			b = (b/r) * CHROMAMAX;
			r = CHROMAMAX;
		}
		if (b > CHROMAMAX) {
			r = (r/b) * CHROMAMAX;
			b = CHROMAMAX;
		}

		/* Compute pixel location. */
		px = b * CHROMAMAP_WIDTH / CHROMAMAX;
		if (px >= CHROMAMAP_WIDTH) px = CHROMAMAP_WIDTH - 1;
		py = r * CHROMAMAP_HEIGHT / CHROMAMAX;
		if (py >= CHROMAMAP_HEIGHT) py = CHROMAMAP_HEIGHT - 1;

		/* Set this pixel to white. */
		((unsigned short *)(wd->sl[py]->R))[px] = 65535;
		((unsigned short *)(wd->sl[py]->G))[px] = 65535;
		((unsigned short *)(wd->sl[py]->B))[px] = 65535;

		/* Store data needed for computing convex hull. */
		if (px < wd->xmin) wd->xmin = px;
		if (px > wd->xmax) wd->xmax = px;
		if (wd->yrange[px].valid) {
			if (py < wd->yrange[px].min) wd->yrange[px].min = py;
			if (py > wd->yrange[px].max) wd->yrange[px].max = py;
		}
		else {
			wd->yrange[px].valid = 1;
			wd->yrange[px].min = py;
			wd->yrange[px].max = py;
		}
	}
	if (wd->freedata) {
		MRI_FreeScanLine (sl);
	}
}

void
ChromaMapClose (void *private)
{
	struct ChromaMapData *wd = private;
	int i, nv;

	fprintf (stderr, "Chroma: xmin=%d, xmax=%d\n", wd->xmin, wd->xmax);

	/* Build approximate convex hull. */
	wd->vertices = (struct vertex *)malloc(sizeof(struct vertex) * (wd->xmax - wd->xmin + 1) * 2);
	wd->vertices[0].x = wd->xmin;
	wd->vertices[0].y = wd->yrange[wd->xmin].min;
	wd->nVertices = 1;
	for (i = wd->xmin+1; i <= wd->xmax; i++) {
		if (wd->yrange[i].valid) {
			while (wd->nVertices > 1) {
				int dx = wd->vertices[wd->nVertices-1].x - wd->vertices[wd->nVertices-2].x;
				int dy = wd->vertices[wd->nVertices-1].y - wd->vertices[wd->nVertices-2].y;
				if (dx * wd->yrange[i].min >
				    dx * wd->vertices[wd->nVertices-2].y +
				    dy * (i - wd->vertices[wd->nVertices-2].x))
				    break;
				wd->nVertices--;
			}
			wd->vertices[wd->nVertices].x = i;
			wd->vertices[wd->nVertices].y = wd->yrange[i].min;
			wd->nVertices++;
		}
	}
	if (wd->yrange[wd->xmax].max != wd->yrange[wd->xmax].min) {
		wd->vertices[wd->nVertices].x = wd->xmax;
		wd->vertices[wd->nVertices].y = wd->yrange[wd->xmax].max;
		wd->nVertices++;
	}
	nv = 1;
	for (i = wd->xmax-1; i >= wd->xmin; i--) {
		if (wd->yrange[i].valid) {
			while (nv > 1) {
				int dx = wd->vertices[wd->nVertices-1].x - wd->vertices[wd->nVertices-2].x;
				int dy = wd->vertices[wd->nVertices-1].y - wd->vertices[wd->nVertices-2].y;
				if (dx * wd->yrange[i].max >
				    dx * wd->vertices[wd->nVertices-2].y +
				    dy * (i - wd->vertices[wd->nVertices-2].x))
				    break;
				wd->nVertices--;
				nv--;
			}
			if (i != wd->xmin ||
			    wd->yrange[wd->xmin].min != wd->yrange[wd->xmin].max)
			    {
			        wd->vertices[wd->nVertices].x = i;
			        wd->vertices[wd->nVertices].y = wd->yrange[i].max;
			        wd->nVertices++;
			        nv++;
			    }
		}
	}

	/* Output convex hull vertices to stderr. */
	fprintf (stderr, "Convex hull: ");
	for (i = 0; i < wd->nVertices; i++) {
		int x = wd->vertices[i].x;
		int y = wd->vertices[i].y;
		fprintf (stderr, " %d,%d", wd->vertices[i].x, wd->vertices[i].y);
		((unsigned short *)(wd->sl[y]->R))[x] = 0;
		((unsigned short *)(wd->sl[y]->B))[x] = 0;
	}
	fprintf (stderr, "\n");

	/* Output chromaticity diagram. */
	for (i = 0; i < CHROMAMAP_HEIGHT; i++)
		(*wd->next->row) (wd->next->private, wd->sl[i]);
	(*wd->next->close) (wd->next->private);
	free (wd);
}

struct link *
MRI_GenChromaMap (MRI_Link *next)
{
	struct ChromaMapData *wd = malloc (sizeof (struct ChromaMapData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct ChromaMapData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = ChromaMapStart;
	ep->row = ChromaMapRow;
	ep->close = ChromaMapClose;
	ep->private = wd;
	wd->next = next;
	return ep;
}

int
main (int argc, char *argv[])
{
	MRI *mri;
	struct link *head;
	char		*errmsg;
	int	dx, dy;

	progname = argv[0];
	mri = MRW_Loader (stdin, &errmsg);
	if (mri == (MRI *)0) {
		fprintf (stderr, "%s: cannot read image from standard input (%s).\n", progname, errmsg);
		Usage ();
	}

	/* Create output filter sequence starting from the rear.
	 * The last link in the pipeline outputs scanlines to the output device.
	 */
	head = MRI_GenPPMWriter (8, 0, stdout);
	head = MRI_GenChromaMap (head);
	if (argc == 4) {
		MRI_balance b;
		b.rgain = atoi (argv[1]);
		b.ggain = atoi (argv[2]);
		b.bgain = atoi (argv[3]);
		head = MRI_GenColorBalancer (b, head);
	}
	head = MRI_GenSubsampler (2, 2, head);
        MRI_ProcessImage (head, mri, 0);
        MRI_FlushPipeline (head);

	exit (0);
}
