/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <unistd.h>
#include <string.h>

#include "MRI.h"

void
InitBalanceMap (MRI_BalanceMap **mapptr)
{
	*mapptr = NULL;
}

void
FreeBalanceMap (MRI_BalanceMap *map)
{
	MRI_BalanceMap *next;

	while (map) {
		next = map->next;
		free (map->name);
		free (map);
		map = next;
	}
}

static void
ClearBalance (MRI_BalanceMap **mapptr, const char *name)
{
	MRI_BalanceMap *ptr;
	MRI_BalanceMap *prev;

	prev = NULL;
	ptr = *mapptr;
	while (ptr) {
		if (strcmp (ptr->name, name) == 0) {
			if (prev)
				prev->next = ptr->next;
			else
				*mapptr = ptr->next;
			free (ptr->name);
			free (ptr);
			return;
		}
		prev = ptr;
		ptr = ptr->next;
	}
}

static void
InsertBalance (MRI_BalanceMap **mapptr, const char *name, MRI_balance bal)
{
	MRI_BalanceMap *new = (MRI_BalanceMap *)malloc(sizeof(MRI_BalanceMap));

	if (new == NULL)
		fprintf (stderr, "InsertBalance: cannot allocate memory\n");
	else {
		new->name = strdup (name);
		new->bal = bal;
		new->next = *mapptr;
		*mapptr = new;
	}
}

static int
LookupBalance (MRI_balance *bal, MRI_BalanceMap *map, const char *name)
{
	while (map) {
		if (strcmp (map->name, name) == 0) {
			*bal = map->bal;
			return 1;
		}
		map = map->next;
	}
	return 0;
}

void
MRI_SetBalance (MRI *mri, MRI_balance wb)
{
    double Ravg, Gavg, Bavg;

    mri->wBalance = wb;
    Ravg = mri->Ravg * wb.rgain;
    Gavg = mri->Gavg * wb.ggain;
    Bavg = mri->Bavg * wb.bgain;
#if 0
    fprintf (stderr, "After balance: Ravg = %g\n", Ravg);
    fprintf (stderr, "After balance: Gavg = %g\n", Gavg);
    fprintf (stderr, "After balance: Bavg = %g\n", Bavg);
#endif
}

void
MRI_SetDFactor (MRI *mri, int df)
{
    mri->dFactor = df;
}

const char *
MRI_GetCameraName (const MRI *mri)
{
    return (*mri->methods->CameraName) (mri);
}

int
MRI_GetFlash (const MRI *mri)
{
    return (*mri->methods->Flash) (mri);
}

int
MRI_GetWidth (const MRI *mri)
{
    return (*mri->methods->Width) (mri);
}

int
MRI_GetHeight (const MRI *mri)
{
    return (*mri->methods->Height) (mri);
}

void
MRI_ClearBalance (MRI *mri, const char *name)
{
    ClearBalance (&mri->balanceMap, name);
}

int
MRI_GetBalanceRegion (MRI_balance *bal, MRI *mri, const char *name, MRI_Region *region)
{
    int retval;

    if (LookupBalance(bal, mri->balanceMap, name))
    	return TRUE;

    if ((*mri->methods->GetPresetBalance)(mri, name, bal))
        return TRUE;

    if (strcmp (name, "grayworld") == 0) {
        *bal = MRI_GrayWorldWB (mri, region);
        retval = TRUE;
    }
    else if (strcmp (name, "bingrayworld") == 0) {
        *bal = MRI_BinGrayWorldWB (mri, region);
        retval = TRUE;
    }
    else if (strcmp (name, "medgrayworld") == 0) {
        *bal = MRI_EstimateWB (mri, region);
        retval = TRUE;
    }
    else if (strcmp (name, "avgestimate") == 0) {
        MRI_balance b1, b2, b3;

        MRI_GetBalanceRegion (&b1, mri, "bingrayworld", region);
        MRI_GetBalanceRegion (&b2, mri, "medgrayworld", region);
        MRI_GetBalanceRegion (&b3, mri, "grayworld", region);
        bal->rgain = (b1.rgain + b2.rgain + b3.rgain) / 3;
        bal->ggain = (b1.ggain + b2.ggain + b3.ggain) / 3;
        bal->bgain = (b1.bgain + b2.bgain + b3.bgain) / 3;
        retval = TRUE;
    }
    else {
        bal->rgain = 256;
        bal->ggain = 256;
        bal->bgain = 256;
        retval = FALSE;
    }

    if (retval)
        InsertBalance (&mri->balanceMap, name, *bal);
    return retval;
}

int
MRI_GetBalance (MRI_balance *bal, MRI *mri, const char *name)
{
    return MRI_GetBalanceRegion (bal, mri, name, NULL);
}

const char *
MRI_GetTimestamp (const MRI *mri)
{
    return (*mri->methods->TimeStamp) (mri);
}

double
MRI_GetShutter (const MRI *mri)
{
    return (*mri->methods->Shutter) (mri);
}

double
MRI_GetAperture (const MRI *mri)
{
    return (*mri->methods->Aperture) (mri);
}

double
MRI_GetISO (const MRI *mri)
{
    return (*mri->methods->ISO) (mri);
}

int
MRI_GetFocusMode (const MRI *mri)
{
    return (*mri->methods->FocusMode) (mri);
}

cmsCIEXYZ *
MRI_GetWhitePoint (const MRI *mri)
{
    return (*mri->methods->WhitePoint) (mri);
}

cmsHPROFILE
MRI_GetNativeProfile (const MRI *mri)
{
    return (*mri->methods->NativeProfile) (mri);
}

double
MRI_GetFocusLen (const MRI *mri)
{
    return (*mri->methods->FocusLen) (mri);
}

double
MRI_GetFocalLen (const MRI *mri)
{
    return (*mri->methods->FocalLen) (mri);
}

void
MRI_ConvertScanLineToLAB (const MRI *mri, struct MRI_ScanLine *sl)
{
    (*mri->methods->ConvertScanLineToLAB) (mri, sl);
}

#define MAXOVAL 0xFFFF

struct MRI_ScanLine *
MRI_GetRow (MRI *mri, int y, int xoff, int width)
{
	struct MRI_ScanLine *sl = MRI_NewScanLine (LINETYPE_SHORT, width);
	unsigned short *slR = sl->R;
	unsigned short *slG = sl->G;
	unsigned short *slB = sl->B;
	int x;

	for (x = 0; x < width; x++) {
		int r, g, b;
		r = (mri->R[y][x+xoff] * (int)mri->wBalance.rgain) >> 8;
		slR[x] = r > MAXOVAL ? MAXOVAL : r;
		g = (mri->G[y][x+xoff] * (int)mri->wBalance.ggain) >> 8;
		slG[x] = g > MAXOVAL ? MAXOVAL : g;
		b = (mri->B[y][x+xoff] * (int)mri->wBalance.bgain) >> 8;
		slB[x] = b > MAXOVAL ? MAXOVAL : b;
	}
	return sl;
}

struct MRI_ScanLine *
MRI_GetCol (MRI *mri, int x, int yoff, int height)
{
	struct MRI_ScanLine *sl = MRI_NewScanLine (LINETYPE_SHORT, height);
	unsigned short *slR = sl->R;
	unsigned short *slG = sl->G;
	unsigned short *slB = sl->B;
	int y;

	for (y = 0; y < height; y++) {
		int r, g, b;
		r = (mri->R[y+yoff][x] * (int)mri->wBalance.rgain) >> 8;
		slR[y] = r > MAXOVAL ? MAXOVAL : r;
		g = (mri->G[y+yoff][x] * (int)mri->wBalance.ggain) >> 8;
		slG[y] = g > MAXOVAL ? MAXOVAL : g;
		b = (mri->B[y+yoff][x] * (int)mri->wBalance.bgain) >> 8;
		slB[y] = b > MAXOVAL ? MAXOVAL : b;
	}
	return sl;
}

void
MRI_AdjustBalanceLuminance (MRI *mri, MRI_balance *balanceSpec, double *luma)
{
   (*mri->methods->AdjustBalanceLuminance) (mri, balanceSpec, luma);
}

void
MRI_Allocate (MRI *mri)
{
	int y, height = mri->height;
	int width = mri->width;
	extern char *progname;

	InitBalanceMap (&mri->balanceMap);
	mri->wBalance.rgain = mri->wBalance.ggain = mri->wBalance.bgain = 256;
	mri->data = (unsigned short *)calloc (3 * height * width * sizeof (unsigned short),1);
	mri->R = malloc(height * sizeof(unsigned short *));
	mri->G = malloc(height * sizeof(unsigned short *));
	mri->B = malloc(height * sizeof(unsigned short *));
	if (mri->data == (unsigned short *)0 ||
	    mri->R == (unsigned short **)0 ||
	    mri->G == (unsigned short **)0 ||
	    mri->B == (unsigned short **)0 ) {
		fprintf (stderr, "%s: Unable to allocate MRI memory.\n", progname);
		exit (1);
	}
	for (y = 0; y < height; y++) {
		mri->R[y] = 3 * width * y + mri->data;
		mri->G[y] = mri->R[y] + width;
		mri->B[y] = mri->G[y] + width;
	}
}


extern void
MRI_Free (MRI *mri)
{
	FreeBalanceMap (mri->balanceMap);
	free (mri->data);
	free (mri->R);
	free (mri->G);
	free (mri->B);
	free (mri);
}

struct MRI_ScanLine *
MRI_NewScanLine (int type, int width)
{
	struct MRI_ScanLine *sl;
	int i, size;

	sl = malloc (sizeof (*sl));
	if (sl == (struct MRI_ScanLine *)0) {
		fprintf (stderr, "Error: out of memory!\n");
		exit (1);
	}
	sl->scanLineType = type;
	sl->sl_Width = width;
	sl->sl_NumChannels = 3;
	sl->sl_Channel = (void **)malloc(sizeof(void *) * sl->sl_NumChannels);
	switch (type) {
	case LINETYPE_SHORT: size=sizeof(unsigned short); break;
	case LINETYPE_LONG: size=sizeof(unsigned int); break;
	case LINETYPE_FLOAT: size=sizeof(float); break;
	case LINETYPE_DOUBLE: size=sizeof(double); break;
	default:
		fprintf (stderr, "MRI_NewScanLine: Illegal line type %d\n", type);
		exit (1);
	}
	if (sl->sl_Channel == (void **)0) {
		fprintf (stderr, "MRI_NewScanLine: out of memory!\n");
		exit (1);
	}
	for (i = 0; i < sl->sl_NumChannels; i++) {
		sl->sl_Channel[i] = (void *)malloc(size * width);
		if (sl->sl_Channel[i] == (void *)0) {
			fprintf (stderr, "MRI_NewScanLine: out of memory!\n");
			exit (1);
		}
	}
	sl->R = sl->sl_Channel[0];
	sl->G = sl->sl_Channel[1];
	sl->B = sl->sl_Channel[2];
	sl->masks = (unsigned long *)malloc(sizeof(unsigned long) * width);
	if (sl->masks == (unsigned long *)0) {
		fprintf (stderr, "MRI_NewScanLine: out of memory!\n");
		exit (1);
	}
	return sl;
}

void
MRI_FreeScanLine (struct MRI_ScanLine *sl)
{
	int i;

	if (sl != (struct MRI_ScanLine *)0) {
		for (i = 0; i < sl->sl_NumChannels; i++) free (sl->sl_Channel[i]);
		free (sl->sl_Channel);
		free (sl->masks);
		free (sl);
	}
}

void
MRI_FlushPipeline (MRI_Link *head)
{
	(*head->close) (head->private);
	free (head);
}
