/* Copyright (C) 2000-2002 Lavtech.com corp. All rights reserved.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
*/

#include "udm_config.h"

#if   (HAVE_MYSQL)
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include "udm_common.h"
#include "udm_db.h"
#include "udm_db_int.h"
#include "udm_utils.h"
#include "udm_vars.h"
#include "udm_sqldbms.h"

#include "udm_xmalloc.h"
#ifdef WIN32
#include <process.h>
#endif
#include <mysql.h>

#define ER_DUP_ENTRY            1062
#define ER_DUP_KEY		1022
#define CR_SERVER_LOST          2013
#define CR_SERVER_GONE_ERROR    2006
#define ER_SERVER_SHUTDOWN      1053

static int UdmMySQLInit(UDM_DB *db)
{
  const char* DBLog=  UdmVarListFindStr(&db->Vars,"sqllog",NULL);
  const char* DBSock= UdmVarListFindStr(&db->Vars,"socket",NULL);
  const char* DBUser= UdmVarListFindStr(&db->Vars,"DBUser",NULL);
  const char* DBPass= UdmVarListFindStr(&db->Vars,"DBPass",NULL);
  const char* DBHost= UdmVarListFindStr(&db->Vars, "DBHost", "localhost");
  int DBPort= UdmVarListFindInt(&db->Vars, "DBPort", 0);
  MYSQL *mysql;
  
  db->specific = mysql = mysql_init(NULL);
  if (! db->specific)
  {
    db->errcode = 1;
    sprintf(db->errstr, "Can't allocate specific. Not enough memory.");
    return UDM_ERROR;
  }

  if(!(mysql_real_connect(mysql, DBHost, DBUser, DBPass,
  			  db->DBName?db->DBName:"mnogosearch",
  			  (unsigned)DBPort, DBSock, 0)))
  {
    db->errcode=1;
    sprintf(db->errstr, "MySQL driver: #%d: %s",
    	    mysql_errno(mysql), mysql_error(mysql));
    return UDM_ERROR;
  }
  db->connected=1;
  if (DBLog)
  {
    char qbuf[64];
    sprintf(qbuf, "SET SQL_LOG_OFF=%d", atoi(DBLog) ? 0 : 1);
    mysql_query(mysql, qbuf);
  }
  return UDM_OK;
}

static
int UdmMySQLStoreResult(UDM_DB *db, UDM_SQLRES *R)
{
  size_t mitems= 0;
  MYSQL_ROW mysqlrow;
  
  while((mysqlrow=mysql_fetch_row((MYSQL_RES *)R->specific)))
  {
    size_t col;
    size_t coloffs= R->nRows * R->nCols;  
    unsigned long  *lengths= mysql_fetch_lengths((MYSQL_RES *)R->specific);
    
    if (coloffs + R->nCols >= mitems)
    {
      mitems= mitems ? mitems * 8 : 256;
      R->Items=(UDM_PSTR*)UdmRealloc(R->Items,mitems*sizeof(UDM_PSTR));
    }
    
    for(col= 0; col < R->nCols ; col++)
    {
      UDM_PSTR *I= &R->Items[coloffs + col];
      size_t len;
      len= I->len= lengths[col];
      I->val= (char*) UdmMalloc(len+1);
      memcpy(I->val, mysqlrow[col], len);
      I->val[len]='\0';
    }
    R->nRows++;
  }
  return UDM_OK;
}


static int UdmMySQLStoreMetaData(UDM_DB *db, UDM_SQLRES *R)
{
  if(R->specific)
  {
    MYSQL_FIELD  *field;
    size_t    nfields;
    
    R->nCols= mysql_num_fields((MYSQL_RES*)R->specific);
    R->nRows=0;
    R->Items=NULL;
    R->Fields=(UDM_SQLFIELD*)UdmMalloc(R->nCols*sizeof(UDM_SQLFIELD));
    bzero(R->Fields,R->nCols*sizeof(UDM_SQLFIELD));
    
    for(nfields=0; (field=mysql_fetch_field((MYSQL_RES *)R->specific)); nfields++)
    {
      R->Fields[nfields].sqlname = (char*)UdmStrdup(field->name);
      R->Fields[nfields].sqllen=field->length;
    }
  }
  return UDM_OK;
}


static int UdmMySQLExecDirect(UDM_DB *db,UDM_SQLRES *R,const char *query)
{
  size_t  i;
  MYSQL *mysql;
  
  db->errcode=0;
    
  if (R)
  {
    bzero((void*) R, sizeof(UDM_SQLRES));
    R->db= db;
  }

  if(!db->connected)
  {
    int rc=UdmMySQLInit(db);
    if(rc!=UDM_OK)return rc;
  }

  mysql = (MYSQL *)db->specific;

  for(i=0;i<2;i++)
  {
    if((mysql_query(mysql,query)))
    {
      if((mysql_errno(mysql)==CR_SERVER_LOST)||
         (mysql_errno(mysql)==CR_SERVER_GONE_ERROR)||
         (mysql_errno(mysql)==ER_SERVER_SHUTDOWN))
      {
        UDMSLEEP(5);
      }
      else
      {
        sprintf(db->errstr,"MySQL driver: #%d: %s",
                mysql_errno(mysql),mysql_error(mysql));
        if((mysql_errno(mysql)!=ER_DUP_ENTRY) &&
           (mysql_errno(mysql)!=ER_DUP_KEY))
        {
          db->errcode=1;
          return UDM_ERROR;
        }
        db->errcode=0;
        return UDM_OK;
      }
    }
    else
    {
      if (R)
      {
        R->specific= mysql_use_result((MYSQL*) db->specific);
        return UdmMySQLStoreMetaData(db, R);
      }
      else
        return UDM_OK;
    }
  }
  db->errcode=1;
  sprintf(db->errstr,"MySQL driver: #%d: %s",
          mysql_errno(mysql),mysql_error(mysql));
  return UDM_ERROR;
}


static int UdmMySQLQuery(UDM_DB *db,UDM_SQLRES *R,const char *query)
{
  int rc= UdmMySQLExecDirect(db, R, query);
  if (rc != UDM_OK)
    return rc;
  if (R->specific)
    return UdmMySQLStoreResult(db, R);
  return UDM_OK;
}


static void UdmMySQLClose(UDM_DB *db)
{
  MYSQL *mysql = (MYSQL *)db->specific;
  mysql_close(mysql);
  db->specific = NULL;
}

static int UdmMySQLBegin(UDM_DB *db)
{
  return UDM_OK;
}

static int UdmMySQLCommit(UDM_DB *db)
{
  return UDM_OK;
}

static char *UdmMySQLEscStr(UDM_DB *db, char *to, const char *from, size_t len)
{
  mysql_escape_string(to, from, len);
  return(to);
}


static int UdmMySQLFreeResult(UDM_DB *db, UDM_SQLRES *R)
{
  UdmSQLFreeResultSimple(db, R);
  if (R->specific)
  {
    mysql_free_result((MYSQL_RES *)R->specific);
    R->specific= NULL;
  }
  return UDM_OK;
}


static
int UdmMySQLFetchRow(UDM_DB *db, UDM_SQLRES *R, UDM_PSTR *buf)
{
  size_t j;
  MYSQL_ROW row;
  unsigned long *lengths;
  if (!(row= mysql_fetch_row((MYSQL_RES *)R->specific)))
    return UDM_ERROR;
  lengths= mysql_fetch_lengths((MYSQL_RES *)R->specific);
  
  for (j = 0; j < R->nCols; j++)
  {
    buf[j].val= row[j];
    buf[j].len= lengths[j];
  }
  
  return(UDM_OK);
}


UDM_SQLDB_HANDLER udm_sqldb_mysql_handler =
{
  UdmMySQLEscStr,
  UdmMySQLQuery,
  UdmMySQLClose,
  UdmMySQLBegin,
  UdmMySQLCommit,
  NULL,
  NULL,
  NULL,
  UdmMySQLFetchRow,
  UdmSQLStoreResultSimple,
  UdmMySQLFreeResult,
  UdmMySQLExecDirect
};
#endif
