#ifndef __MEM_AID_H__
#define __MEM_AID_H__


#include "ann.h"

#ifdef __cplusplus
extern "C" {
#endif
	
#define MA_CORE_VERSION 8 /* change it only when binary file format changes */
/* last changed: on change of 'id' type from time_t to u_long */
	
/*
 * Keep MAX_A_LEN >= MAX_Q_LEN
 */
#define MAX_Q_LEN 4096  /* max question length */
#define MAX_A_LEN 4096  /* max  answer  length */

/* return values of int ma_init() */
#define NN_DATA_DOES_NOT_EXIST 0x01  /* TO REMOVE */
#define BIN_DATA_DOES_NOT_EXIST 0x02 /* TO REMOVE */
#define ASCII_DATA_DOES_NOT_EXIST 0x04
#define DIR_NOT_ACCESSIBLE 0x08
#define MEMAID_ALREADY_RUNNING 0x10
#define XML_DATA_DOES_NOT_EXIST 0x20

#define MA_JUST_RETURN_IT 0 	/* used by ma_get_new_el (MA_JUST_RETURN_IT || MA_DELETE_IT, q, a); */
#define MA_DELETE_IT 1

/* 3 below are used only in ma_run_tags() */
#define CONSOLE_VERSION 0
#define GUI_VERSION 1
#define CGI_VERSION 2
	
struct elem {
	u_short tm_t_rpt;  /* (time to repeat); how many days since begin of learning proccess */
	u_short stm_t_rpt; /* last tm_t_rpt computed on a NOT ahead-of-time review */
	u_long id;         /* id field */
	char *q, *a;       /* question/answer */
	u_short l_ivl;     /* last interval computed by ann */
	u_short rl_l_ivl;  /* real last interval */
	u_short ivl;       /* current interval computed by ANN */
	u_short rp;        /* number of repetitions */
	u_short gr;        /* grade */
	struct elem *nxt;
};

struct ma_stats_st {
	unsigned long total_items;	/* total number of items in the learning proces */
	u_short rpts_for_today;		/* how many repetitions for today */
	u_short days;			/* number of days since first run */
	u_short max_ivl;		/* max current interval */
	u_short min_ivl;		/* min current interval */
	u_short mean_ivl;		/* mean current interval */
	u_short max_l_ivl_ann;		/* max last interval by ann - before last review of every item */
	u_short min_l_ivl_ann;
	u_short mean_l_ivl_ann;
	u_short max_l_ivl_real;		/* max real last interval (we don't know current real interval yet) */
	u_short min_l_ivl_real;
	u_short mean_l_ivl_real;

	float mean_grade;	/* (sum_grades/total_items) */
	
	float mean_reviews;
	unsigned long sum_reviews;	/* how many total reviews,
					 *  not counting final drill and repetition ahead of time
					 * (e.g. final drill)
					 */

	u_short max_reviews;		/* max number of reviews of an item so far */

	float	ann_error;	/* ANN error, rather only for debug,
				 * or just to sure that everything is ok
		 		 */
	unsigned long ann_cases;	/* total number of cases to train ANN */
	unsigned long user_ann_cases;	/* number of cases gathered from the user */

	
	float retention;	/* aproximate retention, proportion:
				 * (items_you_remember/total_items)
				 * (items_you_remember = items that had grade 3 or higher
				 * at last repetition
				 */

	float item_per_day;		/* New items per day */
	time_t first_run;		/* exact time/date of first run */
};


extern int ma_init   (const char *path_to_data_dir);
	/*
	 * initializes MemAid core.
	 * You should check return value if dir is ok, e.g.:
	 * int i = ma_init ("data/");
	 * if (i & DIR_NOT_ACCESSIBLE) {
	 * 	fputs ("Fatal error: cannot access data/ directory\n");
	 *	exit (1);
	 * }
	 */

extern void ma_deinit (void);
	/*
	 * deinitialize MemAid core. Don't forget to call it at the end,
	 * beside freeing memory, etc. it also saves data to disk.
	 */

extern int ma_get_new_el (int delete_it, char *q, char *a);
	/*
	 *  Gets new element from queue.txt file (if file and element exist),
	 * places this element question and answer in q/a pointers and returns 0.
	 * (If not exist returns 1.)
	 * After you take this element AND placed it in the learning process,
	 * you want this element to be deleted from queue.txt
	 * (so next time ma_get_new_element() will return next element, not the same),
	 * so you call ma_get_new_element() yet again with int delete_it = 1,
	 * this time ma_get_new_element() only deletes first element in queue.txt
	 */

extern int ma_list_n_of_els (void);
	/*
	 * Returns number of elements in the list
	 */

extern u_short ma_n_of_rpts_for_today (void);
	/*
	 * Returns how many repetitions are scheduled for today.
	 */

extern u_short ma_rpts_drill (u_short grade_threshold);
	/*
	 * Returns how many repetitions are to be drilled for today.
	 */

extern u_short ma_make_new_el (u_short grade, const char *q, const char *a);
	/*
	 * Makes new element in the learning process.
	 * Returns scheduled interval.
	 */

extern void ma_delete_el (struct elem*);
	/*
	 * Deletes element from the learning process.
	 * (First you would have to have pointer to that element,
	 *  you may get it, e.g., using ma_find_el())
	 */

extern struct elem* ma_el_to_repeat (void);
	/*
	 * Returns pointer to an element scheduled for repetition for today.
	 * (returns NULL if not exist).
	 */

extern struct elem *ma_el_force_repeat (void);
	/*
	 * You don't have to use this function.
	 * It returns any element (even if not scheduled for a review for today)
	 * to "forced repeat". It has many pros and some cons.
	 * All-in-all: it ("forced repetitions") is bad, but sometimes useful,
	 * that's why it's offered.
	 *
	 * Please see http://memaid.sourceforge.net/docs/forced_repetitions.html
	 * for more details about this mode.
	 */

extern struct elem* ma_final_drill (u_short grade_threshold);
	/*
	 * Returns pointer to an element that had review today,
	 * but grade was lower than grade_threshold.
	 * Returns NULL if such element not exist,
	 * or if all elements passed final drill
	 *		(all have grade >= grade_threshold).
	 * This is an optional feature: some UI may use "Final drill" feature.
	 */

extern u_short ma_move_el (struct elem*, const u_short grade);
	/*
	 * After a repetition, having new grade, you should call ma_move_el(),
	 * it will move the element in the linked list to its new place.
	 * Returns new interval.
	 *
	 * NOTE:
	 * This function gives ANN some feedback (new data to train ANN),
	 * You may want to train ANN for some time (e.g. 5 seconds)
	 * after calling ma_move_el()
	 * (not necessarily just after calling ma_move_el())
	 * You may want (or may not want) to train ANN anytime.
	 * But training ANN is only useful if ANN have some new data,
	 * and only calls to ma_move_el() gives that new data.
	 */

extern struct elem* ma_find_el (const char*q, const char*a);
	/*
	 * Greps through and finds (returns pointer to) first matched element
	 * (or NULL if not found).
	 * 
	 * 'q' OR 'a' may be equal to '\0' or point to NULL, in this case only
	 * characters of answer OR question will be searched for a match.
	 *
	 * Matching is case InSeNsItIvE.
	 */

extern void ma_stats (struct ma_stats_st*);
	/*
	 * Fills ma_stats_st with data.
	 */

extern u_short ma_rpts_for (u_short since_day, u_short to_day);
	/*
	 * Returns number of repetition for specified day(s),
	 * (specified days are relative to today)
	 * including to_day,
	 * e.g.
	 * ma_n_rpts_for (0,6) will return number of repetitions in next 7 days.
	 */

extern u_short ma_rpts_upto (u_short to_day);
	/*
	 * Returns number of repetitions up to to_day
	 * to_day is relative to 'today'.
	 */

extern int feedback_to_ann (u_short l_ivl, u_short real_l_ivl, u_short rp,
  u_short l_gr, u_short ivl_by_ann, u_short real_act_ivl, u_short act_grade);
	/*
	 * Give feedback to neural network.
         * Generally you don't have to use this function directly.
         *
	 */

extern int ma_export (const char *filename);
	/*
	 * Exports elements (from the learning process) from category catgr_name
	 * to filename, to simple question/answer format
	 *	(no data about learning is exported).
	 */

extern void ma_run_tags (const char *p, int version);
	/*
	 * You don't have to use it. 
	 * This is an extension
	 * allowing to use some tags (currently <exec> tag </exec>.
	 *
	 * Fires up any tags.
	 * version can be: CONSOLE_VERSION, GUI_VERSION, CGI_VERSION.
	 * (e.g. CGI version may don't have to use <exec> tag as it would be
	 * a security hole)
	 */

extern void ma_remove_tags (char *p);
	/*
	 * You don't have to use it.
	 * This is an extension
	 * allowing to use some tags (currently <exec> tag </exec>.
	 *
	 * Removes all tags from *p
	 * (you may want to call it before printing out answer/question in *p,
	 * to show question/answer without any tags)
	 */

extern struct elem* ma_iterate_list (struct elem* current);
	/*
	 * Use this function to iterate through the list one by one
	 * without interfering the sceduling.
	 *
	 * The parameter current gives the last deliverd element to
	 * the function. It returns then the next element in the list.
	 * To get the very first element in the list just deliver
	 * NULL to the current parameter.
	 */

extern int ma_list_is_loaded (void);
	/*
	 * Returns 1 if the database is loaded,
	 * 0 otherwise.
	 */
extern void ma_store (void);
	/*
	 * Stores the list of elements to disk.
	 * This function should be called after adding new
	 * or changing elements in the learning list.
	 */

extern u_long get_tm_strt (void);
	/*
	 * Returns training starting time.
	 */

extern void set_tm_strt (u_long);
	/*
	 * Sets training starting time.
	 * This function is only needed when bypassing ma_init().
	 */

extern void put_el_in_new_place (struct elem* nw);
	/*
	 * Inserts a new element in the list.
	 * This function is only needed when bypassing ma_init().
	 */

extern void set_data_dir (const char* path_to_data_dir);
	/*
	 * Sets the data dir (for NN).
	 * This function is only needed when bypassing ma_init().
	 */


extern void erase_memory_after_ll (void);
	/*
	 * Deinitialises the linked list.
	 * This function is only needed when bypassing ma_deinit().
	 */

#define MEMAID_DESCRIPTION  "MemAid is like a traditional flash-card program \n"\
	"to help you memorise question/answer pairs, but with an important twist: \n"\
	"it uses a neural network to schedule the best time for an item to come up for review.\n"\
	"Difficult items that you tend to forget quickly will be scheduled more often, \n"\
	"while MemAid won't waste your time on things you remember well.\n"\
	"This means your learning process becomes much more efficient, \n"\
	"also because the neural network gradually adapts to your personal memory model."

#define MEMAID_LICENSE "MemAid, a memorisation tool.\n"\
	"Copyright (C) 2003 David Calinski\n"\
	"Portions of the code: Copyright (c) 1996 Karsten Kutza\n\n"\
	"This program is free software; you can redistribute it and/or modify\n"\
	"it under the terms of the GNU General Public License as published by\n"\
	"the Free Software Foundation; either version 2 of the License, or\n"\
	"(at your option) any later version.\n\n"\
	"This program is distributed in the hope that it will be useful,\n"\
	"but WITHOUT ANY WARRANTY; without even the implied warranty of\n"\
	"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the\n"\
	"GNU General Public License for more details.\n\n"\
	"You should have received a copy of the GNU General Public License\n"\
	"along with this program; if not, write to the Free Software\n"\
	"Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA\n"\
	"02111-1307  USA"

#define MEMAID_CORE_THANK_AND_CREDIT_LIST "MemAid core code Thank/Credit list:\n\n"\
"David Calinski - main coding\n"\
"Karsten Kutza - original ANN code\n"\
"Dr Piotr Wozniak - for SuperMemo software, ideas, encouragement to develop ANN idea for sheduling repetitions\n"\
"Andy Shaw - for Input Easy (next software that was inspiration for MemAid), ideas in Input Easy documentation"

#ifdef __cplusplus
}
#endif

#endif /* ifndef __MEM_AID_H__ */

