/*
   MemAid, a memorisation tool.
   Copyright (C) 2003 David Calinski
   Portions of the code: Copyright (c) 1996 Karsten Kutza

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307  USA
*/

#include <ctype.h>
#include <math.h>
#include <stdlib.h>
#include "memaid.h"

#ifdef MEMAID_SELFTEST
#define MEMAID_DEBUG
#endif

static struct elem *List = NULL; /* pointer to the first element of linked list containing elements */
time_t tm_strt;			 /* time of start learning process (the initial run) */


/* returns number of days, since first day
 * (but taken 3:30am), up to tme.
  * If (tme == 0) returns number of days up to today.
 */
static u_short days (time_t tme) {
	time_t night_start;
	struct tm *start = localtime (&tm_strt);
		/* watch out: we operate here on an internal static structure
		 * which is not our.
		 * In a multi-thread app be careful when calling
		 * gmtime(), localtime() and ctime(),
		 * as these functions overwrite this struct.
		 */
	start->tm_hour = 3;
	start->tm_min  = 30;
	start->tm_sec  = 0;

	night_start = mktime (start);
	
	if (tme == 0)
		tme = time (NULL);
	
	/* bad method (not ANSI C), but fast and works on Unix and POSIX compatible O/S
	 * (AFAIK POSIX tells that time_t should type store seconds)
	 * and probably most other O/S */
	return ((unsigned short)((tme - night_start) / 86400L)); /* 86400 secs = 1 day */
	
	/* the right method; slower, but portable. should be tested before replacing method above,
	 * I am not sure if difftime() can return negative numbers. */  
/*	return ((unsigned_short)(((long)difftime(tme, night_start)) / 86400L)); */
}

INLINE u_short ma_n_of_rpts_for_today (void) {
	return ma_rpts_upto(0);
}

INLINE static u_short real_interval (struct elem *e) {
	return (u_short)((signed short)e->ivl + (signed short)(days(0) - e->tm_t_rpt));
}

void put_el_in_new_place (struct elem* nw) {
	register struct elem *li, *tmp;

	if (List == NULL) { /* List is empty, we add first element */
		nw->nxt = NULL;
		List = nw;
		return;
	}

	li = List;
	
	if (List->nxt == NULL) { /* there is just one element */
		if (List->tm_t_rpt < nw->tm_t_rpt) {
			List->nxt = nw;
			nw->nxt = NULL;
		}
		else {
			List = nw;
			nw->nxt = li;
		}
		return;
	}

	if (nw->tm_t_rpt <= li->tm_t_rpt) { /* new element will be first on the list */
		List = nw;
		nw->nxt = li;
		return;
	}
	
	/* we have to add new element between two */
	do {
		if (li->nxt == NULL) { /* end of List */
			li->nxt = nw;
			nw->nxt = NULL;
			return;
		}
		tmp = li;
		li = li->nxt;
	} while (nw->tm_t_rpt > li->tm_t_rpt);
			
	tmp->nxt = nw;
	nw->nxt = li;
	return;
}

u_short ma_make_new_el (u_short grade, const char *q, const char *a) {
	struct elem *nw;
	u_short tme, interval;	

	interval = ma_new_interval (0U, 0U, 0U, grade);
	tme = days(0) + interval;
	
	nw = (struct elem*) malloc (sizeof (struct elem));
	nw->tm_t_rpt  = tme;
	nw->stm_t_rpt = tme;
	nw->l_ivl     = 0U;
	nw->rl_l_ivl  = 0U;
	nw->ivl       = interval;
	nw->rp        = 0U;
	nw->gr        = grade;
	nw->id        = (u_long)time(NULL); /* XXX This is not right, unportable.
					     * (well, portable to at least POSIX compatible systems,
					     * as AFAIK POSIX requires time_t to store seconds.
					     */
	nw->q         = (char*) malloc ((strlen (q)+1) * sizeof(char));
	nw->a         = (char*) malloc ((strlen (a)+1) * sizeof(char));
	strcpy (nw->q,  q);
	strcpy (nw->a,  a);

	put_el_in_new_place (nw);
	return interval;
}

void ma_delete_el (struct elem *li) {
	register struct elem *prev;

	if (li == List) { /* delete first element */
		List = List->nxt;
		free (li->q);
		free (li->a);
		free (li);
		return;
	}

	prev = List;

	if (prev->nxt != li)
		do
			prev = prev->nxt;
		while (prev->nxt != li);

	prev->nxt = li->nxt;

	free (li->q);
	free (li->a);
	free (li);
	return;
}

int ma_list_n_of_els (void) {
	register int i = 0;
	register struct elem *e = List;

	if (e == NULL)
		return 0;
	do
		++i;
	while ((e = e->nxt) != NULL);
	
	return i;
}

/* moves the first element (user had just repetition from it)
 * to it's new place in the linked List
 */
static void move (struct elem *el, u_short ninterval, u_short ngrade) {
	struct elem *prv;
	u_short rl_interval = real_interval(el);
	u_short ntime = days(0) + ninterval;

	if (el->stm_t_rpt <= days(0)) { /* if not ahead-of-scheduled time repetition */
		if (el->rp < MA_MAX_N_OF_RPTS)
			++el->rp;
		el->stm_t_rpt = ntime;
	}
	
	el->tm_t_rpt = ntime;
	el->l_ivl = el->ivl;
	el->rl_l_ivl = rl_interval;
	el->ivl = ninterval;
	el->gr = ngrade;

	if (List->nxt == NULL) {
		assert (List == el);
		return;	/* nothing to do, there is only 1 element in the List */
	}

	if (List == el) /* first element (the head of ll) to move */
		List = el->nxt;
	else {
		prv = List;
		while (prv->nxt != el)
			prv = prv->nxt;
		prv->nxt = el->nxt;
	}
	
	/* "el" is "free" (not linked) here. We are searching for a new place.
	 * We know that there is (at least) one element in the list at moment.
	 */

	put_el_in_new_place (el);
	return;
}

/* get new element to repeat (if exists and is scheduled for repetition).
 */
struct elem *ma_el_to_repeat (void) {
	u_short i;
	struct elem *el = List;

	/* BTW: new interval must always be > 0,
	 * otherwise we would stuck in an endless loop.
	 * - it's an old comment.. I'm not sure if still valid. 
	 */

	if (List == NULL || (i = ma_rpts_upto (0)) == 0)
		return NULL;

	/* Try to return a bit in a random way.
	 * Lack of randomness here does help remembering
	 * by associations of items with each other, but 
	 * it's not really good for "really" remembering facts.
	 *
	 * NOTE: we want just *a bit* of randomness;
	 * first items on the list are more important to repeat ASAP.
	 */

	if (i > 1) {
		i = ma_rand(i);
		if (i > 1)
			i = ma_rand (i);
		while (i) {
			el = el->nxt;
			--i;
		}
	}
	return el;
}

/* get a element to repeat even if not scheduled for repeat today
*/
struct elem *ma_el_force_repeat (void) {
	struct elem *el = List;
	int i;

	if (List == NULL)
		return NULL;

	i = ma_rand (ma_list_n_of_els());
	if (i > 1)
		i = ma_rand(i); /* we prefer the closer */
	if (i > 1)
		i = ma_rand(i); /* do it once more */
	if (i > 0)
		do
			el = el->nxt;
		while (--i > 0);

	return el;
}

#define MAX_FINAL_DRILL_RAND_EL 5

struct elem *ma_final_drill (u_short grade_threshold) {
	u_short today = days(0);
	struct elem *el = List;
	struct elem *e_rand[MAX_FINAL_DRILL_RAND_EL];
	int found = 0;

	/* find an element that had review today and obtained grade
	 * was lower than grade_threshold
	 */
	
	if (el == NULL)
		return NULL;

	do
		if ((el->tm_t_rpt - el->ivl) == today && el->gr < grade_threshold) {
			e_rand[found] = el;
			if (++found >= MAX_FINAL_DRILL_RAND_EL)
				break;
		}
	while ((el = el->nxt) != NULL);
	
	return (found ? e_rand[ ma_rand(found) ] : NULL);
}

static INLINE void str_to_up (char *s) {
	if (s == NULL || *s == '\0')
		return;
	do
		*s = toupper(*s);
	while (*++s != '\0');
}

static int str_case_str (const char *s0, const char *s1) {
	char t0[MAX_A_LEN], t1[MAX_A_LEN];
	
	if (*s0 == '\0' || s0 == NULL) /* 's1' was already checked in ma_find_el() */
		return 0;

	strncpy (t0, s0, MAX_A_LEN-1);
	strncpy (t1, s1, MAX_A_LEN-1);
	t0[MAX_A_LEN-1] = '\0';
	t1[MAX_A_LEN-1] = '\0';
	str_to_up (t0);
	str_to_up (t1);
	
	return ((strstr (t0, t1) != NULL) ? 1 : 0);
}

struct elem *ma_find_el (const char *q, const char *a) {
	register struct elem *li = List;
	
	if (q != NULL && *q != '\0') {
		if (li != NULL)
			do
				if (str_case_str (li->q, q)) goto after_q;
			while ((li = li->nxt) != NULL);
		return NULL;
	}
after_q:
	if (a != NULL && *a != '\0') {
		if (li != NULL) 
			do
				if (str_case_str (li->a, a)) goto after_a;
			while ((li = li->nxt) != NULL);
		return NULL;
	}
after_a:
	return li;
}

static void save_bin (char *filename) {
	register struct elem *e = List;
	FILE *fout;
	u_short len;

	if (List == NULL && days(0) == 0)
		return;
	if (!(fout = fopen (filename, "wb"))) {
		perror ("Critical error: cannot save elements.bin");
		ma_error ("Critical error: cannot save %s\n", filename);
		return;
	}

	len = (u_short) MA_CORE_VERSION;
	fwrite (&len, sizeof(u_short), 1, fout);
	fwrite (&tm_strt, sizeof(time_t), 1, fout);
	do {
		fwrite (&e->tm_t_rpt,  sizeof(u_short), 1, fout);
		fwrite (&e->stm_t_rpt, sizeof(u_short), 1, fout);
		fwrite (&e->id, sizeof(u_long), 1, fout);
		
		len = strlen (e->q) + 1; /* q */
		fwrite (&len, sizeof (u_short), 1, fout);
		fwrite (e->q, sizeof(char), len, fout);
		
		len = strlen (e->a) + 1; /* a */
		fwrite (&len, sizeof (u_short), 1, fout);
		fwrite (e->a, sizeof(char), len, fout);
		
		fwrite (&e->l_ivl,    sizeof(u_short), 1, fout);
		fwrite (&e->rl_l_ivl, sizeof(u_short), 1, fout);
		fwrite (&e->ivl,      sizeof(u_short), 1, fout);
		fwrite (&e->rp,       sizeof(u_short), 1, fout);
		fwrite (&e->gr,       sizeof(u_short), 1, fout);
	} while ((e = e->nxt) != NULL);
	fclose (fout);
	return;
}

void erase_memory_after_ll (void) {
	register struct elem *li = List;

	if (li == NULL)
		return;
	do {
		List = li->nxt;
		free (li->q);
		free (li->a);
		free (li);
	} while ((li = List) != NULL);
	return;
}

static void bin_error (char *filename) {
	ma_error ("Serious ERROR while reading binary file format detected.\n"
		"\tPossible reasons:\n"
		"\t1. Bug in MemAid core related to load/save binary file format routines.\n"
		"\t2. You are trying to read *badly* saved file by another MemAid backend.\n"
		"\t3. On last MemAid session data was not full saved (e.g. sudden lack of power supply while writting database\n"
		"Therefore I can't read %s, I will try to load database from XML format.\n", filename);
	erase_memory_after_ll ();
	tm_strt = time (NULL);
	return;
}

static int load_bin (char *filename) {
	struct elem *p = NULL, *e;
	FILE *fin;
	u_short len = 0U;

	if (List != NULL) {
		ma_error ("load_bin(): List is not empty! I will call erase_memory_after_ll() myself.\n");
		erase_memory_after_ll();
	}
	if (!(fin = fopen (filename, "rb"))) {
		tm_strt = time (NULL);
		return 1;
	}
	if (fread (&len, sizeof (u_short), 1, fin) != 1 || len != (u_short)MA_CORE_VERSION) {
		printf ("Old format of binary file detected in %s\n", filename);
		tm_strt = time(NULL);
		fclose (fin);
		return 1;
	}
	fread (&tm_strt, sizeof (time_t), 1, fin);
	e = (struct elem*) malloc (sizeof(struct elem));
	List = e;
	len = 0U;

	while ((fread (&e->tm_t_rpt,  sizeof(u_short), 1, fin) == 1)) {
		fread (&e->stm_t_rpt, sizeof(u_short), 1, fin);
		fread (&e->id,        sizeof(u_long),  1, fin);
		
		fread (&len, sizeof(u_short), 1, fin); /* q */
		e->q = (char*) malloc (len * sizeof(char)); 
		fread (e->q, sizeof(char), len, fin);
		
		fread (&len, sizeof(u_short), 1, fin); /* a */
		e->a = (char*) malloc (len * sizeof(char));
		fread (e->a, sizeof(char), len, fin);
		
		fread (&e->l_ivl,    sizeof(u_short), 1, fin);
		fread (&e->rl_l_ivl, sizeof(u_short), 1, fin);
		fread (&e->ivl,      sizeof(u_short), 1, fin);
		fread (&e->rp,       sizeof(u_short), 1, fin);
		if (fread (&e->gr,   sizeof(u_short), 1, fin) != 1) {
			e->nxt = NULL;
			bin_error(filename);
			return 1;
		}
		e->nxt = (struct elem*) malloc (sizeof(struct elem));
		p = e;
		e = e->nxt;
	}
	if (len) /* something has been read from the file */
		p->nxt = NULL;
	else
		List = NULL;
	free (e);
	fclose (fin);
	return 0;
}

/*
static int save_ascii (char *filename) {
	register struct elem *e = List;
	FILE *fout;

	if (List == NULL && days(0) == 0) {
		assert (ma_list_n_of_els() == 0);
		return 1;
	}

	if (!(fout = fopen (filename, "w"))) {
		ma_error ("save_ascii(): Cannot write/create/access %s\n", filename);
		return 2;
	}
	
	fputs (" This file contains human readable equivalent of elements.bin\n", fout);
	fputs (" Normally MemAid creates this file every time it quits, "\
		"and doesn't load it at all\n", fout);
	fputs (" So it's only for you - you can see how you learning data look like.\n", fout);
	fputs (" However you have possibility to force MemAid "\
		"to load data from this file\n", fout);
	fputs (" You can do that by deleting elements.bin\n\n", fout);
	fputs (" WARNING: PLEASE be very careful and do a back-up copy "\
		"before doing it!\n\n", fout);
	fputs ("\tLEGEND:\n\n", fout);
	fputs ("\ttm_strt=[first_day_of_your_MemAid_learning_in_Unix_time_format]\n\n", fout);
	fputs ("\tItem data:\n", fout);
	fputs ("\ttm_t_rpt=[time_to_repetition_of_an_item]\n", fout);
	fputs ("\tstm_t_rpt=[time_to_repetition_of_an_item_computed_on_a_scheduled_review]\n", fout);
	fputs ("\tid=[id_string (a note about item; category, name, etc.)]\n", fout);
	fputs ("\tq=<QUESTION>question</QUESTION>\n", fout);
	fputs ("\ta=<ANSWER>answer</ANSWER>\n", fout);
	fputs ("\tl_ivl=[last_interval_computed_by_ann (in days: 0-2048)]\n", fout);
	fputs ("\trl_l_ivl=[real_last_interval (in days: 0-2048)]\n", fout);
	fputs ("\tivl=[interval (in days: 0-2048)]\n", fout);
	fputs ("\trp=[number_of_repetitions(0-128)]\n", fout);
	fputs ("\tgr=[grade(0-5), 0=the worst, 5=the best]\n\n", fout);
#ifdef MEMAID_DEBUG
	fprintf (fout, "Debug info:\n");
	fprintf (fout, "days since first use=%hu\n", (unsigned short) ((time(NULL) - tm_strt) / 86400L));
	fprintf (fout, "days since first use (according to days(), may be +/-1)=%hu\n\n",
			(unsigned short) days(0));
#endif
	fputs ("~  ~  ~  ~  ~  ~  ~  ~  ~  ~  ~  ~  ~  ~\n\n", fout);
	fprintf (fout, "tm_strt = [%ld]\n\n", (long int)tm_strt);

	do {
		fprintf (fout, "tm_t_rpt=[%hu]\n",  e->tm_t_rpt);
		fprintf (fout, "stm_t_rpt=[%hu]\n", e->stm_t_rpt);
		fprintf (fout, "id=[%lu]\n", (u_long)e->id);
		fprintf (fout, "q=<QUESTION>%s</QUESTION>\n", e->q);
		fprintf (fout, "a=<ANSWER>%s</ANSWER>\n", e->a);
		fprintf (fout, "l_ivl=[%hu]\n", e->l_ivl);
		fprintf (fout, "rl_l_ivl=[%hu]\n", e->rl_l_ivl);
		fprintf (fout, "ivl=[%hu]\n", e->ivl);
		fprintf (fout, "rp=[%hu]\n", e->rp);
		fprintf (fout, "gr=[%hu]\n\n", e->gr);
		fprintf (fout, "*  *  *  *  *  *  *  *  *  *  *  *  *  *\n\n");
		e = e->nxt;
	} while (e != NULL);

	fprintf (fout, "\nEOF");
	fclose (fout);
	return 0;
} */

/* reads a number from string
 */
static unsigned long read_a_number (register char *s) {
	register unsigned long result = 0;
	char *original = s;
	
	if (!isdigit (*s))
		do
			++s;
		while (!isdigit(*s) && *s != '\0');
	
	if (*s == '\0') {
		ma_error ("read_a_number(): cannot find any digit in string: %s\n", original);
		return 0U;
	}
	
	do
		result = 10 * result + *s - '0';
	while (isdigit (*++s));

	return result;
}

static char *extract_cdata (register char *s, FILE *f) {
	register char *p;
	char addbuf[MAX_A_LEN]; /* MAX_A_LEN is >= MAX_Q_LEN */

	while (!(p=strstr (s, "<![CDATA[")) && !feof(f))
		fgets (s, MAX_A_LEN, f);
	
	if ((p=strstr (s, "<![CDATA["))) {
		while (!strstr (s, "]]>") && fgets (addbuf, MAX_Q_LEN, f))
			strcat (s, addbuf);
		if ((p=strstr (s, "]]>")))
			*p = '\0';
		else {
			ma_error ("extract_cdata(): Fatal error while loading a question.\n");
			return NULL;
		}
		return (strstr (s, "<![CDATA[") + strlen ("<![CDATA["));
	}
	if (feof(f))
		ma_error ("ERROR: extract_cdata() hit EOF, your elements.xml file doesn't look well...\n");
	return NULL;
}

static int save_xml (char *filename) {
	register struct elem *e = List;
	FILE *fout;

	if (List == NULL && days(0) == 0)
		return 1;

	if (!(fout = fopen (filename, "w"))) {
		ma_error ("save_xml(): Cannot write/create/access %s\n", filename);
		return 2;
	}

	fputs ("<?xml version=\"1.0\" ?>\n"
		"<!DOCTYPE memaid [\n"
		"\t<!ELEMENT memaid (item)>\n"
		"\t<!ELEMENT item (Q, A)>\n"
		"\t<!ELEMENT Q (#PCDATA)>\n"
		"\t<!ELEMENT A (#PCDATA)>\n"
		"\t<!ATTLIST memaid core_version CDATA #IMPLIED\n"
			"\t\ttime_of_start CDATA #REQUIRED>\n"
		"\t<!ATTLIST item id ID #REQUIRED\n"
			"\t\ttm_t_rpt CDATA #REQUIRED\n"
			"\t\tstm_t_rpt CDATA #REQUIRED\n"
			"\t\tl_l_ivl CDATA #REQUIRED\n"
			"\t\trl_ivl CDATA #REQUIRED\n"
			"\t\tivl CDATA #REQUIRED\n"
			"\t\trp CDATA #REQUIRED\n"
			"\t\tgr CDATA #REQUIRED>\n"
	"]>\n\n", fout);
	fputs ("<!-- MemAid saves both elements.bin (binary, very fast and efficent format)     -->\n"
		"<!-- and elements.xml (this file, contains the same data, but in XML format)    -->\n"
		"<!-- MemAid reads elements.xml (this file) ONLY when elements.bin doesn't exist -->\n"
		"<!-- By default (if both files are available) MemAid reads only elements.bin    -->\n"
		"<!-- Therefore, obviously you need to delete (or just rename) elements.bin      -->\n"
		"<!-- if you want to force MemAid to read this file                              -->\n\n",
		fout);

	fputs ("<!-- LEGEND:                                                                  -->\n"
		"<!--  time_of_start = first day of your MemAid learning (in Unix time format) -->\n"
		"<!--  ITEM DATA:                                                              -->\n"
		"<!--   id = id number (in fact: time of creating item in Unix time format)    -->\n"
		"<!--   tm_t_rpt = time to repeat                                              -->\n"
		"<!--   stm_t_rpt = time to repeat computed on a *scheduled* (or late) review  -->\n",fout);
	fputs ("<!--   l_ivl = last interval computed by ANN (in days: 0-2048)                -->\n"
		"<!--   rl_l_ivl = real last interval (in days: 0-2048)                        -->\n"
		"<!--   ivl = interval (in days: 0-2048)                                       -->\n"
		"<!--   rp = number of repetitions (0-128)                                     -->\n"
		"<!--   gr = grade (0-5, 0=the worst, 5=the best)                              -->\n"
		"<!--   Q = question, A = answer                                               -->\n\n\n",fout);

#ifdef MEMAID_DEBUG
	fprintf (fout, "<!-- Debug info: -->\n");
	fprintf (fout, "<!--  days since first use=\"%hu\" -->\n",
			(unsigned short) ((time(NULL) - tm_strt) / 86400L));
	fprintf (fout, "<!--  days since first use (according to days(), may be +/-1)=\"%hu\" -->\n\n\n",
			(unsigned short) days(0));
#endif

	fprintf (fout, "<memaid core_version=\"%d\" time_of_start=\"%ld\">\n\n\n", 
			(int)MA_CORE_VERSION, (long int)tm_strt);
	
	do
		fprintf (fout, "<item id=\"%lu\" tm_t_rpt=\"%hu\""
			" stm_t_rpt=\"%hu\""
			" l_ivl=\"%hu\""
			" rl_l_ivl=\"%hu\""
			" ivl=\"%hu\""
			" rp=\"%hu\""
			" gr=\"%hu\">\n"
			"<Q><![CDATA[%s]]></Q>\n"
			"<A><![CDATA[%s]]></A>\n</item>\n\n",
				e->id, e->tm_t_rpt,
				e->stm_t_rpt,
				e->l_ivl,
				e->rl_l_ivl,
				e->ivl,
				e->rp,
				e->gr,
				e->q,
				e->a);
	while ((e = e->nxt) != NULL);

	fputs ("</memaid>\n", fout);
	fclose (fout);
	return 0;
} /* save_xml() */ 

/* findstr(s0, s1) is a strstr(s0, s1) variation that search for s1 is s0,
 * but until first '<' character.
 */
static char *findstr (const char *s0, const char *s1) {
	char *p;

	p = strstr (s0, s1);
	if (p == NULL || strchr (s0, '<') < p)
		return NULL;
	return p;
}

/* load_xml() loads data from elements.xml
 * Theoretically here we should a 'real' XML parser, I think.
 * Practically: it would bloat code a lot,
 * and I say 'no' to external XML parser library: no need to add dependencies.
 * So, we use a simplified aproach.
 */

#define BUFLEN (MAX_A_LEN*2)

static int load_xml (char *filename) {
	FILE *fin;
	struct elem *e = NULL;
	char buf[BUFLEN], *p, *pos;
	char item_open = 0, comment_on = 0; /* booleans */
	register int i;

	if (!(fin = fopen (filename, "r")))
		return 1;
	
	erase_memory_after_ll ();
	i = 0;
	/* 'buf' is our buffer
	 * 'pos' is our current position in buffer - we work on it
	 * 'p' is an additonal pointer that sometimes is setted up ahead, sometimes points to NULL
	 */
	while (fgets (buf, BUFLEN, fin)) {
		if (strncmp (buf, "<!--", strlen("<!--")) == 0)
			comment_on = 1;
		if (comment_on) {
			if ((p = strstr (buf, "-->"))) {
				pos = p + strlen("-->");
				comment_on = 0;
			}
			else
				continue;
		}
		else
			pos = buf;
		if (i == 0) {
			if ((p = strstr (pos, "<memaid"))) {
				pos = p + strlen ("<memaid"); 
				++i;
			}
			else /* haven't encountered <memaid so don't search for other things */
				continue;
		}
		if ((p = strstr (pos, " core_version="))) {
			pos = p + strlen (" core_version=");
			if (read_a_number (pos) > MA_CORE_VERSION) {
				fprintf (stderr, "ERROR:\n"
				"your \'MemAid core version\' = %d, while elements.xml file I am just reading"
				" was saved using %d version...\n"
				"I am NOT prepared to this version! Please upgrade me...\n",
				(int)MA_CORE_VERSION, (int)read_a_number(pos));
				return 1;
			}
		}
		if ((p = strstr (pos, " time_of_start="))) {
			tm_strt = read_a_number (p + strlen(" time_of_start="));
			break;
		}
	}	

	if (feof(fin)) {
		ma_error ("Fatal error while reading %s\n", filename);
		fclose (fin);
		return 1;
	}

	i = 0;
	while (fgets (buf, BUFLEN, fin)) {
		if (strncmp (buf, "<!--", strlen("<!--")) == 0)
			comment_on = 1;
		if (comment_on) {
			if ((p = strstr (buf, "-->"))) {
				pos = p + strlen("-->");
				comment_on = 0;
			}
			else
				continue;
		}
		else
			pos = buf;
		
		if (item_open == 0 && !strstr (pos, "<item "))
				continue;
		item_open = 1;
		
		if ((p=strstr(pos, " id="))) {
			pos = p + strlen (" id=");
			e = (struct elem*) malloc (sizeof(struct elem));
			assert (e);
			e->id = (time_t)read_a_number(pos);
		}
		if ((p=strstr(pos, " tm_t_rpt="))) {
			pos = p + strlen (" tm_t_rpt=");
			e->tm_t_rpt = (u_short) read_a_number(pos);
		}
		if ((p=strstr(pos, " stm_t_rpt="))) {
			pos = p + strlen (" stm_t_rpt=");
			e->stm_t_rpt = (u_short) read_a_number (pos);
		}
		if ((p=strstr(pos, " l_ivl="))) {
			pos = p + strlen (" l_ivl=");
			e->l_ivl = (u_short) read_a_number (pos);
		}
		if ((p=strstr(pos, " rl_l_ivl="))) {
			pos = p + strlen (" rl_l_ivl=");
			e->rl_l_ivl = (u_short) read_a_number (pos);
		}
		if ((p=strstr(pos, " ivl="))) {
			pos = p + strlen (" ivl=");
			e->ivl = (u_short) read_a_number (pos);
		}
		if ((p=strstr(pos, " rp="))) {
			pos = p + strlen (" rp=");
			e->rp = (u_short) read_a_number (pos);
		}
		if ((p=strstr(pos, " gr="))) {
			pos = p + strlen (" gr=");
			e->gr = (u_short) read_a_number (pos);
		}
		if ((p=findstr (pos, "<Q>"))) {
			char *ptr = extract_cdata(p, fin);
			if (ptr == NULL) {
				ma_error ("Aborting reading %s\n", filename);
				fclose (fin);
				return 1;
			}
			e->q = (char*) malloc (sizeof(char) * (strlen(ptr)) + 1);
			assert (e->q);
			strcpy (e->q, ptr);
			continue;
		}
		if ((p=findstr (pos, "<A>"))) {
			char *ptr = extract_cdata(p, fin);
			if (ptr == NULL) {
				ma_error ("Aborting reading %s\n", filename);
				fclose (fin);
				return 1;
			}
			e->a = (char*) malloc (sizeof(char) * (strlen(ptr)) + 1);
			assert (e->a);
			strcpy (e->a, ptr);
			continue;
		}
		if (strstr (pos, "</item>")) {
			put_el_in_new_place (e);
			item_open = 0;
		}
	}
	fclose (fin);
	return 0;
}

static void save_database (void) {
	save_bin   (pathed ("elements.bin"));	/* saves linked list with elements data (binary file) */
	save_xml   (pathed ("elements.xml"));   /* saved in XML format */
	/* save_ascii (pathed ("elements.txt")); // saves also in txt mode, in another file */
}

static void ll_deinit (void) {
	save_database();
	erase_memory_after_ll();
}

void ma_store (void) {
	save_database();
	nn_save_weights_to_file();
}

#define MIN(x,y) ((x < y) ? x : y)
#define MAX(x,y) ((x > y) ? x : y)
#define GRADE_THRESHOLD 3

void ma_stats (register struct ma_stats_st *st) {
	register struct elem *e = List;
	u_long sum_ivl = 0, sum_l_ivl_real = 0, sum_l_ivl_ann = 0;
	u_long sum_grades = 0, sum_forgot = 0;
	u_short today;

	memset (st, 0, sizeof (struct ma_stats_st));
	
	if (List == NULL)
		return;

	today = days(0);

	st->min_ivl        = MA_MAX_INTERVAL;
	st->min_l_ivl_real = MA_MAX_INTERVAL;
	st->min_l_ivl_ann  = MA_MAX_INTERVAL;
	
	do {
		++st->total_items;

		if (e->tm_t_rpt <= today)
			++st->rpts_for_today;

		st->min_ivl        = MIN (st->min_ivl,        e->ivl);
		st->min_l_ivl_real = MIN (st->min_l_ivl_real, e->rl_l_ivl);
		st->min_l_ivl_ann  = MIN (st->min_l_ivl_ann,  e->l_ivl);

		st->max_ivl        = MAX (st->max_ivl,        e->ivl);
		st->max_l_ivl_real = MAX (st->max_l_ivl_real, e->rl_l_ivl);
		st->max_l_ivl_ann  = MAX (st->max_l_ivl_ann,  e->l_ivl);

		st->max_reviews    = MAX (st->max_reviews, e->rp);
		st->sum_reviews   += e->rp;
		sum_ivl           += e->ivl;
		sum_l_ivl_real    += e->rl_l_ivl;
		sum_l_ivl_ann     += e->l_ivl;
		sum_grades        += e->gr;
		if (e->gr < GRADE_THRESHOLD || e->rl_l_ivl == 0)
			++sum_forgot;
			/* if e->rl_l_ivl == 0
			 * then most probably last review was on final drill,
		 	 * so real grade was < GRADE_THRESHOLD
			 */
	} while ((e = e->nxt) != NULL);

	st->mean_ivl         = sum_ivl        / st->total_items;
	st->mean_l_ivl_real  = sum_l_ivl_real / st->total_items;
	st->mean_l_ivl_ann   = sum_l_ivl_ann  / st->total_items;

	st->mean_reviews     = (float)st->sum_reviews / (float)st->total_items;
	st->mean_grade       = (float)sum_grades / (float)st->total_items;

	st->days	     = days(0);
	
	st->ann_error        = test_net();
	st->ann_cases        = NData;
	st->user_ann_cases   = NData - NData_general;

	st->retention        = 100.0f * ((float) (st->total_items - sum_forgot) / (float)st->total_items);
	st->item_per_day     = (float)st->total_items / (float)st->days;
	st->first_run        = tm_strt;
	return;
}

u_short ma_rpts_upto (u_short to_day) { /* to_day is relative to 'today' */
	register u_short count = 0;
	register struct elem *e = List;

	if (e == NULL)
		return 0;
	
	to_day = days(0) + to_day;
	
	if (e->tm_t_rpt > to_day)
		return count;
	
	do
		++count;
	while ((e = e->nxt) != NULL && e->tm_t_rpt <= to_day);
	
	return count;
}
u_short ma_rpts_drill (u_short grade_threshold) { /* bruno: how many to drill  */
	register u_short count = 0;
	register struct elem *e = List;

	if (e == NULL)
		return 0;
	u_short to_day;
	to_day = days(0) ;
	
	/*if (e->tm_t_rpt - e->ivl > to_day )
		return count;*/
	
	do
	if (e->tm_t_rpt - e->ivl == to_day && e->gr < grade_threshold)
		++count;
	while ((e = e->nxt) != NULL );
	return count;
}
	
u_short ma_rpts_for (u_short since_day, u_short to_day) { /* since_day and to_day are relative to 'today' */
	register u_short count = 0;
	register struct elem *e = List;

	if (e == NULL)
		return 0;
	
	since_day = days(0) + since_day;
	to_day    = days(0) + to_day;

	if (e->tm_t_rpt < since_day)
		do 
			e = e->nxt;
		while (e != NULL && e->tm_t_rpt < since_day);
	
	if (e == NULL || e->tm_t_rpt > to_day)
		return count;
	
	do
		++count;
	while ((e = e->nxt) != NULL && e->tm_t_rpt <= to_day);
	
	return count;
}

static char *extract_qa (char *s, FILE *f) {
	register char *p;
	char addbuf[MAX_A_LEN]; /* MAX_A_LEN is >= MAX_Q_LEN */

	if ((p=strstr (s, "<QUESTION>"))) {
		while (!strstr (s, "</QUESTION>") && fgets (addbuf, MAX_Q_LEN, f))
			strcat (s, addbuf);
		if ((p=strstr (s, "</QUESTION>")))
			*p = '\0';
		else {
			ma_error ("extract_qa(): Error while loading a question.\n");
			exit (20);
		}
		return (strstr (s, "<QUESTION>") + 10);
	}
	if ((p=strstr (s, "<ANSWER>"))) {
		while (!strstr (s, "</ANSWER>") && fgets (addbuf, MAX_A_LEN, f))
			strcat (s, addbuf);
		if ((p=strstr (s, "</ANSWER>")))
			*p = '\0';
		else {
			ma_error ("extract_qa(): Error while loading an answer...\n");
			exit (21);
		}
		return (strstr (s, "<ANSWER>") + 8);
	}
	return "";
}

static int load_ascii (char *filename) {
	struct elem *e = NULL;
	FILE *fin;
	char buf[MAX_A_LEN], *pos;
	register int i = 0;

	if (!(fin = fopen (filename, "r")))
		return 1;
	
	erase_memory_after_ll ();

	while (fgets (buf, MAX_A_LEN, fin)) {
		if (strncmp (buf, "tm_strt", 7) != 0)
			continue;
		tm_strt = read_a_number (buf);

#define EQ(x) (strncmp(buf,x,strlen(x))==0)

		while (fgets (buf, MAX_A_LEN, fin)) {
			switch (i) {
			case 0:
				if (EQ("tm_t_rpt")) {
					e = (struct elem*) malloc (sizeof(struct elem));
					assert (e);
					e->id = (time_t)0;
					e->tm_t_rpt = (u_short) read_a_number(buf);
					
					/* to handle old format without stm_t_rpt */
					e->stm_t_rpt = e->tm_t_rpt; 
					
					++i;
				}
				break;
			case 1:
				if (EQ("stm_t_rpt")) {
					e->stm_t_rpt = (u_short) read_a_number(buf);
					break;
				}
				if (EQ("id=[")) {
					char *end, *bg = strstr (buf, "id=[");
					bg += strlen ("id=[");
					if (!(end = strchr (bg, ']')))
						break;
					*end = '\0';
					if (isdigit (*bg))
						e->id = (time_t) read_a_number(buf);
					else {
						ma_error("an id field in one of your elements"
							" in elements.txt contains an non-digit character!\n"
							"I will ignore that id (%s)\n", bg);
						e->id = (time_t) 0;
					}
					break;
				}
				if (EQ("q=<QUESTION>")) {
					pos = extract_qa (buf, fin);
					if (*pos == '\0')
						ma_error ("extract_qa(): you have an empty question\n");
					e->q = (char*) malloc (sizeof(char) * (strlen(pos) + 1));
					assert (e->q);
					strcpy (e->q, pos);
					++i;
				}
				break;
			case 2:
				if (EQ("a=<ANSWER>")) {
					pos = extract_qa (buf, fin);
					if (*pos == '\0')
						ma_error ("extract_qa: you have an empty answer!\n");
					e->a = (char*) malloc (sizeof(char) * (strlen(pos) + 1));
					assert (e->a);
					strcpy (e->a, pos);
					++i;
				}
				break;
			case 3:
				if (EQ("l_ivl=[")) {
					e->l_ivl = (u_short) read_a_number(buf);
					/* handle old format (from MemAid < 0.4.0 era)
					 * without rl_l_ivl:
					 */
					e->rl_l_ivl = e->l_ivl;
					++i;
				}
				break;
			case 4:
				if (EQ("rl_l_ivl=[")) {
					e->rl_l_ivl = (u_short) read_a_number(buf);
					break;
				}
				if (EQ("ivl=[")) {
					e->ivl = (u_short) read_a_number(buf);
					break;
				}
				if (EQ("rp=[")) {
					e->rp = (u_short) read_a_number(buf);
					++i;
				}
				break;
			case 5:
				if (EQ("gr=[")) {
					e->gr = (u_short) read_a_number(buf);
					/* fix for a bug that was in MemAid-0.4.5: */
					if (e->rp == 0 &&
						(e->stm_t_rpt == 0 || e->stm_t_rpt >= MA_MAX_INTERVAL))
							e->stm_t_rpt = e->tm_t_rpt;
					put_el_in_new_place (e);
					i = 0;
				}
				break;
			}
		}
	}
	fclose (fin);
	return 0;
}

/* a little all-in-one to make easy API:
 * 1. get (and return) ma_new_interval
 * 2. move element in the linked list
 */
u_short ma_move_el (struct elem *e, const u_short grade) {
	u_short i;
	
	feedback_to_ann (e->l_ivl, e->rl_l_ivl, e->rp, e->gr, e->ivl, real_interval(e), grade);
	i = ma_new_interval (e->ivl, real_interval(e), e->rp+1, grade);

	/* remember to always call feedback_to_ann() _BEFORE_ move()
	 * as move() equate e->gr with grade
	 */
	move (e, i, grade);
	return i;
}

#define DUMB_FILE "dumbfile" /* this plays also as "lock" file to not allow
			      * other copies of MemAid run at the same time
			      */
static int dir_ok (void) {
	FILE *ftemp;
	
	if ((ftemp = fopen (pathed (DUMB_FILE), "r")) != NULL)
		return 1; /* MemAid seems already running */

	if ((ftemp = fopen (pathed (DUMB_FILE), "wb")) == NULL)
		return 2; /* cannot access directory */

	fclose (ftemp);
	return 0;
}	

static void remove_dumb_file (void) {
	if (remove (pathed (DUMB_FILE)))
		perror ("Error while deleting lock file");
	return;
}

void set_data_dir (const char* path_to_data_dir) {
	const char default_path[] = "data/";
	int i;

	if (path_to_data_dir == NULL)
		strcpy (ma_Path, default_path);
	else {
		i = strlen (path_to_data_dir);
		if (i >= MAX_PATH_LEN - 1) {
			ma_error ("ERROR in ma_init(): path_to_data_dir too long\n");
			exit (1);
		}
		strcpy (ma_Path, path_to_data_dir);
		if (ma_Path[i-1] != '/')
			strcat (ma_Path, "/");
	}
	return;
}

int ma_init (const char *path_to_data_dir) {
	int i;

	set_data_dir (path_to_data_dir);
	switch (dir_ok()) {
		case 0: /* ok */
			break;
		case 1:
			fprintf (stderr, "ERROR: Seems a copy of MemAid is already running\n"
				"\ttry deleting %s if this is not true "
				"(e.g. previous copy of MemAid crashed)\n",
				pathed (DUMB_FILE));
			return MEMAID_ALREADY_RUNNING;
		case 2:
			fprintf (stderr, "ERROR in ma_init(): dir (%s) not accessible!\n", ma_Path);
			return DIR_NOT_ACCESSIBLE;
	}
#ifdef MEMAID_DEBUG
	F_debug = fopen (pathed ("debug.txt"), "a");
	if (!F_debug) {
		fputs ("ERROR: cannot create/access debug.txt\n", stderr);
		F_debug = stderr;
	}
#endif
	i = 0;

	if (nn_init())
		i = i | NN_DATA_DOES_NOT_EXIST;
	if (load_bin (pathed ("elements.bin"))) {
		i = i | BIN_DATA_DOES_NOT_EXIST;
		if (load_xml (pathed ("elements.xml"))) {
			i = i | XML_DATA_DOES_NOT_EXIST;
			if (load_ascii (pathed ("elements.txt")))
				i = i | ASCII_DATA_DOES_NOT_EXIST;
		}
	}
	return i;
}

void ma_deinit (void) { /* deinitialize */
	if (List == NULL)
		goto end;
	ll_deinit();
	nn_deinit(); /* deinitialize neural network */
end:
#ifdef MEMAID_DEBUG
	if (F_debug != stderr)
		fclose (F_debug);
#endif
	remove_dumb_file(); /* remove a "lock" */
	return;
}


/* if delete_element == 0 only gets new element
   if delete_element != 0 gets new element as well (but you don't have to use
   it this time) _AND_ deletes it from file (queue.txt) */

int ma_get_new_el (int delete_element, char *q, char *a) {
	char b[MAX_A_LEN], *p;
	int q_left = MAX_Q_LEN-1, a_left = MAX_Q_LEN-1;
	FILE *fout, *fin = fopen (pathed("queue.txt"), "r");

	if (!fin) {	
		ma_error ("Cannot open \"%s\" file\n", pathed ("queue.txt"));
		return 1;
	}
	
	while (fgets (b, MAX_A_LEN, fin)) {
		if (strncmp (b, "<q>", 3) == 0) {
			strncpy (q, &b[3], q_left);
			q[MAX_Q_LEN-1] = '\0';
			q_left -= strlen(&b[3]);
		}
		else
			continue;
		if ((p=strstr (q, "</q>")))
			*p = '\0';
		else {
			while (1) {
				if (fgets (b, MAX_A_LEN, fin) == NULL) {
					ma_error ("Error while getting a new element (EOF?)\n");
					perror ("Error while getting a new element");
					return 1;
				}
				strncat (q, b, q_left);
				q_left -= strlen (b);
				if ((p=strstr (q, "</q>"))) { 
					*p = '\0';
					break;
				}
			}
		}
		while (1) {
			if (fgets (b, MAX_A_LEN, fin) == NULL) {
				ma_error ("Error while getting a new element (EOF?)\n");
				perror ("Error while getting a new element");
				return 1;
			}
			if (strncmp (b, "<a>", 3) == 0) {
				strncpy (a, &b[3], a_left);
				a[MAX_A_LEN-1] = '\0';
				a_left -= strlen (&b[3]);
				break;
			}
		}
		while (1) {
			if ((p=strstr (a, "</a>"))) {
				*p = '\0';
				break;
			}
			if (!fgets (b, MAX_A_LEN, fin)) {
				ma_error ("Error while getting new element\n");
				perror ("Error while getting a new element");
				return 1;
			}
			strncat (a, b, a_left);
			a_left -= strlen(b);
		}
		if (!delete_element) {
			fclose (fin);
			return 0;
		}
		if (!(fout = fopen (pathed ("queue.new"), "w"))) {
			ma_error ("Whoa, an error: cannot create %s !\n", pathed("queue.new"));
			exit (1);
		}
		while (fgets (b, MAX_A_LEN, fin))
			fputs (b, fout);
		fclose (fin);
		fclose (fout);
		if (remove (pathed ("queue.txt"))) {
			ma_error ("Whoa. That's a serious error, I cannot remove %s!\n", pathed("queue.txt"));
			exit (1);
		}

		if (rename (pathed ("queue.new"), pathed("queue.txt"))) {
			ma_error ("Whoa. That's a serious error, I cannot rename %s to %s!\n",
					pathed ("queue.new"), pathed ("queue.txt"));
			exit (1);
		}
		return 0;
	}
	assert (fin);
	fclose (fin);
	return 1;
}

#define DELETE_THIS_TAG 0
#define FIRE_THIS_TAG 1
#define REPLACE_WITH_HTML_CODE 2
#define REPLACE_WITH_MEMAID_CODE 3 /* e.g. <img>blah.jpg</img>, <sound>blah.mp3</sound> */

static int handle_exec_tag (char*, int);

/* Don't call parse_tags on el->q or anything that should not be altered,
 * run parse_tags() on a copy of string */
void ma_run_tags (const char *s, int version) {
	char buf[MAX_A_LEN];

	assert (strlen (s) < MAX_A_LEN);
	strcpy (buf, s);
	
	switch (version) {
		case CONSOLE_VERSION:
			handle_exec_tag (buf, FIRE_THIS_TAG);
			break;
		case GUI_VERSION:
			handle_exec_tag (buf, FIRE_THIS_TAG);
			break;
		case CGI_VERSION:
			/* handle_exec_tag (buf, DELETE_THIS_TAG); no need to do anything */
			break;
		default:
			ma_error ("Whoa. This error should never happen...\n");
			return;
	}
	return;
}

void ma_remove_tags (char *s) {
	handle_exec_tag (s, DELETE_THIS_TAG);

	return;
}

/* we won't use it on CGI/web server */
static int handle_exec_tag (char *s, int fire_it) {
	char *begin, *end;
	int len, i;
#ifdef __GNUC__
	FILE *f_pipe;
#endif
							
	if (!(begin=strstr (s, "<exec>")) || !(end=strstr (s, "</exec>")) || end < begin)
		return 1;

	len = strlen (begin + strlen ("</exec>"));
	if (fire_it == FIRE_THIS_TAG) {
		*end = '\0'; /* we will alter *s anyway */
#ifdef __GNUC__
		f_pipe = popen (begin + strlen ("<exec>"), "r");
		i = 0;
		while (getc (f_pipe) != EOF) {
			if (++i > 9999) {
				ma_error ("Error while reading from a pipe - "\
						"a huge number of characters. o_O\n");
				break;
			}
			continue;
		}

		pclose (f_pipe);
#else		
		system (begin + strlen ("<exec>"));
#endif
	}
	memmove (begin, end + strlen ("</exec>"), len);
	handle_exec_tag (s, fire_it);
	return 0;
}

int ma_export (const char *filename) {
			/* returns:
			 * 0 = ok
			 * 1 = no elements to export
			 * 2 = file already exist
			 * 3 = no access, cannot write
			 */
	FILE *fout;
	struct elem *el = List;
	
	if (el == NULL)
		return 1;

	if ((fout = fopen (filename, "r"))) {
		fclose (fout);
		return 2;
	}
	if (!(fout = fopen (filename, "w")))
		return 3;

	do {
		fprintf (fout, "<q>%s</q>\n<a>%s</a>\n\n", el->q, el->a);
	} while ((el = el->nxt) != NULL);

	fclose (fout);
	return 0;
}

INLINE struct elem* ma_iterate_list (register struct elem* current) {
	return current == NULL ? List : current->nxt;
}

INLINE int ma_list_is_loaded (void) {
	return List != NULL;
}

INLINE u_long get_tm_strt (void) {
	return tm_strt;
}

INLINE void set_tm_strt (u_long t) {
	tm_strt = (time_t)t;
}


/* Some auto-test.
 * Keep in mind they test only *some* of code in the core.
 */

#ifdef MEMAID_SELFTEST

#ifdef NDEBUG
#undef NDEBUG
#endif

#define MAXTEST 100

int main (void) {
	int i;
	float f;
	u_short us;
	
	F_debug = stderr;

	puts ("Basic self-test, start...");
	
	for (i = 0; i < MAXTEST; ++i) {
		us = ma_rand (MA_MAX_INTERVAL);
		f = f_interval (us);
		assert (f >= 0 && f <= 1.0);
		assert (us == us_interval (f));
	}
	puts ("f_interval() and us_interval() seems to be OK");

	for (i = 0; i < MAXTEST; ++i) {
		us = ma_rand (MA_MAX_N_OF_RPTS);
		f = f_n_of_rpt (us);
		assert (f >= 0 && f <= 1.0);
		assert (us == us_n_of_rpt (f));
	}
	puts ("f_n_of_rpt() and us_no_of_rpt() seems to be OK");

	for (i = 0; i < MAXTEST; ++i) {
		us = ma_rand (6); /* max grade == 5 */
		f = f_grade (us);
		assert (f >= 0 && f <= 1.0);
		assert (us == us_grade (f));
	}
	puts ("f_grade() and us_grade() seems to be OK");

	
	/* to be continued... */

	
	puts ("All tests passed.");
	return 0;
}

#endif /* MEMAID_SELFTEST */

