/*
 * medussa - a distributed cracking system
 * Copyright (C) 1999 Kostas Evangelinos <kos@bastard.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 * 
 */

/*
 * $Id: hashpool.c,v 1.35 2003/02/05 04:40:14 kos Exp $
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <time.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <pthread.h>
#include <stdarg.h>

#include "common.h"
#include "xmalloc.h"
#include "llog.h"
#include "array.h"
#include "hashpool.h"
#include "generator.h"
#include "configfile.h"
#include "method.h"

/* Aptly named */
int
get_action(char *act) {
  if(!act)
    return -1;
  if(!strcmp(act, "add"))
    return ACTION_ADD;
  if(!strcmp(act, "delete"))
    return ACTION_DELETE;
  if(!strcmp(act, "disable"))
    return ACTION_DISABLE;
  if(!strcmp(act, "enable"))
    return ACTION_ENABLE;
  if(!strcmp(act, "get"))
    return ACTION_GET;
  if(!strcmp(act, "set"))
    return ACTION_SET;
  if(!strcmp(act, "eta"))
    return ACTION_ETA;
  if(!strcmp(act, "percent"))
    return ACTION_PERCENT;
  if(!strcmp(act, "list"))
    return ACTION_LIST;
  if(!strcmp(act, "info"))
    return ACTION_INFO;
  if(!strcmp(act, "totalcps"))
    return ACTION_TOTALCPS;
  if(!strcmp(act, "reset"))
    return ACTION_RESET;
  return -2;
}

static int
hashpool_schedules_done(hashpool_t *h) {
  schedule_t *s;
  int i;

  for(i=0; (s=array_get(h->schedules, i)); i++) {
    if(s->state == ready || s->state == crunching)
      return 0;
  }
  return 1;
}

static int
hashpool_hashes_done(hashpool_t *h) {
  int i;
  hash_t *ha;

  for(i=0; (ha=array_get(h->hashes, i)); i++) {
    if(ha->state == ready || ha->state == crunching)
      return 0;
  }
  return 1;
}

int
hashpool_done(hashpool_t *h) {

  if(hashpool_schedules_done(h) || hashpool_hashes_done(h))
    return 1;
  return 0;
}

state_t
hashpool_state_aton(char *sstate) {
  state_t state;

  if(!strcmp(sstate, "ready")) {
    state = ready;
  } else if(!strcmp(sstate, "done")) {
    state = done;
  } else if(!strcmp(sstate, "crunching")) {
    state = crunching;
  } else if(!strcmp(sstate, "found")) {
    state = found;
  } else if(!strcmp(sstate, "disabled")) {
    state = disabled;
  } else if(!strcmp(sstate, "deceased")) {
    state = deceased;
  } else {
    state = ready;
  }
  
  return state;
}

char *
hashpool_state_ntoa(state_t state) {
  
  switch(state) {
  case ready:
    return "ready";
  case done:
    return "done";
  case crunching:
    return "crunching";
  case found:
    return "found";
  case disabled:
    return "disabled";
  case deceased:
    return "deceased";
  default:
    return "unknown";
  }  
}

workload_t *
workload_init(void) {
  workload_t *w;

  w = (workload_t *)xcalloc(1, sizeof(workload_t));
  memset(w->method, '\0', HASH_LINELEN);
  bstring_zero(&w->hash);
  memset(w->generator, '\0', HASH_LINELEN);
  memset(w->generator_params, '\0', HASH_LINELEN);
  memset(w->start, '\0', HASH_LINELEN);
  memset(w->finish, '\0', HASH_LINELEN);
  return w;
}

int
workload_destroy(workload_t *w) {
  xfree(w);
  return 0;
}

static void
hashpool_seterr(hashpool_t *h, char *fmt, ...) {
  va_list vl;

  va_start(vl, fmt);
  vsnprintf(h->errstring, HASH_LINELEN, fmt, vl);
  va_end(vl);
}

static void
hashpool_noerror(hashpool_t *h) {
  hashpool_seterr(h, "No error");
}

char *
hashpool_geterr(hashpool_t *h) {
  return h->errstring;
}

static int
hashpool_lock(hashpool_t *h) {
  llog(9, "[%d] lock\n", pthread_self());
  return pthread_mutex_lock(h->mutex);
}

static int
hashpool_unlock(hashpool_t *h) {
  llog(9, "[%d] unlock\n", pthread_self());
  return pthread_mutex_unlock(h->mutex);
}

static schedule_t *
schedule_find(hashpool_t *h, char *name) {
  int i;
  schedule_t *s;
  int serial;

  serial = atoi(name);
  for(i=0; (s=array_get(h->schedules, i)); i++) {
    if(serial == s->serial)
      return s;
  }
  return (schedule_t *)NULL;
}

static node_t *
node_find(hashpool_t *h, char *name) {
  int i;
  node_t *n;

  for(i=0; (n=array_get(h->nodes, i)); i++) {
    if(!strcmp(n->name, name))
      return n;
  }
  return (node_t *)NULL;
}

static hash_t *
hash_find(hashpool_t *h, char *name) {
  int i;
  hash_t *ha;

  for(i=0; (ha=array_get(h->hashes, i)); i++) {
    if(!strcmp(ha->name, name))
      return ha;
  }
  return (hash_t *)NULL;
}

static int
nodes_all_finished(hashpool_t *h) {
  node_t *n;
  int i;
  
  for(i=0; (n=array_get(h->nodes, i)); i++)
    if(n->state == crunching || n->state == deceased)
      return 0;
  return 1;
}

static int 
verify_node(hashpool_t *h, node_t *n, uchar *ukey, int ukeylen, uchar *ukeyindex) {
  generator_t *g;
  uchar key[HASH_LINELEN];
  unsigned int keylen;
  key_index_t index;
  method_t *m;
  int res;

  if(!(g = generator_init(h->cur_schedule->generator, h->cur_schedule->generator_params))) {
    hashpool_seterr(h, "verify_node: Couldn't initialize generator %s", h->cur_schedule->generator);
    return 1;
  }
  keyspace_init(index);
  keyspace_fromstr(index, ukeyindex);
  generator_set(g, index);
  keyspace_destroy(index);
  generator_fetch(g, key, HASH_LINELEN, &keylen);
  generator_destroy(g);
  if(keylen != ukeylen || memcmp(ukey, key, keylen)) {
    hashpool_seterr(h, "verify_node: Generated key for index %s is different", ukeyindex);
    return 2;
  }
  if(!(m = method_init(h->cur_schedule->hash->method, ""))) {
    hashpool_seterr(h, "verify_node: Couldn't initialize method %s", h->cur_schedule->hash->method);
    return 3;
  }
  method_sethash(m, h->cur_schedule->hash->hash.p, h->cur_schedule->hash->hash.l);
  method_add(m, key, keylen);
  res = method_crypt(m);
  method_destroy(m);
  
  if(!res) {
    hashpool_seterr(h, "verify_node: Hashed keys don't match");
    return 4;
  }
  hashpool_noerror(h);
  return 0;
}

static node_t *
node_new(hashpool_t *h, char *name, time_t now) {
  node_t n;

  strncpy(n.name, name, HASH_LINELEN);
  keyspace_init(n.start);
  keyspace_init(n.finish);
  n.slice = msg_get_int(h->param, "def_keyslice");
  n.state = ready;
  n.cracks_per_sec = 0;
  n.time_created = now;
  n.time_start = (time_t)0;
  n.time_total = (time_t)0;
  if(array_add(h->nodes, &n)) {
    keyspace_destroy(n.start);
    keyspace_destroy(n.finish);
    return (node_t *)NULL;
  }
  
  return node_find(h, name);
}

static int
invalidate_nodes(hashpool_t *h) {
  node_t *n;
  int i;
  
  for(i=0; (n=array_get(h->nodes, i)); i++) {
    n->state = ready;
    n->slice = msg_get_int(h->param, "def_keyslice");
  }
  return 0;
}

static int
hashpool_hash_add(hashpool_t *h, char *name, char *method, state_t state, unsigned char *hash, int hashlen) {
  hash_t ha;

  hashpool_noerror(h);
  strncpy(ha.name, name, HASH_LINELEN);
  strncpy(ha.method, method, HASH_LINELEN);
  bstring_zero(&ha.hash);
  bstring_zero(&ha.key);
  bstring_set_str(&ha.hash, hash, hashlen);
  ha.state = state;

  hashpool_lock(h);
  if(array_add(h->hashes, &ha)) {
    hashpool_unlock(h);
    hashpool_seterr(h, "array_add failed. Out of memory?");
    return 1;
  }
  hashpool_unlock(h);
  return 0;
}

static int
hashpool_schedule_add(hashpool_t *h, char *hash, state_t state, char *index, char *generator, char *generator_params) {
  schedule_t s;
  generator_t *g;
  hash_t *ha;
  int i;
  
  hashpool_noerror(h);
  for(ha=(hash_t *)NULL,i=0; (ha=array_get(h->hashes, i)); i++) {
    if(!strcmp(hash, ha->name))
      break;
  }

  if(!ha) {
    hashpool_seterr(h, "No such hash %s", hash);
    return 2;
  }

  if(!(g = generator_init(generator, generator_params))) {
    hashpool_seterr(h, "generator_init(%s, %s) failed", generator, generator_params);
    return 3;
  }

  s.serial = h->schedule_serial;
  s.hash = ha;
  s.state = state;
  s.time_start = s.time_end = (time_t)0;
  strncpy(s.generator, generator, HASH_LINELEN);
  strncpy(s.generator_params, generator_params, HASH_LINELEN);
  strncpy(s.method, ha->method, HASH_LINELEN);
  keyspace_init(s.start);
  keyspace_init(s.finish);
  keyspace_init(s.index);
  keyspace_set(s.start, *generator_minindex(g));
  keyspace_set(s.finish, *generator_maxindex(g));
  if(index)
    keyspace_fromstr(s.index, index);
  else
    keyspace_set(s.index, *generator_minindex(g));
  generator_destroy(g);

  hashpool_lock(h);
  h->schedule_serial++;
  if(array_add(h->schedules, &s)) {
    hashpool_unlock(h);
    hashpool_seterr(h, "array_add failed. Out of memory?");
    return 1;
  }
  hashpool_unlock(h);
  return 0;
}

int
hashpool_schedule(hashpool_t *h) {
  schedule_t *s;
  hash_t *ha;
  node_t *n;
  int i;
  int j;
  time_t now;
  
  /* Mess with the schedule itself */
  if(hashpool_done(h)) {
    hashpool_seterr(h, "hashpool_schedule: pool is done");
    return 1;
  }

  if(h->state == disabled) {
    hashpool_seterr(h, "hashpool_schedule: pool is disabled");
    return 2;
  } else if(h->state == ready) {
    h->state = crunching;
  }

  now = time(NULL);
  hashpool_noerror(h);
  hashpool_lock(h);

  /* Reap old nodes */
  for(i=0; (n=array_get(h->nodes, i)); i++) {
    if((n->state == ready || n->state == crunching) && (now-n->time_start) > msg_get_int(h->param, "time_to_die")) {
      n->state = deceased;
      n->slice = msg_get_int(h->param, "def_keyslice");
    }
  }
  
  /* Progress state from hashes to schedules in case of disabled, done, found */
  for(i=0; (ha=array_get(h->hashes, i)); i++) {
    if(ha->state == disabled || ha->state == done) {
      for(j=0; (s=array_get(h->schedules, j)); j++) {
	if(s->hash == ha)
	  s->state = ha->state;
      }
    }
    if(ha->state == found) {
      for(j=0; (s=array_get(h->schedules, j)); j++) {
	if(s->hash == ha) {
	  s->state = done;
	  s->time_end = now;
	}
      }
    }
  }
  
  /* Find schedules that are flagged as crunching and are really done */
  for(i=0; (s=array_get(h->schedules, i)); i++) {
    if(s->state == crunching && 
       !keyspace_cmp(s->index, s->finish) && 
       nodes_all_finished(h)) {
      s->time_end = now;
      s->state = done;
    }
  }

  /* Find a feasible schedule to run with, flag it as crunching */
  for(i=0; (s=array_get(h->schedules, i)); i++) {
    if(s->state == ready || s->state == crunching) {
      s->state = crunching;
      s->time_start = now;
      h->cur_schedule = s;
      break;
    }
  }

#if I_AM_REALLY_TIGHT_ON_MEMORY_AND_WILLING_TO_SUFFER
  /* Clean up after old schedules */
  for(i=0; (s=array_get(h->schedules, i)); i++) {
    if(s->state == done && s->generator) {
      keyspace_destroy(s->start);
      keyspace_destroy(s->finish);
      keyspace_destroy(s->index);
    }
  }
#endif

  hashpool_unlock(h);
  return 0;
}

hashpool_t *
hashpool_init(void) {
  hashpool_t *h;
  time_t now;
  
  now = time(NULL);
  h = xcalloc(1, sizeof(hashpool_t));
  
  if(!(h->nodes = array_init(sizeof(node_t)))) {
    xfree(h);
    return NULL;
  }
  if(!(h->hashes = array_init(sizeof(hash_t)))) {
    array_nuke(h->nodes);
    xfree(h);
    return NULL;
  }
  if(!(h->schedules = array_init(sizeof(schedule_t)))) {
    array_nuke(h->hashes);
    array_nuke(h->nodes);
    xfree(h);
    return NULL;
  }

  h->cur_schedule = (schedule_t *)NULL;
  h->schedule_serial = 0;  
  h->param = msg_new();
  msg_add_int(h->param, "def_keyslice", DEF_KEYSLICE);
  msg_add_int(h->param, "max_keyslice", MAX_KEYSLICE);
  msg_add_int(h->param, "min_keyslice", MIN_KEYSLICE);
  msg_add_int(h->param, "time_to_crunch", DEF_TIMETOCRUNCH);
  msg_add_int(h->param, "time_to_die", DEF_TIMETODIE);
  h->mutex = (pthread_mutex_t *)xcalloc(1, sizeof(pthread_mutex_t));
  h->state = ready;
  h->time_start = now;
  pthread_mutex_init(h->mutex, NULL);
  hashpool_noerror(h);
  return h;
}

static int
recalc_slice(hashpool_t *h, node_t *n, time_t now) {
  int slice;
  key_index_t diff;

  if(n->time_start == now)
    return msg_get_int(h->param, "def_keyslice");
  keyspace_init(diff);
  keyspace_sub(diff, n->finish, n->start);
  slice = keyspace_toint(diff);
  keyspace_destroy(diff);
  if(slice != n->slice)
    return n->slice;
  
  slice = (msg_get_int(h->param, "time_to_crunch") * n->slice) / (now - n->time_start);
  if(slice > msg_get_int(h->param, "max_keyslice"))
    slice = msg_get_int(h->param, "max_keyslice");
  if(slice < msg_get_int(h->param, "min_keyslice"))
    slice = msg_get_int(h->param, "min_keyslice");
  return slice;
}

int
hashpool_node_exists(hashpool_t *h, char *name) {
  return (node_find(h, name) != (node_t *)NULL);
}

int
hashpool_node_register(hashpool_t *h, char *name, workload_t *w) {
  node_t *n;
  node_t *n2;
  time_t now;
  key_index_t start, finish;
  int i;

  /* hashpool_schedule returns its own errors */
  if(hashpool_schedule(h))
    return 1;

  hashpool_noerror(h);

  hashpool_lock(h);
  now = time(NULL);
  if(!(n = node_find(h, name))) {
    if(!(n = node_new(h, name, now))) {
      hashpool_unlock(h);
      hashpool_seterr(h, "Adding new node failed. Out of memory?");
      return 5;
    }
  }   
  if(n->state == disabled) {
    hashpool_unlock(h);
    hashpool_seterr(h, "Node %s is disabled", n->name);
    return 4;
  }

  strncpy(w->method, h->cur_schedule->method, HASH_LINELEN);
  bstring_set(&w->hash, &h->cur_schedule->hash->hash);
  strncpy(w->generator, h->cur_schedule->generator, HASH_LINELEN);
  strncpy(w->generator_params, h->cur_schedule->generator_params, HASH_LINELEN);  

  /*
   * If this node is still crunching, we need to get it to recover; Resend the same keyspace
   * Recovery from sleepy nodes doesn't need anything special here since we can't have 
   * switched schedule while this node was lingering 
   */
  if(n->state == crunching) {
    hashpool_unlock(h);
    keyspace_tostr(w->start, HASH_LINELEN, n->start);
    keyspace_tostr(w->finish, HASH_LINELEN, n->finish);
    return 0;
  }

  /* If we have dead nodes, use their keyspace */
  for(i=0; (n2=array_get(h->nodes, i)); i++) {
    if(n2->state == deceased) {
      keyspace_tostr(w->start, HASH_LINELEN, n2->start);
      keyspace_tostr(w->finish, HASH_LINELEN, n2->finish);
      keyspace_set(n->start, n2->start);
      keyspace_set(n->finish, n2->finish);
      n->state = crunching;
      n->time_start = now;
      array_delete(h->nodes, i);
      hashpool_unlock(h);
      return 0;
    }
  }

  if(keyspace_cmp(h->cur_schedule->index, h->cur_schedule->finish) == 0) {
    hashpool_seterr(h, "Current schedule is out of keyspace, wait");
    hashpool_unlock(h);
    return 3;
  }

  /* Get new pair, update workload */  
  keyspace_init(start);
  keyspace_init(finish);
  w->slice = n->slice;
  keyspace_set(start, h->cur_schedule->index);
  keyspace_fromint(w->slice, finish);
  keyspace_add(finish, start, finish);

  if(keyspace_cmp(finish, h->cur_schedule->finish) > 0)
    keyspace_set(finish, h->cur_schedule->finish);    
  
  /* We now have a valid workload to furnish. Fill it out */
  keyspace_tostr(w->start, HASH_LINELEN, start);
  keyspace_tostr(w->finish, HASH_LINELEN, finish);  

  /* Update node information */
  keyspace_set(n->start, start);
  keyspace_set(n->finish, finish);
  keyspace_set(h->cur_schedule->index, n->finish);
  n->slice = w->slice;
  n->state = crunching;  
  n->time_start = now;

  hashpool_unlock(h);
  keyspace_destroy(start);
  keyspace_destroy(finish);
  return 0;
}

int
hashpool_node_update(hashpool_t *h, char *name, char *s, char *f, unsigned char *key, int keylen, unsigned char *keyindex) {
  node_t *n;
  char i1[HASH_LINELEN];
  char i2[HASH_LINELEN];
  time_t now;
  key_index_t diff;
  int res;

  /* hashpool_schedule returns its own errors */
  hashpool_schedule(h);
  hashpool_noerror(h);
  if(!(n = node_find(h, name))) {
    hashpool_seterr(h, "Unknown client %s tries to update", name);
    return 4;
  }

  if(n->state != crunching && 
     n->state != disabled && 
     n->state != deceased) {
    hashpool_seterr(h, "client %s tries to update while being in state %s", name, 
		    hashpool_state_ntoa(n->state));
    return 5;
  }

  keyspace_tostr(i1, HASH_LINELEN, n->start);
  keyspace_tostr(i2, HASH_LINELEN, n->finish);
  if(strcmp(s, i1) || strcmp(f, i2)) {
    hashpool_seterr(h, "client %s tries to update with bogus indexes: %s %s", n->name, s, f);
    return 2;
  }

  /* verify_node returns its own errors */
  if(key && (res = verify_node(h, n, key, keylen, keyindex))) {
    return 3;
  }

  now = time(NULL);
  hashpool_lock(h);
  n->time_total += (now - n->time_start);
  if(key) {
    h->cur_schedule->hash->state = found;
    bstring_set_str(&h->cur_schedule->hash->key, key, keylen);
  } else {
    keyspace_init(diff);
    keyspace_sub(diff, n->finish, n->start);    
    if(now != n->time_start)
      n->cracks_per_sec = keyspace_toint(diff)/(now - n->time_start);
    else
      n->cracks_per_sec = keyspace_toint(diff);
    keyspace_destroy(diff);
    n->slice = recalc_slice(h, n, now);    
  }
  hashpool_unlock(h);
  hashpool_schedule(h);
  hashpool_lock(h);

  if(n->state == crunching)
    n->state = ready;
  hashpool_unlock(h);
  return 0;
}

static int 
checkpoint_save_schedule(schedule_t *s, char *name, config_t *c) {
  char scratch[DEF_LINELEN];
  
  snprintf(scratch, DEF_LINELEN, "%d", s->serial); config_class_set(c, name, "serial", scratch);
  snprintf(scratch, DEF_LINELEN, "%s", hashpool_state_ntoa(s->state)); config_class_set(c, name, "state", scratch);
  snprintf(scratch, DEF_LINELEN, "%s", s->hash->name); config_class_set(c, name, "hash", scratch);
  snprintf(scratch, DEF_LINELEN, "%s", s->generator); config_class_set(c, name, "generator", scratch);
  snprintf(scratch, DEF_LINELEN, "%s", s->generator_params); config_class_set(c, name, "generator_params", scratch);
  snprintf(scratch, DEF_LINELEN, "%s", s->method); config_class_set(c, name, "method", scratch);
  snprintf(scratch, DEF_LINELEN, "%ld", s->time_start); config_class_set(c, name, "time_start", scratch);
  snprintf(scratch, DEF_LINELEN, "%ld", s->time_end); config_class_set(c, name, "time_end", scratch);
  keyspace_tostr(scratch, DEF_LINELEN, s->start); config_class_set(c, name, "start", scratch);
  keyspace_tostr(scratch, DEF_LINELEN, s->finish); config_class_set(c, name, "finish", scratch);
  keyspace_tostr(scratch, DEF_LINELEN, s->index); config_class_set(c, name, "index", scratch);
  return 0;
}

static int
checkpoint_save_hash(hash_t *h, char *name, config_t *c) {
  char scratch[DEF_LINELEN];
  int scratch2;
  
  snprintf(scratch, DEF_LINELEN, "%s", hashpool_state_ntoa(h->state)); config_class_set(c, name, "state", scratch);
  snprintf(scratch, DEF_LINELEN, "%s", h->name); config_class_set(c, name, "name", scratch);
  snprintf(scratch, DEF_LINELEN, "%s", h->method); config_class_set(c, name, "method", scratch);
  memset(scratch, '\0', DEF_LINELEN);
  textify(h->hash.p, h->hash.l, scratch, &scratch2); config_class_set(c, name, "hash", scratch);
  memset(scratch, '\0', DEF_LINELEN);
  textify(h->key.p, h->key.l, scratch, &scratch2); config_class_set(c, name, "key", scratch);
  return 0;
}

int
hashpool_checkpoint(hashpool_t *h, char *fname) {
  char lfname[DEF_LINELEN];
  char scratch[DEF_LINELEN];
  config_t *save;
  schedule_t *s;
  hash_t *ha;
  int i;

  strncpy(lfname, fname, DEF_LINELEN);
  strncat(lfname, ".tmp", DEF_LINELEN);
  
  hashpool_noerror(h);
  if(!h->cur_schedule)
    return 0;

  if(!(save = config_init("save"))) {
    hashpool_seterr(h, "config_init failed");
    return 1;
  }

  /* add hashpool global stuff */
  snprintf(scratch, DEF_LINELEN, "%d", h->schedule_serial); config_set(save, "schedule_serial", scratch);
  snprintf(scratch, DEF_LINELEN, "%ld", h->time_start); config_set(save, "time_start", scratch);
  snprintf(scratch, DEF_LINELEN, "%s", hashpool_state_ntoa(h->state)); config_set(save, "state", scratch);
  snprintf(scratch, DEF_LINELEN, "%d", h->cur_schedule->serial); config_set(save, "cur_schedule", scratch);
  
  /* add hashes */
  for(i=0; (ha=array_get(h->hashes, i)); i++) {
    snprintf(scratch, DEF_LINELEN, "hash.%s", ha->name);
    checkpoint_save_hash(ha, scratch, save);
  }

  /* add schedules */
  for(i=0; (s=array_get(h->schedules, i)); i++) {
    snprintf(scratch, DEF_LINELEN, "schedule.%d", s->serial);
    checkpoint_save_schedule(s, scratch, save);
  }

  /* commit */
  if(config_save(save, lfname)) {
    hashpool_seterr(h, "config_save: %s", config_perror(save));
    return 2;
  }    
  config_destroy(save);
  rename(lfname, fname);
  return 0;
}

int 
hashpool_restore(hashpool_t *h, char *fname) {
  config_t *restore;
  int i;
  char *clname;
  char *rclname;
  msg *params;
  
  hashpool_noerror(h);
  if(!(restore = config_init("restore"))) {
    hashpool_seterr(h, "config_init failed");
    return 1;
  }
  
  if(config_load(restore, fname)) {
    hashpool_seterr(h, "config_load failed: %s", config_perror(restore));
    return 2;
  }
  
  for(i=0; i<config_nclasses(restore); i++) {
    clname = config_class_name(restore, i);
    if(!strncmp(clname, "hash.", 5)) {
      rclname = clname+5;
      params = msg_new();
      msg_add_str(params, "name", config_class_char_get(restore, clname, "name"), 0);
      msg_add_str(params, "method", config_class_char_get(restore, clname, "method"), 0);
      msg_add_str(params, "hash", config_class_char_get(restore, clname, "hash"), 0);
      msg_add_str(params, "state", config_class_char_get(restore, clname, "state"), 0);
      hashpool_manage(h, "hash", "add", params);
      msg_destroy(params);
    } else if(!strncmp(clname, "schedule.", 9)) {
      rclname = clname+9;
      params = msg_new();
      msg_add_str(params, "serial", config_class_char_get(restore, clname, "serial"), 0);
      msg_add_str(params, "generator", config_class_char_get(restore, clname, "generator"), 0);
      msg_add_str(params, "generator_params", config_class_char_get(restore, clname, "generator_params"), 0);
      msg_add_str(params, "method", config_class_char_get(restore, clname, "method"), 0);
      msg_add_str(params, "hash", config_class_char_get(restore, clname, "hash"), 0);
      msg_add_str(params, "state", config_class_char_get(restore, clname, "state"), 0);
      msg_add_str(params, "start", config_class_char_get(restore, clname, "start"), 0);
      msg_add_str(params, "finish", config_class_char_get(restore, clname, "finish"), 0);
      msg_add_str(params, "index", config_class_char_get(restore, clname, "index"), 0);
      hashpool_manage(h, "schedule", "add", params);
      msg_destroy(params);
    }
  }

  if(h->schedule_serial != config_int_get(restore, "schedule_serial")) {
    hashpool_seterr(h, "BUG: restore produced inconsistent results");
    return 3;
  }
  
  config_destroy(restore);
  return 0;
}

char *
hashpool_stats(hashpool_t *h, int type, int serial, char *buf, int len) {
  schedule_t *s;
  node_t *n;
  int i;
  int total_cps;
  key_index_t a;
  key_index_t b;
  char buf2[HASH_LINELEN];
  time_t eta;
  double da;
  double db;
  struct tm *stm;
  
  hashpool_noerror(h);
  if(serial == SCHED_CURRENT) {
    if(!(s = h->cur_schedule) || !array_nelems(h->nodes)) {
      hashpool_seterr(h, "No current schedule or no registered nodes");
      strcpy(buf, "-");
      return buf;
    }
  } else {
    for(i=0; (s=array_get(h->schedules, i)); i++) {
      if(s->serial == serial)
	goto found;      
    }
    hashpool_seterr(h, "No such schedule");
    strcpy(buf, "-");
    return buf;
  }

 found:
  for(total_cps=0,i=0; (n=array_get(h->nodes, i)); i++)
    if(n->cracks_per_sec)
      total_cps+=(n->cracks_per_sec);

  if(!total_cps) {
    hashpool_seterr(h, "No stats available for nodes");
    strcpy(buf, "-");
    return buf;
  }
    
  switch(type) {
  case STATS_ETA:
    /* eta = (finish-start) / total_cps */
    /* days = eta / (60*60*24) */
    keyspace_init(a);
    keyspace_init(b);
    keyspace_fromint(total_cps, b);
    keyspace_sub(a, s->index, s->start);
    keyspace_sub(a, s->finish, a);
    keyspace_div(a, a, b);
    eta = (time_t)keyspace_toint(a);
    keyspace_destroy(b);
    keyspace_destroy(a);    
    stm = gmtime(&eta);
    strftime(buf2, HASH_LINELEN, "%H:%M:%S", stm);
    snprintf(buf, len, "%lu:%s", eta/(60*60*24), buf2);
    break;
  case STATS_PERCENT:
    keyspace_init(a);
    keyspace_init(b);
    keyspace_sub(a, s->finish, s->start);
    keyspace_sub(b, s->index, s->start);
    da = keyspace_todouble(a);
    db = keyspace_todouble(b);
    keyspace_destroy(b);
    keyspace_destroy(a);    
    if(!da) {
      hashpool_seterr(h, "No percentage available");
      strcpy(buf, "-");
    } else
      snprintf(buf, len, "%2.3f", (100*db)/da);
    break;
  case STATS_TOTALCPS:
    snprintf(buf, len, "%d", total_cps);
    break;
  }

  return buf;
}

int
hashpool_destroy(hashpool_t *h) {  
  array_nuke(h->nodes);
  array_nuke(h->hashes);
  array_nuke(h->schedules);
  msg_destroy(h->param);
  xfree(h->mutex);
  xfree(h);
  return 0;
}

/*
 * Input:
 * Output:
 */

int
pool_manage(hashpool_t *h, int action, msg *params) {
  int retv = 0;
  
  switch(action) {
  case ACTION_DISABLE:
    hashpool_lock(h);
    h->state = disabled;
    hashpool_unlock(h);
    msg_zero(params);
    break;
  case ACTION_ENABLE:
    hashpool_lock(h);
    h->state = ready;
    hashpool_unlock(h);
    msg_zero(params);
    break;
  case ACTION_INFO:
    msg_zero(params);
    msg_add_strn(params, "state", hashpool_state_ntoa(h->state));
    msg_add_int(params, "current schedule", h->cur_schedule?h->cur_schedule->serial:-1);
    msg_add_int(params, "start time", h->time_start);
    break;
  default:
    hashpool_seterr(h, "no such action %d", action);
    msg_zero(params);
    retv = 10;
    break;
  }

  return retv;
}

/* 
 * Input: [schedule]
 * output: eta, totalcps, percent
 */

int
stats_manage(hashpool_t *h, int action, msg *params) {
  schedule_t *s = (schedule_t *)NULL;
  node_t *n;
  int i;
  int total_cps;
  key_index_t a;
  key_index_t b;
  char buf[HASH_LINELEN];
  char buf2[HASH_LINELEN];
  time_t eta;
  double da;
  double db;
  struct tm *stm;
  char *schedule;

  if(!array_nelems(h->nodes) || !(s = h->cur_schedule)) {
    hashpool_seterr(h, "no registered nodes or no current schedule");
    return 21;
  }

  if((schedule = msg_get_strp(params, "name")) && (s = schedule_find(h, schedule))) {
    /* nothing */
  } else if(h->cur_schedule) {
    s = h->cur_schedule;
  } else {
    hashpool_seterr(h, "no such schedule");
    return 22;
  }

  for(total_cps=0,i=0; (n=array_get(h->nodes, i)); i++)
    if(n->cracks_per_sec)
      total_cps+=(n->cracks_per_sec);
  
  if(!total_cps) {
    hashpool_seterr(h, "No stats available for nodes");
    return 23;
  }

  switch(action) {
  case ACTION_ETA:
    /* eta = (finish-start) / total_cps */
    /* days = eta / (60*60*24) */
    keyspace_init(a);
    keyspace_init(b);
    keyspace_fromint(total_cps, b);
    keyspace_sub(a, s->index, s->start);
    keyspace_sub(a, s->finish, a);
    keyspace_div(a, a, b);
    eta = (time_t)keyspace_toint(a);
    keyspace_destroy(b);
    keyspace_destroy(a);    
    stm = gmtime(&eta);
    strftime(buf2, HASH_LINELEN, "%H:%M:%S", stm);
    snprintf(buf, HASH_LINELEN, "%lu:%s", eta/(60*60*24), buf2);
    msg_zero(params);
    msg_add_strn(params, "eta", buf);
    break;
  case ACTION_PERCENT:
    keyspace_init(a);
    keyspace_init(b);
    keyspace_sub(a, s->finish, s->start);
    keyspace_sub(b, s->index, s->start);
    da = keyspace_todouble(a);
    db = keyspace_todouble(b);
    keyspace_destroy(b);
    keyspace_destroy(a);    
    if(!da) {
      hashpool_seterr(h, "No percentage available");
      return 24;
    }
    snprintf(buf, HASH_LINELEN, "%2.3f", (100*db)/da);
    msg_zero(params);
    msg_add_strn(params, "percent", buf);
    break;
  case ACTION_TOTALCPS:
    snprintf(buf, HASH_LINELEN, "%d", total_cps);  
    msg_add_str(params, "totalcps", buf, HASH_LINELEN);
    break;
  default:
    hashpool_seterr(h, "no such action %d", action);
    msg_zero(params);
    return 20;
  }
  
  return 0;
}

/*
 * Input: [name]
 * Output: [name], [node_params]
 */

int
node_manage(hashpool_t *h, int action, msg *params) {
  unsigned char *name;
  int i;
  node_t *n = (node_t *)NULL;
  
  if((name = msg_get_strp(params, "name")))
    n = node_find(h, name);

  if(action != ACTION_LIST && !n) {
    hashpool_seterr(h, "no such node");
    return 31;
  }

  switch(action) {
  case ACTION_LIST:
    msg_zero(params);
    for(i=0; (n=array_get(h->nodes, i)); i++)
      msg_add_strn(params, n->name, n->name);
    break;

  case ACTION_INFO:
    msg_zero(params);
    msg_add_strn(params, "name", n->name);
    msg_add_strn(params, "start", keyspace_ntoa(n->start));
    msg_add_strn(params, "finish", keyspace_ntoa(n->finish));
    msg_add_int(params, "slice", n->slice);
    msg_add_strn(params, "state", hashpool_state_ntoa(n->state));
    msg_add_int(params, "cps", n->cracks_per_sec);
    msg_add_int(params, "time_created", n->time_created);
    msg_add_int(params, "time_start", n->time_start);
    msg_add_int(params, "time_total", n->time_total);
    break;

  case ACTION_DISABLE:
    hashpool_lock(h);    
    n->state = disabled;   
    hashpool_unlock(h);
    msg_zero(params);
    break;

  case ACTION_ENABLE:
    hashpool_lock(h);
    n->state = ready;
    hashpool_unlock(h);
    msg_zero(params);
    break;
    
  case ACTION_DELETE:
    hashpool_lock(h);
    n->state = deceased;
    hashpool_unlock(h);
    msg_zero(params);
    break;

  case ACTION_RESET:
    hashpool_lock(h);
    n->slice = msg_get_int(h->param, "def_keyslice");
    hashpool_unlock(h);
    msg_zero(params);
    break;

  default:
    hashpool_seterr(h, "no such action %d", action);
    msg_zero(params);
    return 30;
  }
  return 0;
}

/*
 * Input: [name] [value]
 * Output: [name/value pairs]
 */

int
parameter_manage(hashpool_t *h, int action, msg *params) {
  char *name;
  char *value;
  char index[HASH_LINELEN];
  char junk[HASH_LINELEN];
  int i;
  int j;

  name = msg_get_strp(params, "name");
  value = msg_get_strp(params, "value");

  if(action != ACTION_LIST && !name) {
    hashpool_seterr(h, "parameter \"name\" missing");
    return 41;
  }

  if(action == ACTION_SET && !value) {
    hashpool_seterr(h, "parameter \"value\" missing");
    return 42;
  }
  
  switch(action) {
  case ACTION_LIST:
    msg_zero(params);
    for(i=0; i<msg_nelems(h->param); i++) {
      msg_get_index(h->param, i, index, HASH_LINELEN, junk, HASH_LINELEN, &j);
      msg_add_strn(params, index, index);
    }
    break;
  case ACTION_GET:        
    msg_zero(params);
    if(msg_get_str(h->param, name, index, HASH_LINELEN, &i)) {
      hashpool_seterr(h, "no such parameter %s", name);
      return 43;
    }
    msg_add_str(params, name, index, i);
    break;
  case ACTION_SET:
    hashpool_lock(h);
    msg_add_strn(h->param, name, value);
    hashpool_unlock(h);
    msg_zero(params);
    break;
  default:
    hashpool_seterr(h, "no such action %d", action);
    msg_zero(params);
    return 40;
  }

  return 0;
}

/*
 * Input: [name], [generator], [generator_params], [method], [start], [finish]
 * Output: [name], [schedule_params]
 * XXX: "delete" needs to clean up afterwards
 */

int
schedule_manage(hashpool_t *h, int action, msg *params) {
  schedule_t *s = (schedule_t *)NULL;
  int i;
  char *name;
  char *hash;
  char *generator;
  char *generator_params;
  char *method;
  char *start;
  char *finish;
  char temp[HASH_LINELEN];
  char *sstate;
  state_t state;
  char *index;
 
  name = msg_get_strp(params, "name");
  hash = msg_get_strp(params, "hash");
  generator = msg_get_strp(params, "generator");
  generator_params = msg_get_strp(params, "generator_params");
  method = msg_get_strp(params, "method");
  start = msg_get_strp(params, "start");
  finish = msg_get_strp(params, "finish");
  index = msg_get_strp(params, "index");
  
  if((sstate = msg_get_strp(params, "state")))
    state = hashpool_state_aton(sstate);
  else
    state = ready;

  switch(action) {
  case ACTION_INFO:
  case ACTION_ENABLE:
  case ACTION_DISABLE:
  case ACTION_DELETE:
  case ACTION_RESET:
    if(name && (s = schedule_find(h, name))) {
      /* nothing */
    } else if(h->cur_schedule) {
      s = h->cur_schedule;
    } else {
      hashpool_seterr(h, "no such schedule");
      return 22;
    }
    break;
  }

  switch(action) {
  case ACTION_LIST:
    msg_zero(params);
    for(i=0; (s=array_get(h->schedules, i)); i++) {
      snprintf(temp, HASH_LINELEN, "%d", s->serial);
      msg_add_strn(params, temp, temp);
    }
    break;
  case ACTION_INFO:
    msg_zero(params);
    msg_add_int(params, "name", s->serial);
    msg_add_strn(params, "state", hashpool_state_ntoa(s->state));
    msg_add_strn(params, "hash", s->hash->name);
    msg_add_strn(params, "method", s->method);
    msg_add_int(params, "time_start", s->time_start);
    msg_add_int(params, "time_end", s->time_end);
    msg_add_strn(params, "generator", s->generator);
    msg_add_strn(params, "generator_params", s->generator_params);
    msg_add_strn(params, "start", keyspace_ntoa(s->start));
    msg_add_strn(params, "finish", keyspace_ntoa(s->finish));
    msg_add_strn(params, "index", keyspace_ntoa(s->index));
    if(s->state != done)
      msg_add_strn(params, "key", "none");
    else
      msg_add_str(params, "key", s->hash->key.p, s->hash->key.l);
    break;

  case ACTION_ENABLE:
    hashpool_lock(h);
    s->state = ready;
    hashpool_unlock(h);
    msg_zero(params);
    break;
  case ACTION_DISABLE:
    hashpool_lock(h);
    s->state = disabled;
    hashpool_unlock(h);
    msg_zero(params);
    break;
  case ACTION_DELETE:
    hashpool_lock(h);
    if(s == h->cur_schedule)
      invalidate_nodes(h);      
    s->state = disabled;
    hashpool_unlock(h);
    msg_zero(params);
    break;
  case ACTION_ADD:
    if(!hash || !generator || !generator_params) {
      hashpool_seterr(h, "hash/generator/params missing");
      msg_zero(params);
      return 52;
    }
    msg_zero(params);
    return hashpool_schedule_add(h, hash, state, index, generator, generator_params);    
  case ACTION_RESET:
    hashpool_lock(h);
    keyspace_set(s->index, s->start);
    h->cur_schedule->state = ready;
    s->state = crunching;
    hashpool_unlock(h);
    msg_zero(params);
    break;
  default:
    hashpool_seterr(h, "no such action %d", action);
    msg_zero(params);
    return 50;
  }

  return 0;
}

/*
 * Input: [name] [method] [hash]
 * Output: [name][ [method] [hash] [key]
 */

int
hash_manage(hashpool_t *h, int action, msg *params) {
  char *name;
  char *method;
  char hash[HASH_LINELEN];
  int hashlen;
  char key[HASH_LINELEN];
  int keylen;
  hash_t *ha = (hash_t *)NULL;
  int i;
  state_t state;

  if((name = msg_get_strp(params, "state"))) 
    state = hashpool_state_aton(name);
  else
    state = ready;

  name = msg_get_strp(params, "name");  
  method = msg_get_strp(params, "method");  
  msg_get_str(params, "hash", hash, HASH_LINELEN, &hashlen);
  msg_get_str(params, "key", key, HASH_LINELEN, &keylen);  

  if(name)
    ha = hash_find(h, name);

  switch(action) {
  case ACTION_INFO:
  case ACTION_ENABLE:
  case ACTION_DISABLE:
  case ACTION_DELETE:
    if(!name) {
      hashpool_seterr(h, "parameter \"name\" missing");
      return 61;
    }
    if(!ha) {
      hashpool_seterr(h, "no such hash \"%s\"", name);
      return 62;
    }
    break;
  }

  switch(action) {
  case ACTION_LIST:
    msg_zero(params);
    for(i=0; (ha=array_get(h->hashes, i)); i++)
      msg_add_strn(params, ha->name, ha->name);
    break;
  case ACTION_INFO:
    msg_zero(params);
    msg_add_strn(params, "name", ha->name);
    msg_add_strn(params, "state", hashpool_state_ntoa(ha->state));
    msg_add_strn(params, "method", ha->method);
    msg_add_str(params, "hash", ha->hash.p, ha->hash.l);
    if(ha->key.l)
      msg_add_str(params, "key", ha->key.p, ha->key.l);
    break;
  case ACTION_ENABLE:
    hashpool_lock(h);
    ha->state = ready;
    hashpool_unlock(h);
    msg_zero(params);
    break;
  case ACTION_DISABLE:
    hashpool_lock(h);
    ha->state = disabled;
    hashpool_unlock(h);
    msg_zero(params);
    break;
  case ACTION_DELETE:
    hashpool_lock(h);
    /* XXX: implement this */
    ha->state = disabled;
    hashpool_unlock(h);
    msg_zero(params);
    break;
  case ACTION_ADD:
    if(!name || !method || !hash) {
      hashpool_seterr(h, "name/method/hash missing");
      msg_zero(params);
      return 62;
    }
    return hashpool_hash_add(h, name, method, state, hash, hashlen);
    break;
  default:
    hashpool_seterr(h, "no such action %d", action);
    msg_zero(params);
    return 60;
  }
  
  return 0;
}

int
hashpool_manage(hashpool_t *h, char *object, char *action, msg *params) {
  int retv;

  hashpool_noerror(h);
  if(!strcmp(object, "pool"))
    retv = pool_manage(h, get_action(action), params);
  else if(!strcmp(object, "stats"))
    retv = stats_manage(h, get_action(action), params);
  else if(!strcmp(object, "node"))
    retv = node_manage(h, get_action(action), params);
  else if(!strcmp(object, "parameter"))
    retv = parameter_manage(h, get_action(action), params);
  else if(!strcmp(object, "schedule"))
    retv = schedule_manage(h, get_action(action), params);
  else if(!strcmp(object, "hash"))
    retv = hash_manage(h, get_action(action), params);
  else {
    hashpool_seterr(h, "no such object %s", object);
    return 1;
  }

  return retv;
}

#ifdef HASHPOOL_DEBUG

#include <unistd.h>
#include "bruteforce.h"
#include "dictionary.h"

#define DEF_LINELEN 1024

hashpool_t *h;

void *
client(void *param) {
  int id;
  char cid[1024];
  int sleeptime;
  workload_t *w;
  int start, finish;

  w = workload_init();
  id = *(int *)param;
  sprintf(cid, "client%d", id);
  sleep(1);

  llog(1, "%s: starting up\n", cid);
  while(!hashpool_done(h)) {
    if(hashpool_node_register(h, cid, w)) {
      llog(1, "hashpool_register: Out of keyspace\n");
      sleep(5);
      continue;
    }
    start = atoi(w->start);
    finish = atoi(w->finish);
    sleeptime = 1+((finish-start)/60000) + (rand() / (RAND_MAX/4));
    llog(1, "%s: Got %d %d, sleeping for %d\n", cid, 
	   start, finish, sleeptime);
    sleep(sleeptime);
    hashpool_node_update(h, cid, w->start, w->finish, NULL);
  }  
  llog(1, "%s: Done\n", cid);
  return NULL;
}

int
find_eta(int neta, char *buf, int len) {
  struct tm *stm;
  time_t lneta;
  char buf2[DEF_LINELEN];

  lneta = (time_t)neta;
  stm = gmtime(&lneta);  
  strftime(buf2, DEF_LINELEN, "%H:%M:%S", stm);
  snprintf(buf, len, "%d:%s", stm->tm_mday-1, buf2);
  return 0;
}

int
main(int argc, char **argv) {
  pthread_attr_t attr;
  pthread_t thrid;
  char eta[1024];
  int i;

  if(!(h = hashpool_init())) {
    printf("hashpool_init failed. Bah.\n");
    exit(1);
  }

  llog_init(LLOG_STDERR);
  llog_level(9);

  for(i=0; i<1; i++) {
    pthread_attr_init(&attr);
    pthread_attr_setscope(&attr, PTHREAD_SCOPE_SYSTEM);
    pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
    pthread_create(&thrid, &attr, client, (void *)&i);
    pthread_attr_destroy(&attr);
  }
  
  hashpool_schedule_add(h, "original_test", "bruteforce", "maxlen=6,poss=est01", "unixcrypt");
  hashpool_schedule_add(h, "original_test", "dictionary", "maxlen=6", "unixcrypt");
  hashpool_hash_add(h, "original_test", "unixcrypt", "aapmDihkgvxcs");
  hashpool_hash_add(h, "second_test", "unixcrypt", "aano7uKa0D6AU");
  hashpool_schedule(h);
  
  while(!hashpool_done(h)) {    
    hashpool_dump(h);
    find_eta(hashpool_eta(h), eta, DEF_LINELEN);
    sleep(1);
  }  

  hashpool_destroy(h);
  exit(0);
}

#endif
