// Copyright (C) 1994 The New York Group Theory Cooperative
// See magnus/doc/COPYRIGHT for the full notice.

// Contents: Definition of class QEqnSolutionsEnumerator
//
// Principal Author: Dmitry Pechkin, Eugeny Paderin
//
// Status: Under trial
//
// Description:
//
// * A composition of any stabilizing automorphism with any basic
//   solution gives a solution of the quadratic equation. Generally this 
//   solution is "parametrized", i.e. images of variables contain variables.
//   If so, then any specialization of this solution will also be
//   a solution.
//
//   We used algorithm described in:
//
//   L.Comerford, C.Edmunds, Solutions of equations in free groups,
//   in Group Theory: proceedings of the Singapore Group Theory
//   Conference held at the National University of Singapore, June 8-19,
//   1987.   ISBN 3-11-011366-X.
//
// Revision History:
//
// * 10/96 EP: GeneratorOfRandomSolutions added.
//
// Special Notes:
//
// * The greatest problem of the algorithm of enumeration of solutions 
//   is repeating solutions. Unfortunately, the same solution will repeat 
//   too often, but we do not know whether it is possible to avoid repetitions.
//   We took certain measures to reduce them, but there is still
//   a great field for improvements.
//

#ifndef _QEQN_ENUM_SOLUTIONS_H_
#define _QEQN_ENUM_SOLUTIONS_H_

#include "QEqnSolutions.h"
#include "TupleEnumerator.h"
#include "RandomNumbers.h"


class QEqnSolutionsEnumerator
{
public:

  /////////////////////////////////////////////////////////////////////////
  //                                                                     //
  // Constructors:                                                       //
  //                                                                     //
  /////////////////////////////////////////////////////////////////////////

  // no default constructor

  QEqnSolutionsEnumerator(const QEqnSolutionsInFreeGroup& equation);
  // Construct enumerator by given (partially) solved equation
  // with known basic solutions and generators of stabilizer.

  // Copy constructor, operator=, and destructor supplied by compiler

  /////////////////////////////////////////////////////////////////////////
  //                                                                     //
  // Activation members and status queries:                              //
  //                                                                     //
  /////////////////////////////////////////////////////////////////////////

  Endomorphism getSolution() const {
    if( done() )
      error("EQnSolutionsEnumerator::getSolution(): "
	    "all known solutions have been enumerated");
    return solution;
  }
  // get a current solution.

  bool nextSolution();
  // advances the enumerator. Returns true iff all solutions are enumerated.

  bool done() const { return allSolutionsAreEnumerated; }
  // Returns `true' iff all solutions of the equations have been enumerated.

  void reset();
  // Back to initial state of enumerator.

private:

  bool areBasicSolutionsParametrized();
  // Returns true iff there is a basic solution mapping theEquation 
  // to the word containing variables.

  /////////////////////////////////////////////////////////////////////////
  //                                                                     //
  // Data Members:                                                       //
  //                                                                     //
  /////////////////////////////////////////////////////////////////////////

  Word theEquation;     
  int theNumberOfGenerators;
  int theNumberOfVariables;
  FreeGroup theGroup; 

  VectorPtrOf<Automorphism> theRegStabGenerators;
  VectorPtrOf<Automorphism> theRegStabInvGenerators;
  VectorPtrOf<Endomorphism> theBasicSolutions;
  Automorphism prefixAuto; 

  bool allSolutionsAreEnumerated;
  Trichotomy numberOfSolutionsIsFinite;
  Endomorphism solution;
  Automorphism stabAuto;
  int currentBasicSolution;
  Endomorphism specEndo;
  EnumeratorOfWordTuples tuplesEnumerator;
  VectorOf<bool> invGensComputed;

};

//--------------------------- Random Solution ---------------------------------

const int randomSolutionsThreshold = 500;
// We cannot make all solutions to be different, there will be repetitions
// always. We can register solutions in the SetOf<Endomorphism> to eliminate
// "too close" repetitions. But we cannot do this endlessly, so sometimes
// we have to clean the set and start to fill it again. The constant
// defines maximal size of the set.


class GeneratorOfRandomSolutions 
{
public:

  GeneratorOfRandomSolutions( const FreeGroup& group, const Word& equation, 
			      int numberOfVariables );

  // copy constructor, operator= and destructor are generated by compiler

  /////////////////////////////////////////////////////////////////////////
  //                                                                     //
  //    Modifiers                                                        //
  //                                                                     //
  /////////////////////////////////////////////////////////////////////////

  void setBasicSolutions( const VectorPtrOf<Endomorphism>& newBasicSolutions )
  {
    basicSolutions = newBasicSolutions;
  }

  void setRegStabGenerators( const VectorPtrOf<Automorphism>& newGenerators )
  {
    regStabGenerators = newGenerators;
    regStabGeneratorsInv = VectorPtrOf<Automorphism>( newGenerators.length() );
  }

  void setThreshold( int newThreshold ) { threshold = newThreshold; }

  //////////////////////////////////////////////////////////////////////////
  //                                                                      //
  //    Accessors:                                                        //
  //                                                                      //
  //////////////////////////////////////////////////////////////////////////

  bool hasSolutions() const;
  // Returns true iff some basic solutions found so far.

  bool generateSolution();
  // This function generates a random solution made of the currently found
  // basic solutions and RegStab generators.
  // You should call hasSolutions before.

  Endomorphism getSolution() const { return solution; }
  // Returns the last generated solution.

  int getThreshold( ) const { return threshold; }

  int getCurrentThreshold( ) const { return solutionSet.cardinality(); }


  const VectorPtrOf<Endomorphism>& getBasicSolutions( ) const 
  {
    return basicSolutions;
  }

  const VectorPtrOf<Automorphism>& getRegStabGenerators(  ) const 
  {
    return regStabGenerators; 
  }


protected:

  void generateSomeSolution();
  // Generates solution not taking care about uniqueness

  // Data members

private:

  FreeGroup theGroup;
  int numOfVar;    // number of variables
  int numOfGen;    // number of generators


  VectorPtrOf<Endomorphism> basicSolutions;
  VectorPtrOf<Automorphism> regStabGenerators;
  VectorPtrOf<Automorphism> regStabGeneratorsInv;

  SetOf<Endomorphism> solutionSet;
  // The set of solutions

  Endomorphism solution;
  // The current solution

  int threshold;
  // The maximal number of registered solutions

  UniformRandom rnd;

  Endomorphism variablesEliminator;
};

#endif





  
