/*************************************************************************/
/*                                                                       */
/*  MacroSystem - Powerful C++ template system.                          */
/*                                                                       */
/*  http://projects.nn.com.br/                                           */
/*                                                                       */
/*  Copyright (C) 2000 Gustavo Niemeyer <gustavo@nn.com.br>              */
/*                                                                       */
/*  This library is free software; you can redistribute it and/or        */
/*  modify it under the terms of the GNU Library General Public          */
/*  License as published by the Free Software Foundation; either         */
/*  version 2 of the License, or (at your option) any later version.     */
/*                                                                       */
/*  This library is distributed in the hope that it will be useful,      */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU    */
/*  Library General Public License for more details.                     */
/*                                                                       */
/*  You should have received a copy of the GNU Library General Public    */
/*  License along with this library; if not, write to the                */
/*  Free Software Foundation, Inc., 59 Temple Place - Suite 330,         */
/*  Boston, MA  02111-1307, USA.                                         */
/*                                                                       */
/*************************************************************************/

// $Revision: 1.8 $
// $Date: 2000/03/14 20:02:52 $

#include "macrosystem.hh"

#ifdef DEBUG
#  include <fstream>
#endif

using namespace macrosystem;

/** Parse macros from given string.
 *
 *  It replaces found macros with equivalent macro and 
 *  search for new macros from the beginning of relpaced
 *  string, so it's recursive. But it DOESN'T handle
 *  infinit recursion. It handles nested macros.
 *
 *  @return Parsed string
 */

string
MacroSystem::parse(const string& _s)
{

  list<string> query_list;

  string query_name;
  string parsed = _s;
  string::size_type startpos = 0;

  enum macro_type_enum {
    SIMPLE,
    QUERY_START,
    QUERY_END,
    QUERY_INV,
    INVALID
  };

#ifdef DEBUG
  ofstream debug("macro_debug.txt");
  debug << "Start debugging!" << endl;
#endif

  while((startpos = parsed.find(m_mstart,startpos)) != string::npos)
    {
      string::size_type endpos = parsed.find(m_mend,startpos+m_mstart.length());
      if(endpos != string::npos && parsed.find(" ",startpos) >= endpos)
        {
	  macro_type_enum macro_type = SIMPLE;
	  string::size_type macro_size = endpos-startpos-m_mstart.length();
	  
	  if(macro_size>=m_mquery.length())
	    {
	      if(string(parsed,endpos-m_mquery.length(),m_mquery.length())
		 == m_mquery)
		macro_type = QUERY_START;
	      if(!query_list.empty())
		{
		  if(string(parsed,startpos+m_mstart.length(),m_mquery.length())
		     == m_mquery)
		    {
		      if((macro_type == QUERY_START)
			 && (macro_size>=(m_mquery.length()*2)))
			macro_type = QUERY_INV;
		      else
			macro_type = QUERY_END;
		    }
		}
	    }

	  string macro_string;
	  bool negate;

	  switch(macro_type)
	    {
	    case SIMPLE:
	      {
		macro_string = string(parsed,
				      startpos+m_mstart.length(),
				      macro_size);
		negate = this->m_is_negation(macro_string);
		break;
	      }

	    case QUERY_START:
	      {
		macro_string = string(parsed,
				      startpos+m_mstart.length(),
				      macro_size-m_mquery.length());
		negate = this->m_is_negation(macro_string);
		break;
	      }

	    case QUERY_END:
	      {
		macro_string = string(parsed,
				      startpos+m_mstart.length()+m_mquery.length(),
				      macro_size-m_mquery.length());
		negate = this->m_is_negation(macro_string);
		if(macro_string != query_list.back())
		  macro_type = INVALID;
		break;
	      }
	    case QUERY_INV:
	      {
		macro_string = string(parsed,
				      startpos+m_mstart.length()+m_mquery.length(),
				      macro_size-(2*m_mquery.length()));
		negate = !this->m_is_negation(macro_string);
		if(macro_string != query_list.back())
		  macro_type = INVALID;
		break;
	      }
	    case INVALID:
	      break;
	    }

	  string macro_name = macro_string;
	  this->m_remove_negation(macro_name);

#ifdef DEBUG
	  {
	    static const char* macro_type_string[] =
	    {
	      "SIMPLE",
	      "QUERY_START",
	      "QUERY_END",
	      "QUERY_INV",
	      "INVALID"
	    };
	    debug << "Found macro --------------------" << endl
		  << "Macro type:"   << macro_type_string[macro_type] << endl
		  << "Macro string:" << macro_string << endl
		  << "Macro name:"   << macro_name << endl;
	    if(query_list.size() > 0)
	      debug << "Query list back:" << query_list.back() << endl;
	  }
#endif // DEBUG

	  switch(macro_type)
	    {
	    case SIMPLE:
	      {
		this->m_msm.m_find(macro_name);
		if(this->m_msm.exist())
		  {
		    parsed.replace(startpos,endpos+m_mend.length()-startpos,
				   (*(this->m_msm.m_current)).second.first);
		    if(!(*(this->m_msm.m_current)).second.second)
		      startpos += (*(this->m_msm.m_current)).second.first.length();
		  }
		else
		  parsed.erase(startpos,endpos+m_mend.length()-startpos);
		break;
	      }

	    case QUERY_INV:
	    case QUERY_START:
	      {
		this->m_msm.m_find(macro_name);
		bool exists = this->m_msm.exist();
		if((!negate && !exists) || (negate && exists))
		  {
		    string query_start     = m_mstart+macro_string+m_mquery+m_mend;
		    string query_end_start = m_mstart+m_mquery+macro_string;
		    string query_inv_end   = m_mquery+m_mend;
		    // query_end_start+m_mend = query_end
		    // query_end_start+query_inv_end = query_inv
		    
		    enum status_type
		    {
		      NOTFOUND,
		      FOUNDEND,
		      FOUNDINV,
		      ENDOFMACRO
		    };

		    status_type status = NOTFOUND;
		    
		    string::size_type queryendpos = startpos;
		    string::size_type querylastendpos;
		    
		    do
		      {
			queryendpos++;
			querylastendpos = queryendpos;
			queryendpos = parsed.find(query_end_start,querylastendpos);
			if(queryendpos != string::npos)
			  {
			    if(string(parsed,queryendpos+query_end_start.length(),m_mend.length())
			       == m_mend)
			      status = FOUNDEND;
			    else if(string(parsed,queryendpos+query_end_start.length(),query_inv_end.length()) == query_inv_end)
			      status = FOUNDINV;

			    if(status != NOTFOUND)
			      {
				string inside_macro = string(parsed,querylastendpos,queryendpos+1-querylastendpos);
				if(inside_macro.find(query_start) != string::npos)
				  {
				    if(status == FOUNDINV)
				      {
					queryendpos = parsed.find(query_end_start+m_mend,queryendpos+1);
					// Look for the end_of_macro before go on
					// If we reach end_of_macro we'll get it on the next turn
				      }
				    else
				      status = NOTFOUND;
				  }
			      }
			  }
			else
			  status = ENDOFMACRO;

		      } while(status == NOTFOUND);
		    
		    if(macro_type == QUERY_INV) {
#ifdef DEBUG
		      debug << "Action: popping macro from query list" << endl;
#endif DEBUG
		      query_list.pop_back();
		    }

		    switch(status)
		      {
		      case FOUNDINV:
#ifdef DEBUG
			debug << "Action: erasing contents until invertion"
			      << endl;
			debug << "Action: pushing macro into query list"
			      << endl;
#endif
			parsed.erase(startpos,queryendpos-startpos);
			query_list.push_back(macro_string);
			// Do not erase invertion macro from parsed string
			// and insert it into the list so we can detect the
			// invertion on the next turn
			break;

		      case FOUNDEND:
#ifdef DEBUG
			debug << "Action: erasing contents and macro end declaration" << endl;
#endif
			parsed.erase(startpos,queryendpos-startpos+query_end_start.length()+m_mend.length());
			break;

		      case ENDOFMACRO:
#ifdef DEBUG
			debug << "Action: didn't find macro end" << endl;
			debug << "Action: erasing everything" << endl;
#endif
			parsed.erase(startpos,string::npos);
			break;

		      case NOTFOUND:
			break;

		      }
		  }
		else
		  {
#ifdef DEBUG
		    debug << "Action: erasing macro declaration" << endl;
		    debug << "Action: pushing macro into query list" << endl;
#endif
		    parsed.erase(startpos,endpos-startpos+m_mend.length());
		    if(macro_type == QUERY_START)
		      query_list.push_back(macro_string);
		    // If it is QUERY_INV we have already pushed_back()
		    // PS. If we change to != QUERY_INV it gives a beautiful
		    // error when compiled with -ggdb flag! :-)
		  }
		break;
	      }

	    case QUERY_END:
	      {
#ifdef DEBUG
		debug << "Action: erasing macro declaration" << endl;
		debug << "Action: popping macro from query list" << endl;
#endif
		parsed.erase(startpos,endpos-startpos+m_mend.length());
		query_list.pop_back();
		break;
	      }

	    case INVALID:
	      {
#ifdef DEBUG
		debug << "Action: erasing macro declaration" << endl;
#endif
		parsed.erase(startpos,endpos-startpos+m_mend.length());
		break;
	      }
	    }

	}
      else
	startpos++;
    }
  return parsed;
}

istream& macrosystem::operator>>(istream& _is, MacroSystem& _ms)
{
  enum state_enum {
    NAMESTART,
    NAME,
    VALUESTART,
    VALUE,
    IGNORE,
    MULTILINEVALUESTART,
    MULTILINEVALUE,
    MULTILINEIGNORE,
  } state;

  state = NAMESTART;

  string name;
  string value;
  bool parse;

  char c;

  while(_is.get(c))
    {
      switch(state)
	{
	case NAMESTART:
	  switch(c)
	    {
	    case '\r':
	    case '\n': break;
	    case ' ':
	    case '#': state = IGNORE; break;
	    default: name += c; state = NAME; break;
	    }
	  break;

	case NAME:
	  switch(c)
	    {
	    case '\r':
	    case '\n': name.erase(); state = NAMESTART;
	    case ':': state = VALUESTART; break;
	    case ' ': state = IGNORE; break;
	    default: name += c;
	    }
	  break;

	case VALUESTART:
	  switch(c)
	    {
	    case '\r': break;
	    case '\n': state = MULTILINEVALUESTART; break;
	    case ' ': state = VALUE; break;
	    default: value += c; state = VALUE;
	    }
	  break;

	case VALUE:
	  switch(c)
	    {
	    case '\r':
	    case '\n':
	      if(name[name.length()-1] == '*') {
		name.erase(name.length()-1);
		parse = false;
	      }
	      else
		parse = true;
	      _ms[name].set(value,parse);
	      name.erase();
	      value.erase();
	      state = NAMESTART;
	      break;

	    default: value += c;
	    }
	  break;

	case IGNORE:
	  switch(c)
	    {
	    case '\n': state = NAMESTART;
	    default: break;
	    }
	  break;

	case MULTILINEVALUESTART:
	  switch(c)
	    {
	    case '\r':
	    case '\n': break;
	    case '\t': state = MULTILINEVALUE; break;
	    case ' ':
	    case '#': state = MULTILINEIGNORE; break;
	    default:
	      if(name[name.length()-1] == '*') {
		name.erase(name.length()-1);
		parse = false;
	      }
	      else
		parse = true;
	      _ms[name].set(value,parse);
	      name.erase();
	      value.erase();
	      name += c;
	      state = NAME;
	    }
	  break;

	case MULTILINEVALUE:
	  switch(c)
	    {
	    case '\n': value += c; state = MULTILINEVALUESTART; break;
	    default: value += c;
	    }
	  break;

	case MULTILINEIGNORE:
	  switch(c)
	    {
	    case '\n': state = MULTILINEVALUESTART;
	    default: break;
	    }
	  break;

	default: break;

	}

    }

  if(state == MULTILINEVALUE || state == MULTILINEVALUESTART)
    {
      if(name[name.length()-1] == '*') {
	name.erase(name.length()-1);
	parse = false;
      }
      else
	parse = true;
      _ms[name].set(value,parse);
    }

  return _is;
}

ostream& macrosystem::operator<<(ostream& _os, MacroSystem& _ms)
{
  MacroSystem::iterator i;
  for(i=_ms.begin(); i!=_ms.end(); i++)
    {
      if(i->parse_on())
	_os << i->name() << ":";
      else
	_os << i->name() << "*:";
      string value = i->value();
      string::size_type pos;
      if((pos = value.find("\n")) != string::npos)
	{
	  value.insert(0,"\t");
	  while(pos != string::npos)
	    {
	      value.insert(pos+1,"\t");
	      pos = value.find("\n",pos+2);
	    }
	  if(value[value.length()-1] == '\t')
	    value[value.length()-1] = '\n';
	  else
	    value += "\n\n";
	  _os << "\n" << value;
	}
      else
	_os << " " << value << "\n";
    }
  return _os;
}  
    
