/***************************************************************************
 $RCSfile: accountjobs.h,v $
                             -------------------
    cvs         : $Id: accountjobs.h,v 1.22 2003/06/13 21:50:00 aquamaniac Exp $
    begin       : Mon Nov 26 2001
    copyright   : (C) 2001 by Martin Preuss
    email       : openhbci@aquamaniac.de

 ***************************************************************************
 *                                                                         *
 *   This library is free software; you can redistribute it and/or         *
 *   modify it under the terms of the GNU Lesser General Public            *
 *   License as published by the Free Software Foundation; either          *
 *   version 2.1 of the License, or (at your option) any later version.    *
 *                                                                         *
 *   This library is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU     *
 *   Lesser General Public License for more details.                       *
 *                                                                         *
 *   You should have received a copy of the GNU Lesser General Public      *
 *   License along with this library; if not, write to the Free Software   *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston,                 *
 *   MA  02111-1307  USA                                                   *
 *                                                                         *
 ***************************************************************************/


/*
 */


#ifndef ACCOUNTJOBS_H
#define ACCOUNTJOBS_H

#include <openhbci/pointer.h>
#include <openhbci/error.h>
#include <openhbci/value.h>
#include <openhbci/date.h>
#include <openhbci/balance.h>
#include <openhbci/transaction.h>
#include <openhbci/account.h>
#include <openhbci/user.h>
#include <openhbci/job.h>
#include <openhbci/standingorder.h>

namespace HBCI {
/**
 * This class reads your balance from the hbci server.
 * @author Martin Preuss<openhbci@aquamaniac.de>
 */
class DLLIMPORT JOBGetBalance: public Job {
private:
    Pointer<Account> _acc;
    AccountBalance _ab;

public:
    /**
     * Constructor.
     * @param cust customer whose account you like to inspect
     * @param acc pointer to the @ref accountParams describing your account
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    JOBGetBalance(Pointer<Customer> cust,
                  Pointer<Account> acc);

    ~JOBGetBalance();

    /**
     * internal
     */
    bool needsToBeSigned() const { return true; };

    /**
     * internal
     */
    bool needsToBeEncrypted() const { return true; };

     /**
     * internal
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    string toString(int firstseg);

    /**
     * internal
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    void parseResponse(const string& response);

    /**
     * Returns the balance read.
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    const AccountBalance &getBalance() const { return _ab; };
};


/**
 * This job transfers money from one account to another one.
 * @author Martin Preuss<openhbci@aquamaniac.de>
 */
class DLLIMPORT JOBSingleTransfer: public Job {
private:
    Transaction _xaction;

public:

    /**
     * Describes the parameters for this job.
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    struct jobData {
        /**
         * Maximum number of purpose lines allowed
         */
        int maxPurposeLines;
        /**
         * List of allowed text keys
         */
        list<int> textKeys;
        jobData();
        ~jobData();
    };

    /**
     * Constructor
     * @param cust the customer whose account you want to use
     * @param xa The transaction to be performed.
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    JOBSingleTransfer(Pointer<Customer> cust,
                      const Transaction& xa);

    ~JOBSingleTransfer();

    /**
     * internal
     */
    bool needsToBeSigned() const { return true; };

    /**
     * internal
     */
    bool needsToBeEncrypted() const { return true; };

     /**
     * internal
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    string toString(int firstseg);

    /**
     * internal
     * @author Martin Preuss<openhbci@aquamaniac.de>
    */
    void parseResponse(const string& response);

    /**
     * returns the job data for this job. You have to "delete" it when you
     * don't need it anymore
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    jobData *getJobData() const;
};



/**
 * This jobs reads a list of transactions for an account from your institute.
 * @author Martin Preuss<openhbci@aquamaniac.de>
 */
class DLLIMPORT JOBGetTurnover: public Job {
private:
    Pointer<Account> _acc;
    Date _fromdate;
    Date _todate;

    list<Transaction> _transactions;
    Balance _lastBalance;
    string _attachPoint;
public:

    /**
     * Describes the parameters for this job.
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    struct jobData {
	/**
	 * How long the transactions are saved
	 */
	int savedForDays;

	jobData();
	~jobData();
    };

    const list<Transaction> &transactions() const
    { return _transactions;};

    /**
     * Returns the last balance detected by this job. This can be used
     * if the bank does not support the JOBGetBalance.
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    const Balance &lastBalance() const { return _lastBalance;};

    /**
     * Constructor.
     * @param cust customer whose account you like to inspect
     * @param account the account to inspect
     * @param fromDate the date of the first transaction to be listed
     * if this is not set no start date is assumed
     * @param toDate the date of the last transaction to be listed
     * if not set no end date is assumed
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    JOBGetTurnover(Pointer<Customer> cust,
                   Pointer<Account> account,
                   const Date& fromDate,
                   const Date& toDate);
    /**
     * Internal Constructor.
     * @param cust customer whose account you like to inspect
     * @param account the account to inspect
     * @param fromDate the date of the first transaction to be listed
     * if this is not set no start date is assumed
     * @param toDate the date of the last transaction to be listed
     * if not set no end date is assumed
     * @param lastJob Previous job (only used if the bank sent an
     * attachPoint (Aufsetzpunkt)
     */
    JOBGetTurnover(Pointer<Customer> cust,
		   Pointer<Account> account,
		   const Date& fromDate,
		   const Date& toDate,
		   Pointer<Job> lastJob);

    ~JOBGetTurnover();

    /**
     * internal
     */
    bool needsToBeSigned() const { return true; };

    /**
     * internal
     */
    bool needsToBeEncrypted() const { return true; };

     /**
     * internal
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    string toString(int firstseg);

    /**
     * internal
     * @author Martin Preuss<openhbci@aquamaniac.de>
    */
    void parseResponse(const string& response);

    /**
     * returns the job data for this job.
     * Throws an exception if job is not supported
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    jobData getJobData() const;

    /**
     * internal
     */
    bool attachMore();
};


/**
 * This job emits a debit note ("Lastschrift").
 * @author Martin Preuss<openhbci@aquamaniac.de>
 */
class DLLIMPORT JOBDebitNote: public Job {
private:
    Transaction _xaction;

public:

    /**
     * Describes the parameters for this job.
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    struct jobData {
        /**
         * Maximum number of purpose lines allowed
         */
        int maxPurposeLines;
        /**
         * List of allowed text keys
         */
        list<int> textKeys;
        jobData();
        ~jobData();
    };

    /**
     * Constructor
     * @param cust the customer whose account you want to use
     * @param t The transaction representing the debit note to be issued.
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    JOBDebitNote(Pointer<Customer> cust,
                 const Transaction& t);

    ~JOBDebitNote();

    /**
     * internal
     */
    bool needsToBeSigned() const { return true; };

    /**
     * internal
     */
    bool needsToBeEncrypted() const { return true; };

     /**
     * internal
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    string toString(int firstseg);

    /**
     * internal
     * @author Martin Preuss<openhbci@aquamaniac.de>
    */
    void parseResponse(const string& response);

    /**
     * returns the job data for this job. You have to "delete" it when you
     * don't need it anymore
     * @author Martin Preuss<openhbci@aquamaniac.de>
     */
    jobData *getJobData() const;
};



/**
 * This jobs reads a list of standing orders for an account from 
 * your institute.
 * @author Fabian Kaiser<openhbci@derzach.de>
 */
class DLLIMPORT JOBGetStandingOrders: public Job {
private:
    Pointer<Account> _acc;
    string _attachPoint;
    list<Pointer<StandingOrder> > _orders;
public:

    /**
     * Constructor.
     * @param cust customer whose account you like to inspect
     * @param account the account to inspect
     */
    JOBGetStandingOrders(Pointer<Customer> cust,
                         Pointer<Account> account);

    JOBGetStandingOrders(Pointer<Customer> cust,
                         Pointer<Job> lastJob);
    ~JOBGetStandingOrders();

    const list<Pointer<StandingOrder> > &getOrders() const;

    /**
     * internal
     */
    bool needsToBeSigned() const { return true; };

    /**
     * internal
     */
    bool needsToBeEncrypted() const { return true; };

    /**
     * internal
     */
    string toString(int firstseg);

    /**
     * internal
     */
    void parseResponse(const string& response);

    /**
     * internal
     */
    bool attachMore();
};


/**
 * This job creates a new standing order.
 * @author Fabian Kaiser<openhbci@derzach.de>
 */
class DLLIMPORT JOBNewStandingOrder: public Job {
private:
    Pointer<Account> _acc;
    StandingOrder _newOrder;
public:

    /**
     * Constructor.
     * @param cust customer who wants to create a standing order
     * @param account the account from which money will be transfered
     * @param newOrder The new standing order to be added.
     */
    JOBNewStandingOrder(Pointer<Customer> cust,
                        Pointer<Account> account,
                        const StandingOrder& newOrder);

    ~JOBNewStandingOrder();

    /**
     * internal
     */
    bool needsToBeSigned() const { return true; };

    /**
     * internal
     */
    bool needsToBeEncrypted() const { return true; };

    /**
     * internal
     */
    string toString(int firstseg);

    /**
     * internal
     */
    void parseResponse(const string& response);
};


/**
 * This job deletes an existing standing order.
 * @author Fabian Kaiser<openhbci@derzach.de>
 */
class DLLIMPORT JOBDeleteStandingOrder: public Job {
private:
    Pointer<Account> _acc;
    StandingOrder _order2Delete;
public:

    /**
     * Constructor.
     * @param cust customer who wants to delete a standing order
     * @param account the account from which money is transfered
	 * @param order2Delete The standing order to be deleted
     */
    JOBDeleteStandingOrder(Pointer<Customer> cust,
                        Pointer<Account> account,
                        const StandingOrder& order2Delete);

    ~JOBDeleteStandingOrder();

    /**
     * internal
     */
    bool needsToBeSigned() const { return true; };

    /**
     * internal
     */
    bool needsToBeEncrypted() const { return true; };

    /**
     * internal
     */
    string toString(int firstseg);

    /**
     * internal
     */
    void parseResponse(const string& response);
};

} /* namespace HBCI */
#endif


