/*
 * dmnt.c -- Solaris mount functions for lslk
 *
 * V. Abell
 * Purdue University Computing Center
 */


/*
 * Copyright 1996 Purdue Research Foundation, West Lafayette, Indiana
 * 47907.  All rights reserved.
 *
 * Written by Victor A. Abell.
 *
 * This software is not subject to any license of the American Telephone
 * and Telegraph Company or the Regents of the University of California.
 *
 * Permission is granted to anyone to use this software for any purpose on
 * any computer system, and to alter it and redistribute it freely, subject
 * to the following restrictions:
 *
 * 1. Neither the authors nor Purdue University are responsible for any
 *    consequences of the use of this software.
 *
 * 2. The origin of this software must not be misrepresented, either by
 *    explicit claim or by omission.  Credit to the authors and Purdue
 *    University must appear in documentation and sources.
 *
 * 3. Altered versions must be plainly marked as such, and must not be
 *    misrepresented as being the original software.
 *
 * 4. This notice may not be removed or altered.
 */
#ifndef lint
static char copyright[] =
"@(#) Copyright 1996 Purdue Research Foundation.\nAll rights reserved.\n";
static char *rcsid = "$Id: dmnt.c,v 1.5 99/11/10 15:01:31 abe Exp $";
#endif


#include "lslk.h"


/*
 * readmnt() -- read mount table information
 */

int
readmnt()
{
	char c;
	dev_t dev;
	char *dn = NULL;
	int err, n, na, nl, v;
	MALLOC_S len;
	char *ln, *m, *o, *oo, *s;
	FILE *mfp;
	struct mnttab me, *mp;
	struct stat sb;
/*
 * Open the mount table (usually /etc/mnttab).
 */
	if ((mfp = fopen(MNTTAB, "r")) == NULL) {
	    (void) fprintf(stderr, "%s: can't open %s: %s\n",
		Pn, MNTTAB, strerror(errno));
	    return(1);
	}
/*
 * Read the mount table.
 */
	for (mp = &me, na = 0; getmntent(mfp, mp) == 0;) {

	/*
	 * Skip all but UFS devices -- i.e., the ones that have inodes.
	 */
	    if (strcmp(mp->mnt_fstype, MNTTYPE_UFS))
		continue;
	/*
	 * Get the MNTOPT_DEV mount option character address.
	 * Use Readlink() and statsafely() on mount entries without one.
	 */
	    if (!(oo = hasmntopt(mp, MNTOPT_DEV))) {

	    /*
	     * Interpolate a possible symbolic directory link.
	     */
		if (dn)
		    (void) free((MALLOC_P *)dn);
		if ((dn=(char *)malloc((MALLOC_S)(strlen(mp->mnt_mountp) + 1)))
		== (char *)NULL) {
		    (void) fprintf(stderr,
			"%s: no space for mount at %s (%s)\n",
			Pn, mp->mnt_special, mp->mnt_mountp);
		    Exit(1);
		}
		(void) strcpy(dn, mp->mnt_mountp);
		if ((ln = Readlink(dn)) == NULL) {
		    if (Owarn) {
			(void) fprintf(stderr,
			    "      Output information may be incomplete.\n");
		    }
		    continue;
		}
		if (ln != dn) {
		    (void) free((MALLOC_P *)dn);
		    dn = ln;
		}
		if (statsafely(dn, &sb)) {
		    (void) fprintf(stderr,
			"%s: can't stat() %s file system: %s\n",
			Pn, mp->mnt_mountp, strerror(errno));
		    (void) fprintf(stderr,
			"      Output information may be incomplete.\n");
		    continue;
		}
		dev = sb.st_dev;
	    } else {

	    /*
	     * Convert the value after MNTOPT_DEV= to a hexadecimal device
	     * number.
	     */
		o = oo + strlen(MNTOPT_DEV) + 1;
		if (strncasecmp(o, "0x", 2) == 0)
		    o += 2;
		for (nl = (sizeof(dev_t) * 2), dev = err = n = 0; *o; o++) {
		    if (isdigit(*o))
			v = (unsigned char)(*o - 0);
		    else if (isalpha(*o) && (c = toupper(*o)) >= 'A'
			 && c <= 'F')
		    {
			v = (unsigned char)(c - 'A' + 10);
		    } else
			break;
		    if (++n > nl) {
			err = 1;
			break;
		    }
		    dev = (dev << 4) | (v & 0xf);
		}
		if (err) {
		    (void) fprintf(stderr,
			"%s: can't convert device (%s) for %s: %s\n",
			Pn, oo, mp->mnt_special, mp->mnt_mountp);
		    return(1);
		}

#if	solaris>=70000 && L_BITSMAJOR!=L_BITSMAJOR32
	  	/*
		 * If this is a Solaris 7 system with a 64 bit kernel, convert
		 * a 32 bit device number to a 64 bit device number.
		 */
		    dev = (((dev >> L_BITSMINOR32) & L_MAXMAJ32)
			   << L_BITSMINOR)
			| (dev & L_MAXMIN32);
#endif	/* solaris>=70000 && L_BITSMAJOR!=L_BITSMAJOR32 */

	    }
	/*
	 * See if there's room in the local mount table for another entry.
	 */
	    if (NMnt >= na) {

	    /*
	     * Increase the space allocated to the local mount table and
	     * reserve space for the increase.
	     */
		na += 10;
		len = (MALLOC_S)(na * sizeof(struct mounts));
		if (Mnt)
		    Mnt = (struct mounts *)realloc((MALLOC_P *)Mnt, len);
		else
		    Mnt = (struct mounts *)malloc(len);
		if (!Mnt) {
		    (void) fprintf(stderr,
			"%s: no room for %d mount structures at %s: %s\n",
			Pn, na, mp->mnt_special, mp->mnt_mountp);
		    (void) fclose(mfp);
		    return(1);
		}
	    }
	/*
	 * Allocate space for what's been mounted (special) and where
	 * it's been mounted (mount point).
	 */
	    if (!(m = (char *)malloc(strlen(mp->mnt_mountp)+1))) {
		(void) fprintf(stderr, "%s: no room for mount point: %s\n",
		    Pn, mp->mnt_mountp);
		(void) fclose(mfp);
		return(1);
	    }
	    (void) strcpy(m, mp->mnt_mountp);
	    if (!(s = (char *)malloc(strlen(mp->mnt_special)+1))) {
		(void) fprintf(stderr, "%s: no room for special: %s\n",
			Pn, mp->mnt_special);
		(void) fclose(mfp);
		return(1);
	    }
	    (void) strcpy(s, mp->mnt_special);
	/*
	 * Create the new local mount information entry.
	 */
	    Mnt[NMnt].dev = dev;
	    Mnt[NMnt].sp = s;
	    Mnt[NMnt].priv = (caddr_t)NULL;
	    Mnt[NMnt++].mntp = m;
	}
	(void) fclose(mfp);
	return(0);
}
