package tests::LprngTest;

use base qw/Lire::Test::DlfConverterTestCase/;

use Lire::DlfConverters::Lprng;
use File::Basename;

sub schemasdir {
    return dirname( __FILE__ ) . "/../..";
}

sub create_converter {
    return new Lire::DlfConverters::Lprng();
}

sub registration_file {
    return dirname( __FILE__ ) . "/../../lprng_init";
}

sub test_process_log_line {
    my $self = $_[0];

    $self->set_up_tz( 'EDT' );

    my $process = $self->create_mock_process();
    my $converter = $process->converter();

    $converter->process_log_line( $process, "jobstart '-Hnavarro' '-nwytze' '-Plp' '-kcfA313navarro' '-b88562' '-t2000-11-29-07:56:42.000'" );

    $self->assert_deep_equals( { 'job-id' => 'cfA313navarro',
                                 'time' => '975484602.000' },
                               $converter->{'cache'}{'cfA313navarro'} );
    $self->assert_num_equals( 0, $process->dlf_count() );

    $converter->process_log_line( $process, "jobend '-Hnavarro' '-nwytze' '-Plp' '-kcfA313navarro' '-b88562' '-t2000-11-29-07:56:45.000'" );
    $self->assert_null( $converter->{'cache'}{'cfA313navarro'} );
    $self->assert_num_equals( 1, $process->dlf_count() );
    $self->assert_deep_equals( [ { 'job-id' => 'cfA313navarro',
                                   'time' => '975484602.000',
                                   'duration' => 3,
                                   'user' => 'wytze',
                                   'client_host' => 'navarro',
                                   'size' => 88562,
                                   'printer' => 'lp',
                                 } ],
                               $process->dlf( 'print' ) );

    $converter->process_log_line( $process, "jobend '-Hnavarro' '-nwytze' '-Pmyprinter' '-kother_id' '-t2000-11-29-07:56:45.000'" );
    $self->assert_null( $converter->{'cache'}{'cfA313navarro'} );
    $self->assert_num_equals( 2, $process->dlf_count() );
    $self->assert_deep_equals( { 'job-id' => 'other_id',
                                 'time' => '975484605.000',
                                 'duration' => 0,
                                 'user' => 'wytze',
                                 'client_host' => 'navarro',
                                 'size' => undef,
                                 'printer' => 'myprinter'
                               },
                               $process->dlf( 'print' )->[1] );
}

sub test_process_log_line_errors {
    my $self = $_[0];

    my $process = $self->create_mock_process();
    my $converter = $process->converter();

    $converter->process_log_line( $process, 'Bad line' );
    $self->assert_num_equals( 1, $process->error_count() );
    $self->assert_matches( qr/failed to parse/,
                           $process->errors()->[0][0] );
    $self->assert_str_equals( 'Bad line', $process->errors()->[0][1] );

    $converter->process_log_line( $process, "jobstart '-Hanhost'" );
    $self->assert_num_equals( 2, $process->error_count() );
    $self->assert_matches( qr/missing 'k' field/,
                           $process->errors()->[1][0] );
    $self->assert_str_equals( "jobstart '-Hanhost'",
                              $process->errors()->[1][1] );

    $converter->process_log_line( $process, "jobstart '-kan_id' '-tbad_time'" );
    $self->assert_num_equals( 3, $process->error_count() );
    $self->assert_matches( qr/bad time field/,
                           $process->errors()->[2][0] );

    $converter->process_log_line( $process, "jobfinish '-kan_id' '-t2003-03-11-15:45:32.123'" );
    $self->assert_num_equals( 4, $process->error_count() );
    $self->assert_matches( qr/unknown entry type/,
                           $process->errors()->[3][0] );

    $converter->process_log_line( $process, "jobfinish '-kan_id'" );
    $self->assert_num_equals( 5, $process->error_count() );
    $self->assert_matches( qr/missing 't' field/,
                           $process->errors()->[4][0] );
}

sub test_finish_conversion {
    my $self = $_[0];

    my $process = $self->create_mock_process();
    my $converter = $process->converter();

    $converter->finish_conversion( $process );
    $self->assert_num_equals( 0, $process->error_count() );
    $self->assert_num_equals( 0, $process->dlf_count() );

    $converter->{'cache'}{'an_id'} = { 'job-id' => 'an_id',
                                       'time' => time,
                                     };
    $converter->{'cache'}{'another_id'} = { 'job-id' => 'another_id',
                                            'time' => time + 10,
                                          };

    my $e_dlf = [ $converter->{'cache'}{'an_id'},
                  $converter->{'cache'}{'another_id'} ];
    $converter->finish_conversion( $process );
    $self->assert_num_equals( 0, $process->error_count() );
    $self->assert_num_equals( 2, $process->dlf_count() );
    $self->assert_deep_equals( $e_dlf, $process->dlf( 'print' ) );
    $self->assert_deep_equals( {}, $converter->{'cache'} );
}

1;
