package Lire::OutputFormats::Text;

use strict;

use base qw/Lire::OutputFormat/;

use Locale::TextDomain 'lire';

use Lire::ReportParser::AsciiWriter;
use Lire::Utils qw/tempdir/;
use Lire::Error qw/ file_not_writable /;

eval "use MIME::Entity";

=pod

=head1 NAME

Lire::OutputFormats::Text - Text output format.

=head1 SYNOPSIS

  my $format = Lire::PluginManager->get_plugin( 'output_format', 'text' );
  $format->format_report( $xml_file, $text_file, $cfg );

=head1 DESCRIPTION

Format a XML report into plain text.

=cut

sub name {
    return 'txt';
}

sub title {
    return __( 'Plain Text' );
}

sub description {
    return '<para>' . __( 'Plain Text format.' ) . '</para>';
}

sub format_report {
    my ( $self, $report_file, $output_file, $cfg ) = @_;

    open my $fh, "> $output_file"
      or die file_not_writable( $output_file ), "\n";
    my $parser =
      new Lire::ReportParser::AsciiWriter( 'output' => $fh,
                                           'encoding' => $cfg->{'encoding'} );
    $parser->parsefile( $report_file );
    close $fh;

    return;
}

sub mime_report {
    my ( $self, $report_file, $cfg ) = @_;

    my $tmpdir = tempdir( __PACKAGE__ . '_XXXXXX', 'CLEANUP' => 1 );
    $self->format_report( $report_file, "$tmpdir/report.txt", $cfg );
    return MIME::Entity->build( 'Type' => 'text/plain',
                                'Charset' => $cfg->{'encoding'},
                                'Encoding' => 'quoted-printable',
                                'Path' => "$tmpdir/report.txt"
                              );
}

1;

__END__

=pod

=head1 SEE ALSO

 Lire::PluginManager(3pm) Lire::OutputFormat(3pm)

=head1 AUTHOR

Francis J. Lacoste <flacoste@logreport.org>

=head1 VERSION

$Id: Text.pm,v 1.1 2004/08/28 03:41:08 flacoste Exp $

=head1 COPYRIGHT

Copyright (C) 2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut

