package Lire::Field;

use strict;

use Lire::DataTypes qw/ check_xml_name check_type is_numeric_type/;
use Lire::I18N qw/ dgettext dgettext_para /;
use Lire::Utils qw/ check_param /;

=pod

=head1 NAME

Lire::Field - Interface to Dlf field's specification.

=head1 SYNOPSIS

    my $field = $schema->field( 'time' );
    print "Name: ", $field->name(), "\n", "Type: ", $field->type(), "\n";

=head1 DESCRIPTION

This module represents Dlf field's specifications. You'll get
reference to this kind of object when you call the field(),
timestamp_field() or fields() methods on a Lire::DlfSchema(3pm)
object. You can obtain information about this field by querying its
methods.

=head1 OBJECT METHODS

=cut

sub new {
    my ($class, %params) = @_;

    check_param( $params{'name'}, 'name', \&check_xml_name,
                 "'name' parameter should be a valid XML name" );
    check_param( $params{'type'}, 'type', \&check_type,
                 "'type' parameter should be a valid Lire type" );

    return bless { 'name'	 => $params{'name'},
                   'type'	 => $params{'type'},
                   'i18n_domain' => $params{'i18n_domain'} || 'lire',
                   'pos'	 => undef,
                   'description' => $params{'description'},
                   'label'	 => $params{'label'},
                 }, $class;
}

=pod

=head2 name()

Returns the name of the field.

=cut

sub name {
    return $_[0]->{'name'};
}

=pod

=head2 label()

Returns the default label that should be used for column containing
values from this field.

=cut

sub label {
    my $self = $_[0];

    return dgettext( $self->{'i18n_domain'},
                     $self->{'label'} ? $self->{'label'} : $self->{'name'} );
}

=pod

=head2 pos()

Returns the field's position in the record. Field's positions are
indexed from 0.

=cut

sub pos {
    return $_[0]->{'pos'};
}

=pod

=head2 type()

Returns the field's type. This will be a string describing the type's
name as used in the XML specifications.

=cut

sub type {
    return $_[0]->{'type'};
}

=pod

=head2 sql_type()

Returns the SQL type that should be used to hold this field's values.

=cut

sub sql_type {
    my $self = $_[0];

    if ( $self->{'type'} =~ /^(timestamp|date|time)$/ ) {
        return "TIMESTAMP";
    } elsif ( $self->{'type'} =~ /^(bytes|duration)$/ ) {
        return "NUMBER(10,1)"
    } elsif ( $self->{'type'} eq "bool" ) {
        return "NUMBER(1,0)";
    } elsif ( $self->{'type'} eq "number" ) {
        return "NUMBER(10,5)";
    } elsif ( $self->{'type'} eq "int" ) {
        return "NUMBER(10,0)";
    } elsif ( $self->{'type'} eq "id" ) {
        return "INTEGER PRIMARY KEY";
    } else {
        return "VARCHAR(512)";
    }
}

=pod

=head2 default()

This method is obsolete since field don't have default anymore.

=cut

sub default {
    return 'LIRE_NOTAVAIL';
}

=pod

=head2 description()

This method will return the field's description. (This is the content
of the description element in the XML specification.) Be aware that
this will most likely contains DocBook markup.

=cut

sub description {
    return dgettext_para( $_[0]->{'i18n_domain'}, $_[0]->{'description'} );
}

# keep perl happy
1;

__END__

=pod

=head1 SEE ALSO

Lire::DlfSchema(3pm) Lire::DataTypes(3pm)

=head1 AUTHORS

  Francis J. Lacoste <flacoste@logreport.org>
  Wolfgang Sourdeau <wolfgang@logreport.org>

=head1 VERSION

$Id: Field.pm,v 1.17 2004/05/17 17:05:19 wsourdeau Exp $

=head1 COPYRIGHT

Copyright (C) 2001-2004 Stichting LogReport Foundation LogReport@LogReport.org

This file is part of Lire.

Lire is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program (see COPYING); if not, check with
http://www.gnu.org/copyleft/gpl.html or write to the Free Software 
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.

=cut
