/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdarg.h>
#include <unistd.h>
#include <time.h>
#include <sys/time.h>
#include <sys/socket.h>
#include "util.h"
#include "error.h"
#include "debug.h"
#include "rc.h"
#include "streamer.h"

FILE		*StreamerLog;
extern int	Debug;

void
streamer_free(STREAMER sr)
{
	if (sr)
	{
		if (sr->priv)
			free(sr->priv);
		free(sr);
	}
}

ssize_t
streamer_write_ascii(int fd, void *buf, size_t count)
{
	if (StreamerLog)
		fprintf(StreamerLog, "> %.*s\n", (int) count, (char *)buf);

	if (Debug >= 5)
	{
		void	*p;
		int	cnt = count;

		p = strchr(buf, '\r');
		if (!p) p = strchr(buf, '\n');
		if (p) cnt = p - buf;
		timestamp(stderr);
		fprintf(stderr, "> %.*s\n", (int) cnt, (char *)buf);
	}
	return write(fd, buf, count);
}

ssize_t
streamer_write_binary(int fd, void *buf, size_t count)
{
	if (StreamerLog)
		fprintf(StreamerLog, "> %.*s\n", (int) count, (char *)buf);

	if (Debug >= 5)
	{
		timestamp(stderr);
		hexdump(stderr, ">", "            ", buf, count);
	}
	return write(fd, buf, count);
}

int
streamer_printf(int fd, char *fmt, ...)
{
	va_list ap;
	char	buf[4096];
	int	len;

	if (!fmt)
		return -2;

	va_start(ap, fmt);
	len = vsnprintf(buf, sizeof(buf), fmt, ap);
	va_end(ap);

	if (len >= sizeof(buf))
		error(1, "Buffer too small in streamer_printf\n");

	if (StreamerLog)
		fprintf(StreamerLog, "> %s\n", buf);

	if (Debug >= 5)
	{
		char	*p;
		int	cnt = len;

		p = strrchr(buf, '\r');
		if (!p) p = strrchr(buf, '\n');
		if (p) cnt = p - buf;
		timestamp(stderr);
		fprintf(stderr, "> %.*s\n", (int) cnt, (char *)buf);
	}

	return write(fd, buf, len);
}

int
streamer_printf2(int fd, char *fmt, ...)
{
	va_list ap;
	char	buf[4096];
	int	len;

	if (!fmt)
		return -2;

	va_start(ap, fmt);
	len = vsnprintf(buf, sizeof(buf), fmt, ap);
	va_end(ap);

	if (len >= sizeof(buf))
		error(1, "Buffer too small in streamer_printf\n");

	if (StreamerLog)
		fprintf(StreamerLog, "2> %s\n", buf);

	if (Debug >= 5)
	{
		char	*p;
		int	cnt = len;

		p = strrchr(buf, '\r');
		if (!p) p = strrchr(buf, '\n');
		if (p) cnt = p - buf;
		timestamp(stderr);
		fprintf(stderr, "2> %.*s\n", (int) cnt, (char *)buf);
	}

	return write(fd, buf, len);
}

ssize_t
streamer_read(STREAMER sr, int fdindex, void *buf, size_t count)
{
	ssize_t	len;
	int	fd = sr->fd[fdindex];

	len = read(fd, buf, count);
	if (len <= 0)
	{
		if (Debug >= 5)
		{
			timestamp(stderr);
			fprintf(stderr, "< fd=%d, rc=%d\n", fd, len);
		}
		return len;
	}
	sr->cnt_rx += len;

	if (Debug >= 5)
	{
		timestamp(stderr);
		hexdump(stderr, "<", "            ", buf, len);
	}

	if (StreamerLog)
		hexdump(StreamerLog, "<", " ", buf, len);

	if (sr->writefile)
		fwrite(buf, 1, len, sr->writefile);

	return len;
}

ssize_t
streamer_mustread(STREAMER sr, int fdindex, void *buf, size_t count)
{
    unsigned char *p = buf;
    int		  len = count;
    int		  rc;

    while (len > 0)
    {
	rc = streamer_read(sr, fdindex, p, len);
	if (rc <= 0)
	    return rc;
	p += rc;
	len -= rc;
    }
    return count;
}

void
linebuf_init(LINEBUF *lb)
{
	lb->fd = -1;
	lb->buf = NULL;
	lb->buflen = 0;
	lb->bufp = NULL;
	lb->bufe = NULL;
	lb->cnt_rx = 0;
	lb->debug_tag = "< ";
}
void
linebuf_tag(LINEBUF *lb, char *tag)
{
	lb->debug_tag = tag;
}

int
linebuf_open(LINEBUF *lb, int fd, int buflen)
{
	// If there is already a buffer, see if we can reuse it
	if (lb->buf)
	{
		if (lb->buflen != buflen)
		{
			free(lb->buf);
			lb->buf = NULL;
			lb->buflen = 0;
		}
	}
	if (!lb->buf)
		lb->buf = malloc(buflen);
	if (!lb->buf)
		return -1;
	lb->buflen = buflen;

	lb->fd = fd;
	lb->bufp = NULL;
	lb->bufe = NULL;

	return (0);
}

void
linebuf_close(LINEBUF *lb)
{
	lb->fd = -1;
	if (lb->buf)
	{
		free(lb->buf);
		lb->buf = NULL;
	}
	lb->bufp = NULL;
	lb->bufe = NULL;
	lb->buflen = 0;
}

int
linebuf_avail(LINEBUF *lb)
{
	if (!lb->bufp)
		return 0;
	return (lb->bufe - lb->bufp);
}

/*
 * Read a line from a LINEBUF.  Discard the \n.  Buffer unused
 * data to be returned later.  Return length of line read.
 */
int
linebuf_gets(LINEBUF *lb, char *buf, int buflen)
{
	int	rc;
	char	*sp, *ep;

	if (lb->bufp == NULL)
	{
		rc = read_timeout(lb->fd, lb->buf, lb->buflen, 20);
		if (rc <= 0)
			return (rc);
		lb->cnt_rx += rc;
		lb->bufp = lb->buf;
		lb->bufe = &lb->buf[rc];
		*lb->bufe = '\0';
		if (StreamerLog)
			fprintf(StreamerLog, "%s%s\n",
					lb->debug_tag, lb->buf);
	}

	sp = lb->bufp;
	ep = strchr(sp, '\n');
	while (!ep)
	{
		rc = read_timeout(lb->fd, lb->bufe,
				&lb->buf[lb->buflen] - lb->bufe, 20);
		if (rc <= 0)
			return (rc);
		if (StreamerLog)
		{
			lb->bufe[rc] = '\0';
			fprintf(StreamerLog, "%s%s\n",
					lb->debug_tag, lb->bufe);
		}
		lb->cnt_rx += rc;
		ep = strchr(lb->bufe, '\n');
		lb->bufe += rc;
		*lb->bufe = '\0';
	}
	*ep++ = 0;
	if (ep >= lb->bufe)
		lb->bufp = NULL;
	else
		lb->bufp = ep;

	if ( (ep-sp) > buflen)
		error(1, "Buffer too small for streamer line\n");

	memcpy(buf, sp, ep-sp);

	if (Debug >= 5)
	{
		timestamp(stderr);
		if (ep-sp)
			fprintf(stderr, "%s%s\n", lb->debug_tag, buf);
		else
			fprintf(stderr, "%s0 length\n", lb->debug_tag);
	}

	return (ep-sp);
}
