/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

/*
 * quotemedia.com
 *
 * Line oriented ASCII using stdio buffering.  Uses helper application
 * linuxtrade.qm to handle authentication, symbol changes, and to open new
 * connection when symbols change.
 */
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <string.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include "error.h"
#include "debug.h"
#include "rc.h"
#include "streamer.h"
#include "linuxtrade.h"
#include "util.h"
#include "p2open.h"

#include "optchain.h"
#include "info.h"

#define         SYMCNT          200
#define         SYMBUFLEN       (SYMCNT * (SYMLEN+1) + 1)

typedef struct streamerpriv
{
	FILE	*fp[2];

	time_t	last_flush;

	char	symbuf[SYMBUFLEN];
	int	symcnt;
} STREAMERDATA;

typedef struct
{
	char	*canon, *sym;
} SYMMAP;

static SYMMAP SymMap[] =
{
	{	"$DJI",		".DJI",		},
	{	"$DJT",		".DJTA",	},
	{	"$DJU",		".DJUA",	},
	{	"$NYA",		".NYA",		},
	{	"$COMP",	"COMP",		},
	{	"$NDX",		".NDX",		},
	{	"$SPX",		".SPX",		},
	{	"$OEX",		".OEX",		},
	{	"$MID",		".MID",		},
	{	"$SML",		".SML",		},
	{	"$RLX",		".RLX",		},
	{	"$XAX",		".XAX",		},
	{	"$IIX",		".IIX",		},
	{	"$BTK",		".BTK",		},
	{	"$XBD",		".XBD",		},
	{	"$DRG",		".DRG",		},
	{	"$XTC",		".XTC",		},
	{	"$GSO",		".GSO",		},
	{	"$HWI",		".HWI",		},
	{	"$RUI",		".RUI",		},
	{	"$RUT",		".RUT",		},
	{	"$RUA",		".RUA",		},
	{	"$SOX",		".SOX",		},
	{	"$OSX",		".OSX",		},
	{	"$XAU",		".XAU",		},
	{	"$GOX",		".GOX",		},
	{	NULL,		NULL		}
};

//
// Convert canonical index names to/from streamer index names
//
static void
streamer_canon2sym(char *out, char *in)
{
	char	*ip, *op;
	char	*p;
	int	len;
	SYMMAP	*map;

	ip = in;
	op = out;
	for (;;)
	{
		p = strchr(ip, '|');
		if (!p) p = strchr(ip, ',');
		if (!p) p = strchr(ip, ' ');
		if (!p) p = strchr(ip, 0);

		len = p - ip;
		memcpy(op, ip, len); op[len] = 0;

		for (map = SymMap; map->canon; ++map)
			if (strcmp(op, map->canon) == 0)
			{
				strcpy(op, map->sym);
				break;
			}

		if (*p == 0)
			break;

		ip += len + 1;
		op = strchr(op, 0);
		*op++ = *p;
		*op = 0;
	}
}

static void
streamer_sym2canon(char *out, char *in)
{
	SYMMAP	*map;

	for (map = SymMap; map->sym; ++map)
		if (strcmp(in, map->sym) == 0)
		{
			strcpy(out, map->canon);
			return;
		}

	if (in != out)
		strcpy(out, in);
}

static void
streamer_init(STREAMER sr)
{
	sr->refresh = 0;
	sr->fd[0] = -1;
	sr->nfd = 1;
	strcpy(sr->id, "quotemedia");

	time(&sr->priv->last_flush);

	sr->priv->fp[0] = NULL;
	sr->priv->fp[1] = NULL;

	sr->priv->symcnt = 0;
}

static int 
streamer_open( STREAMER sr, RCFILE *rcp, FILE *readfile)
{
	int	rc;

	streamer_init(sr);
	++sr->cnt_opens;
	++sr->cnt_realopens;
	time(&sr->time_open);
	time(&sr->time_realopen);

	rc = p2open("/bin/bash", PROGNAMESTR ".qm 2>/dev/null",
			sr->priv->fp);
	if (rc < 0)
		return -1;

	fprintf(sr->priv->fp[1], "%s\n", get_rc_value(rcp, "username"));
	fprintf(sr->priv->fp[1], "%s\n", get_rc_value(rcp, "password"));
	fprintf(sr->priv->fp[1], "%s\n", get_rc_value(rcp, "hostname"));
	fprintf(sr->priv->fp[1], "%s\n", get_rc_value(rcp, "port"));
	fprintf(sr->priv->fp[1], "GE,\n");
	fflush(sr->priv->fp[1]);

	sr->fd[0] = fileno(sr->priv->fp[0]);
	if (sr->fd[0] <= 0)
		return -1;
	debug(5, "fd=%d\n", sr->fd[0]);
	return 0;
}
static int
streamer_select(
		STREAMER sr,
		int n, fd_set *readfds, fd_set *writefds,
		fd_set *exceptfds, struct timeval *timeout
		)
{
	if (readfds && FD_ISSET(sr->fd[0], readfds) && FRcnt(sr->priv->fp[0]))
	{
		FD_ZERO(readfds);
		FD_SET(sr->fd[0], readfds);
		if (writefds)
			FD_ZERO(writefds);
		if (exceptfds)
			FD_ZERO(exceptfds);
		return 1;
	}
	return select(n, readfds, writefds, exceptfds, timeout);
}

static void
streamer_close(STREAMER sr)
{
	p2close(sr->priv->fp);
	sr->fd[0] = -1;
}

static void
streamer_record(STREAMER sr, FILE *fp)
{
	sr->writefile = fp;
}

static void
streamer_timetick(STREAMER sr, time_t now)
{
	if (now > sr->priv->last_flush + 5)
	{
		if (sr->writefile)
			fflush(sr->writefile);
	}
}

static void
send_stocklist(STREAMER sr)
{
	char	buf[SYMBUFLEN];
	char	*p;
	int	i;

	// Walk list of all stocks and write symbols to streamer process
	p = buf;
	for (i = 0; i < NumStock; ++i)
	{
		char sym[SYMLEN+1];

		streamer_canon2sym(sym, Stock[i].sym);
		p += sprintf(p, "%s,", sym);
	}

	fprintf(sr->priv->fp[1], "%s\n", buf);
	fflush(sr->priv->fp[1]);

	if (Debug >= 5)
	{
		timestamp(stderr);
		fprintf(stderr, "> %s\n", buf);
	}
}

static void
streamer_send_quickquote(STREAMER sr, char *sym)
{
	char	csym[SYMLEN+1];

	if (sr->fd[0] < 0 || sr->readfile)
		return;

	streamer_canon2sym(csym, sym);
	fprintf(sr->priv->fp[1], "%s,\n", csym);
	fflush(sr->priv->fp[1]);

	if (Debug >= 5)
	{
		timestamp(stderr);
		fprintf(stderr, "> %s,\n", csym);
	}

	send_stocklist(sr);
}

static void
streamer_send_livequote(STREAMER sr, char *sym)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return;

	if (sr->priv->symcnt >= SYMCNT)
		return;

	if (sr->priv->symcnt == 0)
		strcpy(sr->priv->symbuf, "");

	strcat(sr->priv->symbuf, sym);
	strcat(sr->priv->symbuf, ",");
	++sr->priv->symcnt;
}

static void
streamer_send_livequote_end(STREAMER sr)
{
	if (!sr->priv->symcnt)
		return;

	sr->priv->symcnt = 0;

	fprintf(sr->priv->fp[1], "%s\n", sr->priv->symbuf);
	fflush(sr->priv->fp[1]);

	if (Debug >= 5)
	{
		timestamp(stderr);
		fprintf(stderr, "> %s\n", sr->priv->symbuf);
	}

	send_stocklist(sr);
}

static void
streamer_send_symbols(STREAMER sr, char *symbols, int add) {}

static void
streamer_send_symbols_end(STREAMER sr, int add, int all)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return;

	// Its very expensive to change symbols, so ignore deletes
	if (!add)
		return;

	// Send all the symbols in the current stocklist
	send_stocklist(sr);
}

static int
adv(char *p, char sep)
{
	char	*b = p;

	while (*p && *p++ != sep)
		{}
	return (p - b);
}

static void
do_fullquote(unsigned char *buf)
{
	char		*p;
	int		rc;
	int		len;
	char		*exch;
	QUOTE		q;
	QUICKQUOTE	qq;
	LIVEQUOTE	lq;
	int		hh, mm;

	// SUNW|SUN MICROSYS INC|12.110|-.420|-3.46|12.150|12.290|11.950|
	// 12.530|0|12.230|500|12.260|200|34|60513100|35.125|7.520|16:0|
	// NSD|realtime|0|0|N/A


	memset(&q, 0, sizeof(q));
	memset(&qq, 0, sizeof(qq));

	p = buf;
	len = adv(p, '|');
	if (len > SYMLEN+1)
		return;
	strncpy(q.sym, p, len-1); q.sym[len-1] = 0;

	streamer_sym2canon(q.sym, q.sym);

	strcpy(qq.sym, q.sym);
	strcpy(lq.sym, q.sym);
	p += len;

	sscanf(p, "%[^|]", qq.fullname);
	len = adv(p, '|'); /* fullname */; p += len;

	len = adv(p, '|'); lq.last = qq.last = q.last = atof(p); p += len;
	len = adv(p, '|'); /* change, dollars */; p += len;
	len = adv(p, '|'); /* change, percent */; p += len;
	len = adv(p, '|'); /* unknown */; p += len;
	len = adv(p, '|'); qq.high = q.high = atof(p); p += len;
	len = adv(p, '|'); qq.low = q.low = atof(p); p += len;

	len = adv(p, '|'); lq.close=qq.prev_close = q.close = atof(p); p += len;
	if (lq.last == 0)
		lq.last = qq.last = q.last = q.close;

	len = adv(p, '|'); /* 0=tickdown, 1=tickup */; p += len;
	len = adv(p, '|'); qq.bid = q.bid = atof(p); p += len;
	len = adv(p, '|'); q.bid_size = atoi(p) / 100; p += len;
	len = adv(p, '|'); qq.ask = q.ask = atof(p); p += len;
	len = adv(p, '|'); q.ask_size = atoi(p) / 100; p += len;
	len = adv(p, '|'); q.last_size = atoi(p) * 100; p += len;
	len = adv(p, '|'); qq.volume = q.volume = atoi(p); p += len;

	len = adv(p, '|'); qq.high52 = atof(p); p += len;
	len = adv(p, '|'); qq.low52 = atof(p); p += len;

	rc = sscanf(p, "%d:%d", &hh, &mm);
		if (rc != 2) return;
		qq.timetrade = q.time = hh * 3600 + mm * 60 + 0;
	len = adv(p, '|'); p += len;

	len = adv(p, '|'); exch = p; p += len;
	if (*p == 'd')
		q.delayed = 1;
	len = adv(p, '|'); /* delayed|realtime */; p += len;
	len = adv(p, '|'); /* unknown */; p += len;
	len = adv(p, '|'); /* UPC code 1=restricted */; p += len;
	len = adv(p, '|'); /* unknown */; p += len;

	if (strncmp(exch, "NSD", 3) == 0) qq.exchange = 'Q';
	else if (strncmp(exch, "OBB", 3) == 0) qq.exchange = 'V';
	else if (strncmp(exch, "NYE", 3) == 0) qq.exchange = 'N';
	else if (strncmp(exch, "AMX", 3) == 0) qq.exchange = 'A';
	else qq.exchange = '?';

	qq.bid_id = qq.ask_id = '?';
	qq.last_eps = 12345678;
	qq.cur_eps = 12345678;
	qq.sharesout = 12345678;

	display_quote(&q, 0);
	optchain_quote(&q);

	info_quickquote(&qq);

	display_livequote(&lq);
}

static int
streamer_process(STREAMER sr, int fdindex)
{
	char	buf[2048];
	char	*bp = buf;

	if (!fgets(buf, sizeof(buf), sr->priv->fp[0]))
		return -1;
	sr->cnt_rx += strlen(buf);

	if (sr->writefile)
		fputs(buf, sr->writefile);

	if (Debug >= 5)
	{
		bp = strchr(buf, '\n'); if (bp) *bp = 0;
		timestamp(stderr);
		fprintf(stderr, "< %s\n", buf);
	}

	do_fullquote(buf);

	return 0;
}

STREAMER
quotemedia_new(void)
{
	STREAMER	sr;

	sr = (STREAMER) malloc(sizeof(*sr));
	if (!sr)
		return NULL;
	memset(sr, 0, sizeof(*sr));

	sr->open = streamer_open;
	sr->select = streamer_select;
	sr->close = streamer_close;
	sr->record = streamer_record;
	sr->timetick = streamer_timetick;

	sr->send_quickquote = streamer_send_quickquote;
	sr->send_livequote = streamer_send_livequote;
	sr->send_livequote_end = streamer_send_livequote_end;
	sr->send_symbols = streamer_send_symbols;
	sr->send_symbols_end = streamer_send_symbols_end;

	// sr->send_disconnect = streamer_send_disconnect;
	// sr->send_top10 = streamer_send_top10;
	// sr->send_movers = streamer_send_movers;
	// sr->send_info = streamer_send_info;
	// sr->send_optchain = streamer_send_optchain;
	// sr->send_chart = streamer_send_chart;

	sr->process = streamer_process;

	sr->priv = (STREAMERDATA *) malloc(sizeof(*sr->priv));
	if (!sr->priv)
	{
		free(sr);
		return NULL;
	}

	time(&sr->time_start);

	streamer_init(sr);

	return (sr);
}
