/*b
 * Copyright (C) 2001,2002  Rick Richardson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * Author: Rick Richardson <rickr@mn.rr.com>
b*/

/*
 * datek.com
 *
 * ASCII command letter folowed by binary data.  Opens a new connection
 * whenever symbols change.
 *
 * N.B. Needs code added for livequotes.
 */
#include <stdio.h>
#include <stdlib.h>
#include <ncurses.h>
#include <panel.h>
#include <errno.h>
#include <string.h>
#include <ctype.h>
#include <time.h>
#include <sys/time.h>
#include <unistd.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include "error.h"
#include "rc.h"
#include "streamer.h"
#include "linuxtrade.h"
#include "util.h"
#include "p2open.h"
#include "debug.h"
#include "curse.h"

#include "optchain.h"
#include "info.h"

#if 0
Someday, bring up a second connection (like with Schwab) and get actives....

GET /%21U%3D778679944%26W%3Da2561191a30f6668b4fed9fe0ada9397e719bfd9%26A%3Duserid%3D776887590%26token%3Da25611e9a30f6668b4fed95e0ad29397e719bfd9%26company%3DAMER%26segment%3DAMER%26cddomain%3DA000000007623003%26usergroup%3DACCT%26accesslevel%3DACCT%26authorized%3DY%26acl%3DAQDRMAOLQ1TETFTO%26timestamp%3D1050755789861%7CS%3DQUOTE%26C%3DSUBS%26P%3D%5ECOMPX%2B%5EINDU%2B%5EINX%26T%3D0%2B1%2B2%2B3%2B8%2B10%2B11%2B12%2B13%2B15%2B16%2B19%2B20%2B25%7CS%3DAMTD_UPDATE%26C%3DSUBS%26P%3D778672055%26T%3D0%2B1%2B2%7CS%3DACTIVES_NYSE%26C%3DSUBS%26P%3DNYSE-ALL%26T%3D0%2B1 HTTP/1.1
Connection: Keep-Alive
User-agent: Datek Streamer v 4.4.3-1698956010209215187
cookie:
Host: investment.datek.com

HTTP/1.0 200 OK
Date: Mon 02 Mar 1998 10:20:00 GMT
Server: Movers
Content-type: text/html
Pragma: no-cache
Cache-Control: no-cache

N
#endif

#define         SYMCNT          200
#define         SYMBUFLEN       (SYMCNT * (SYMLEN+1) + 1)

typedef struct streamerpriv
{
	RCFILE	*rcp;
	int	state;

	FILE	*fp;

	char	symbuf[SYMBUFLEN];
	int	symcnt;

	int	variant;
		#define DATEK 0
		#define AMERITRADE 1
		#define FREETRADE 2
	char	*zsep;
	char	*broker;

	time_t	authtime;
	char	authbuf[512];
	char	*token, *timestamp, *cddomain, *acl;

	TOP10	top10[10][10];
	char	top10mkt;
} STREAMERDATA;

typedef struct
{
	char	*canon, *sym;
} SYMMAP;

static SYMMAP SymMap[] =
{
	{	"$DJI",		"^INDU",	},
	{	"$DJT",		"^TRAN",	},
	{	"$DJU",		"^UTIL",	},
	{	"$NYA",		"^NYA.X",	},
	{	"$COMP",	"^COMPX",	},
	{	"$NDX",		"^NDX.X",	},
	{	"$SPX",		"^INX",		},
	{	"$OEX",		"^OEX",		},
	{	"$MID",		"^MID.X",	},
	{	"$SML",		"^SML.X",	},
	{	"$RLX",		"^RLX.X",	},
	{	"$XAX",		"^XAX.X",	},
	{	"$IIX",		"^IIX.X",	},
	{	"$BTK",		"^BTK.X",	},
	{	"$XBD",		"^XBD.X",	},
	{	"$DRG",		"^DRG.X",	},
	{	"$XTC",		"XTC",		},	// TODO
	{	"$GSO",		"^GSO.X",	},
	{	"$HWI",		"^HWI.X",		},
	{	"$RUI",		"RUI",		},	// TODO
	{	"$RUT",		"^RUT.X",	},
	{	"$RUA",		"RUA",		},	// TODO
	{	"$SOX",		"^SOX.X",	},
	{	"$OSX",		"^OSX.X",	},
	{	"$XAU",		"^XAU.X",	},
	{	"$GOX",		"^GOX.X",	},
	{	NULL,		NULL		}
};

//
// Convert canonical index names to/from streamer index names
//
static void
streamer_canon2sym(char *out, char *in)
{
	char	*ip, *op;
	char	*p;
	int	len;
	SYMMAP	*map;

	ip = in;
	op = out;
	for (;;)
	{
		p = strchr(ip, '|');
		if (!p) p = strchr(ip, ',');
		if (!p) p = strchr(ip, ' ');
		if (!p) p = strchr(ip, 0);

		len = p - ip;
		memcpy(op, ip, len); op[len] = 0;

		for (map = SymMap; map->canon; ++map)
			if (strcmp(op, map->canon) == 0)
			{
				strcpy(op, map->sym);
				break;
			}

		if (*p == 0)
			break;

		ip += len + 1;
		op = strchr(op, 0);
		*op++ = *p;
		*op = 0;
	}
}

static void
streamer_sym2canon(char *out, char *in)
{
	SYMMAP	*map;

	for (map = SymMap; map->sym; ++map)
		if (strcmp(in, map->sym) == 0)
		{
			strcpy(out, map->canon);
			return;
		}

	if (in != out)
		strcpy(out, in);
}

static void
streamer_init(STREAMER sr, int variant)
{
	sr->refresh = 0;
	sr->fd[0] = -1;
	sr->nfd = 1;
	sr->priv->rcp = NULL;
	sr->priv->state = 0;
	sr->priv->fp = NULL;
	sr->priv->symcnt = 0;
	sr->priv->variant = variant;
	sr->priv->authtime = 0;

	switch (variant)
	{
	case DATEK:
	    strcpy(sr->id, "datek");
	    sr->priv->zsep = "+";
	    sr->priv->broker = "datek";
	    break;
	case AMERITRADE:
	    strcpy(sr->id, "ameritrade");
	    sr->priv->zsep = "+";
	    sr->priv->broker = "ameritrade";
	    break;
	case FREETRADE:
            strcpy(sr->id, "freetrade");
            sr->priv->zsep = "+";
            sr->priv->broker = "freetrade";
            break;
	}
}

static int
streamer_getc(STREAMER sr)
{
	int	c;

	c = fgetc(sr->priv->fp);
	if (c == EOF)
		return EOF;
	++sr->cnt_rx;

	if (Debug >= 5)
	{
		char	ch = c;
		timestamp(stderr);
		hexdump(stderr, "<", "            ", &ch, 1);
	}
	if (StreamerLog)
		fprintf(StreamerLog, "< %c\n", c);
	if (sr->writefile)
		fputc(c, sr->writefile);

	return (c);
}

static int
streamer_fgets(STREAMER sr, void *vbuf, int buflen)
{
	char	*buf = (char *) vbuf;
	int	len;

	if (fgets(buf, buflen, sr->priv->fp) == NULL)
		return 0;

	len = strlen(buf);

	if (Debug >= 5)
	{
		timestamp(stderr);
		hexdump(stderr, "<", "            ", buf, len);
	}

	if (StreamerLog)
	{
		int	llen = len - 1;

		if (buf[llen-1] == '\r')
			--llen;
		fprintf(StreamerLog, "< %.*s\n", llen, (char *)buf);
	}

	if (sr->writefile)
		fputs(buf, sr->writefile);

	return (len);
}

static int
streamer_fread(STREAMER sr, void *vbuf, int count)
{
	char	*buf = (char *) vbuf;
	int	len;

	len = fread(buf, 1, count, sr->priv->fp);
	if (len <= 0)
		return len;
	sr->cnt_rx += len;

	if (Debug >= 5)
	{
		timestamp(stderr);
		hexdump(stderr, "<", "            ", buf, len);
	}

	if (StreamerLog)
		hexdump(StreamerLog, "<", " ", buf, len);

	if (sr->writefile)
		fwrite(buf, 1, len, sr->writefile);

	return (len);
}

unsigned int
streamer_getN(STREAMER sr, int n)
{
	unsigned char	buf[16];
	int		len;
	int		val = 0;
	unsigned char	*p;

	len = streamer_fread(sr, buf, n);
	if (len != n)
		return 0;

	for (p = buf; n--; ++p)
		val = (val<<8) | *p;
	return val;
}

static int
hexify(char *dp, char *sp)
{
	char		*odp = dp;
	unsigned char	c;

	while ((c = *sp++))
	{
	    if (isalnum(c) || c == ' ' || c == '/' || c == '_' || c == '^')
		    *dp++ = c;
	    else
	    {
		    sprintf(dp, "%%%02X", c);
		    dp += 3;
	    }
	}
	*dp = 0;
	return (dp - odp);
}

static int
realopen(STREAMER sr, char *zsyms, char *optsyms)
{
	struct hostent		*hep;
	struct sockaddr_in	sockaddr;
	int			rc;
	char			buf[3 * (4096+SYMBUFLEN)];
	char			buf2[4096+SYMBUFLEN];
	int			len = 0;
	int			conn_tries;
	RCFILE			*rcp = sr->priv->rcp;
	char			*username = get_rc_value(rcp, "username");
	char			*password = get_rc_value(rcp, "password");
	char			*encpasswd = get_rc_value(rcp, "encpasswd");
	char			*hostname = get_rc_value(rcp, "hostname");
	int			port = atoi(get_rc_value(rcp, "port"));
	int			authage = atoi(get_rc_value(rcp, "auth-age"));
	STREAMERDATA		*priv = sr->priv;
	char			*p;
	int			fd;
	char			*shortname;

	++sr->cnt_realopens;
	time(&sr->time_realopen);

	if ( (priv->variant == AMERITRADE || priv->variant == FREETRADE)
	    && sr->time_realopen > priv->authtime)
	{
	    FILE	*afp[2];

	    display_msg(A_BOLD, "Authorizing account, please wait...");
	    refresh();
            if (priv->variant == AMERITRADE)
	        rc = p2open("/bin/bash",
		        PROGNAMESTR ".auth ameritrade.com 2>/dev/null", afp);
	    else
                rc = p2open("/bin/bash",
                        PROGNAMESTR ".auth freetrade.com 2>/dev/null", afp);
            if (rc < 0)
		    return SR_AUTH;

	    fprintf(afp[1], "%s\n", username);
	    fprintf(afp[1], "%s\n", password);
	    fflush(afp[1]);

	    priv->authbuf[0] = 0;
	    fgets(priv->authbuf, sizeof(priv->authbuf), afp[0]);
	    debug(5, "Auth: <%s>\n", priv->authbuf);

	    rc = p2close(afp);
	    if (rc < 0)
		    return SR_AUTH;

	    if (priv->authbuf[0] == '\r' || priv->authbuf[0] == '\n')
	    {
		mvcenter80(2, "Authorization failed");
		return (SR_AUTH);
	    }

	    priv->token = priv->authbuf;
	    priv->timestamp = strchr(priv->token, ' ');
	    if (!priv->timestamp)
	    {
		display_msg(A_BOLD,
			"Missing timestamp in authorization response!\n");
		return (SR_AUTH);
	    }
	    *priv->timestamp++ = 0;

	    priv->cddomain = strchr(priv->timestamp, ' ');
	    if (!priv->cddomain)
	    {
		display_msg(A_BOLD,
		    "Missing cddomain in authorization response!\n");
		return (SR_AUTH);
	    }
	    *priv->cddomain++ = 0;

	    priv->acl = strchr(priv->cddomain, ' ');
	    if (!priv->acl)
	    {
		display_msg(A_BOLD,
		    "Missing acl in authorization response!\n");
		return (SR_AUTH);
	    }
	    *priv->acl++ = 0;

	    p = strchr(priv->acl, ' ');
	    if (p) *p = 0;
	    p = strchr(priv->acl, '\r');
	    if (p) *p = 0;
	    p = strchr(priv->acl, '\n');
	    if (p) *p = 0;
	    display_msg(A_NORMAL, "");
	}

	hep = mygethostbyname(hostname);
	if (!hep)
		return (-1);

	memcpy(&sockaddr.sin_addr, hep->h_addr, hep->h_length);
	sockaddr.sin_family = AF_INET;
	sockaddr.sin_port = htons(port);

	conn_tries = 0;
reconnect:
	if (++conn_tries >= 5)
		return -4;

	debug(5, "Open socket...\n");

	fd = socket(AF_INET, SOCK_STREAM, 0);
	if (fd < 0)
		return -2;

	debug(5, "Socket fd=%d...\n", fd);
	debug(5, "Connect to '%s'...\n", hostname);

	rc = connect_timeout(fd, (SA *) &sockaddr, sizeof(sockaddr), 15);
	if (rc < 0)
	{
		if (Debug >= 5)
			syserror(0, "Couldn't connect\n");
		goto reconnect;
		return -3;
	}

	debug(5, "Send GET Request...\n");

	switch (sr->priv->variant)
	{
	default:
	case DATEK:
	    len = sprintf(buf,
		"GET /!U=%s"
		"&W=%s"
		"|S=QUOTE&C=SUBS"
		"&P=%s"
		"&T=0+1+2+3+4+5+6+7+8+9+10+11+12+13+14+15+16+17+18+19+20+21",
		username,
		encpasswd,
		zsyms);
	    break;
	case AMERITRADE:
        case FREETRADE:
	    if (sr->priv->variant == AMERITRADE)
		shortname = "AMER";
	    else
		shortname = "FREE";
	    len = sprintf(buf2,
		"GET /!U=%s&W=%s&A=userid=%s"
		"&token=%s&company=%s&segment=%s&cddomain=%s"
		"&usergroup=ACCT&accesslevel=ACCT&authorized=Y&acl=%s"
		"&timestamp=%s"
		"|S=QUOTE&C=SUBS&P=%s"
		"&T=0+1+2+3+4+5+6+7+8+9+10+11+12+13+14+15+16+17+18+19+20+21"
		// "&T=0+1+2+3+8+10+11+12+13+15+16+19+20+25"
		"|S=AMTD_UPDATE&C=SUBS&P=%s&T=0+1+2"
		"%s%s%s" // actives
		, username		// account number
		, priv->token		// hashed password
		, username
		, priv->token
		, shortname
		, shortname
		, priv->cddomain	// no idea.  anything works here?
		, priv->acl		// no idea. must be correct, tho.
		, priv->timestamp
		, zsyms
		, username
		, priv->top10mkt == 'N'
		    ? "|S=ACTIVES_NYSE&C=SUBS&P=NYSE-ALL&T=0+1" : ""
		, priv->top10mkt == 'Q'
		    ? "|S=ACTIVES_NASDAQ&C=SUBS&P=NASDAQ-ALL&T=0+1" : ""
		, priv->top10mkt == 'E'
		    ? "|S=ACTIVES_AMEX&C=SUBS&P=AMEX-ALL&T=0+1" : ""
		);
	    len = hexify(buf, buf2);
            break;
	}

	if (optsyms[0])
		len += sprintf(buf+len,
			"|S=OPTION&C=SUBS"
			"&P=%s"
			"&T=0+1+2+3+4+5+6+7+8+9+10+11+12+13+14+15+16",
			optsyms);

	len += sprintf(buf+len,
		" HTTP/1.1\r\n"
		"Connection: Keep-Alive\r\n"
		"User-Agent: Datek Streamer v 4.4.3-892786132055032531\r\n"
		"Host: %s\r\n"
		"Accept: image/gif, image/jpeg, image/pjpeg, */*\r\n"
		"Accept-Encoding: gzip\r\n"
		"Accept-Language: en\r\n"
		"Accept-Charset: iso-8859-1,*,utf-8\r\n"
		"\r\n", hostname);

	debug(5, "%s", buf);

	rc = streamer_write_binary(fd, buf, len);
	if (rc != len)
		syserror(1, "Error sending request to server\n");

	// Need HTTP response
	sr->priv->state = 0;

	sr->priv->fp = fdopen(fd, "rb+");
	if (!sr->priv->fp)
		error(1, "Couldn't fdopen\n");

	sr->fd[0] = fd;
	priv->authtime = sr->time_realopen + authage*60*60;
	return (0);
}

static int
streamer_open( STREAMER sr, RCFILE *rcp, FILE *readfile)
{
	streamer_init(sr, sr->priv->variant);
	++sr->cnt_opens;
	time(&sr->time_open);

	sr->priv->rcp = rcp;
	return 0;
}
static int
streamer_select(
		STREAMER sr,
		int n, fd_set *readfds, fd_set *writefds,
		fd_set *exceptfds, struct timeval *timeout
		)
{

	if (readfds && sr->fd[0] > 0 &&
		FD_ISSET(sr->fd[0], readfds) && FRcnt(sr->priv->fp))
	{
			FD_ZERO(readfds);
			FD_SET(sr->fd[0], readfds);
			if (writefds)
				FD_ZERO(writefds);
			if (exceptfds)
				FD_ZERO(exceptfds);
			return 1;
	}
	return select(n, readfds, writefds, exceptfds, timeout);
}

static void
streamer_close(STREAMER sr)
{
	if (sr->priv->fp)
	{
		fclose(sr->priv->fp);
		sr->fd[0] = -1;
		sr->priv->fp = NULL;
	}
}

static void
streamer_record(STREAMER sr, FILE *fp)
{
	sr->writefile = fp;
}

static void
streamer_timetick(STREAMER sr, time_t now)
{
	if (sr->writefile)
		fflush(sr->writefile);
}

static void
streamer_send_quickquote(STREAMER sr, char *sym) {}

static void
streamer_send_livequote(STREAMER sr, char *sym)
{
	if (sr->fd[0] < 0 || sr->readfile)
		return;

	if (sr->priv->symcnt >= SYMCNT)
		return;

	if (sr->priv->symcnt == 0)
		strcpy(sr->priv->symbuf, "");

	if (sr->priv->symcnt)
		strcat(sr->priv->symbuf, sr->priv->zsep);
	strcat(sr->priv->symbuf, sym);
	++sr->priv->symcnt;
}

static void
walk_syms(char *buf, char *optbuf, char *zsep)
{
	char	*p, *optp;
	int	i;

	// Walk list of all stocks and create symbol lists
	// compatible with datek streamer
	p = buf; *p = 0;
	optp = optbuf; *optp = 0;
	for (i = 0; i < NumStock; ++i)
	{
		char sym[SYMLEN+1];

		streamer_canon2sym(sym, Stock[i].sym);
		if (strlast(sym) == '.')
		{
			int	len = strlen(sym)-1;

			optp += sprintf(optp, "%s%.*s.%.2s",
				optp == optbuf ? "" : zsep,
				len-2, sym,
				sym+len-2);
		}
		else
			p += sprintf(p, "%s%s", p == buf ? "" : zsep, sym);
	}
}

static void
streamer_send_livequote_end(STREAMER sr)
{
	char	buf[SYMBUFLEN+BUFSIZ];
	char	optbuf[SYMBUFLEN+BUFSIZ];

	if (!sr->priv->symcnt)
		return;

	sr->priv->symcnt = 0;

	// Until we thread this, the UI delay is too painful.
	return;

	// Walk list of all stocks and start new streamer
	walk_syms(buf, optbuf, sr->priv->zsep);
	if (buf[0])
		strcat(buf, sr->priv->zsep);
	strcat(buf, sr->priv->symbuf);

	realopen(sr, buf, optbuf);
}

static void
streamer_send_symbols(STREAMER sr, char *symbols, int add) {}

static void
streamer_send_symbols_end(STREAMER sr, int add, int all)
{
	char	buf[SYMBUFLEN+BUFSIZ];
	char	optbuf[SYMBUFLEN+BUFSIZ];

	// Its very expensive to change symbols, so ignore deletes
	if (!add)
		return;

	// Close old streamer
	if (sr->priv->fp > 0)
	{
		fclose(sr->priv->fp);
		sr->fd[0] = -1;
		sr->priv->fp = NULL;
	}

	// Walk list of all stocks and start new streamer
	walk_syms(buf, optbuf, sr->priv->zsep);
	realopen(sr, buf, optbuf);
}

#if 0
static void
streamer_send_disconnect(STREAMER sr) {}
#endif

static void
streamer_send_top10(STREAMER sr, char market, int type, int add)
{
    sr->priv->top10mkt = 0;
    if (type != TOP_VOL || !add)
	return;
    if (sr->priv->variant != AMERITRADE)
	return;

    sr->priv->top10mkt = market;
    streamer_send_symbols_end(sr, TRUE, TRUE);
}

#if 0
static void
streamer_send_movers(STREAMER sr, int on) {}

static void
streamer_send_info(STREAMER sr, char *sym, int type) {}

static void
streamer_send_optchain(STREAMER sr, char *sym) {}

static void
streamer_send_chart(STREAMER sr, char *sym, int freq, int periods) {}
#endif

static void
do_top10(STREAMER sr, char *sym, char *buf)
{
    char	*p;
    char	*semi_fld[8];
    char	*fld[3+10*3];
    int		nfld;
    int		i;
    int		rank;
    char	market;

    p = strchr(buf, 0xff); if (p) *p = 0;

    debug(1, "top10: %s <%s>\n", sym, buf);

    nfld = strsplit(semi_fld, asizeof(semi_fld), buf, ';');
    if (nfld < 7)
	return;
    debug(1, "top10: %d %s\n", nfld, semi_fld[6]);

    nfld = strsplit(fld, asizeof(fld), semi_fld[6], ':');
    debug(1, "top10: %d\n", nfld);
    if (nfld < 33)
	return;
    if (fld[0][0] != '1')
	return;

    if (strcmp(sym, "NYSE-ALL") == 0)
	market = 'N';
    else if (strcmp(sym, "NASDAQ-ALL") == 0)
	market = 'Q';
    else if (strcmp(sym, "AMEX-ALL") == 0)
	market = 'E';
    else
	return;

    rank = 0;
    for (i = 3; i < nfld; i += 3)
    {
	TOP10	*t;

	debug(1, "top10: %d %s\n", i, fld[i]);

	t = &sr->priv->top10[5][rank++];
	t->rank = rank;
	t->type = TOP_VOL;
	t->market = market;

	if (fld[i+0][1])
	{
		strncpy(t->sym, fld[i+0], SYMLEN);
		t->sym[SYMLEN] = 0;
	}
	t->prev = 0;
	t->volume = atoi(fld[i+1]);
	t->curr = 0;
	t->change = 0;
	display_top10(t);
	if (rank == 10)
	    break;
    }
}

static unsigned int
nextN(unsigned char **bp, int n)
{
	unsigned int val = 0;

	while (n--)
	{
		val <<= 8;
		val |= *(*bp)++;
	}
	return val;
}

static float
nextF(unsigned char **bp)
{
	float		val;
	unsigned char	*d = (unsigned char *) &val;

	d[3] = *(*bp)++;
	d[2] = *(*bp)++;
	d[1] = *(*bp)++;
	d[0] = *(*bp)++;
	return (val);
}

static int
do_quote(STREAMER sr)
{
	int		seglen;
	int		symlen;
	char		exchange;
	int		last_size;
	int		is_trade = 0;
	QUOTE		q;
	QUICKQUOTE	qq;
	LIVEQUOTE	lq;
	STOCK		*sp;
	unsigned char	buf[2048];
	unsigned char	*p;
	unsigned char	*eob;
	int		rc;
	int		isopt;

	memset(&q, 0, sizeof(q));
	memset(&qq, 0, sizeof(qq));

	seglen = streamer_getN(sr, 2);
	if (seglen < 0 || seglen > sizeof(buf))
		error(1, "Bad segment length '%d'\n", seglen);

	rc = streamer_fread(sr, buf, seglen + 2);
	if (rc != (seglen+2))
		return -1;

	p = buf;
	eob = buf + seglen-1; // We need at least 2 bytes, 1: for a valid id, 2: field val
	if (p[1] == 0x12)
		isopt = 1;
	else
		isopt = 0;
	p += 2; // Skip 00 01 and/or 00 12
	p += 2; // Skip 00 00

	symlen = *p++;
	if (isopt)
	{
		char	*d;
		int	i;

		d = q.sym;
		for (i = 0; i < symlen; ++i, ++p)
		{
			if (*p != '.')
				*d++ = *p;
		}
		*d++ = '.';
		*d = 0;
	}
	else
	{
		memcpy(q.sym, p, symlen);
		q.sym[symlen] = 0;
		p += symlen;
	}
	streamer_sym2canon(q.sym, q.sym);
	strcpy(qq.sym, q.sym);
	strcpy(lq.sym, q.sym);

	if (strcmp(q.sym, "NYSE-ALL") == 0)
	{
	    do_top10(sr, q.sym, p+3);
	    goto out;
	}
	if (strcmp(q.sym, "NASDAQ-ALL") == 0)
	{
	    do_top10(sr, q.sym, p+3);
	    goto out;
	}
	if (strcmp(q.sym, "AMEX-ALL") == 0)
	{
	    do_top10(sr, q.sym, p+3);
	    goto out;
	}

	sp = find_stock(q.sym);
	debug(1, "<%s> sp=%p\n", q.sym, sp);

	if (sp && sp->nquotes)
		copy_quote(&sp->cur, &q, &qq, &lq);

	if (isopt)
	{
	    int dlen;
	    double volume, openint;

	    while (p < eob)
	    {
		if (Debug)
		{
			unsigned char	*np = p+1;
			unsigned char	*nnp = p+1;
			fprintf(stderr, "F%d = %f %d\n",
					*p, nextF(&np), nextN(&nnp, 4));
		}
		switch (*p++)
		{
		case 1:
			dlen = nextN(&p, 2);
			if (Debug)
				fprintf(stderr, "Desc=%.*s\n", dlen, p);
			p += dlen;
			break;
		case 2: qq.bid = q.bid = nextF(&p); break;
		case 3:	qq.ask = q.ask = nextF(&p); break;
		case 4:	lq.last = qq.last = q.last = nextF(&p);
			break;
		case 5: qq.high = q.high = nextF(&p); break;
		case 6: qq.low = q.low = nextF(&p); break;
		case 7: lq.close = qq.prev_close = q.close = nextF(&p);
			break;
		case 8:
			/* volume, but store it in last_size */
			volume = nextN(&p, 8);
			// qq.volume = q.volume = volume * 100;
			qq.last_size = q.last_size = volume * 100;
			// if (sp && sp->cur.volume)
			if (sp && sp->cur.last_size)
				is_trade = 1;
			break;
		case 9:
			/* open interest, but store it in volume */
			openint = nextN(&p, 4);
			qq.volume = q.volume = openint * 100;
			break;
		case 10: nextN(&p, 4); break;	/* never see */
		case 11:	//trade time
			qq.timetrade = q.time = nextN(&p, 4);
			break;
		case 12:	//quote time
			p += 4;
			break;
		case 13: /* inout$ = */ nextF(&p); break;
		case 14:
			/* ??? */
			nextN(&p, 4); break;
		case 15:
			/* ??? */
			nextN(&p, 4); break;
		case 16:
			/* year? */
			nextN(&p, 4); break;
		case 255: p = eob; break; /* This should never happen */
		default: p = eob;
			 if (Debug)
				 fprintf(stderr, "OptField %d?\n", p[-1]);
			 //return -1;
		}
	    }
	}
	else
	{
	    while (p<eob)
	    {
		switch (*p++)
		{
		case 1: qq.bid = q.bid = nextF(&p); break;
		case 2:	qq.ask = q.ask = nextF(&p); break;
		case 3:	lq.last = qq.last = q.last = nextF(&p);
			break;
		case 4: qq.bid_size = q.bid_size = nextN(&p, 4);
			break;
		case 5: qq.ask_size = q.ask_size = nextN(&p, 4);
			break;
		case 6: ++p; qq.bid_id = q.bid_id = *p++; break;
		case 7: ++p; qq.ask_id = q.ask_id = *p++; break;
		case 8: qq.volume = q.volume = nextN(&p, 8);
			if (sp && sp->cur.volume)
				is_trade = 1;
			break;
		case 9: last_size = nextN(&p, 4) * 100;
			qq.last_size = q.last_size = last_size;
			break;
		case 10:	//trade time
			qq.timetrade = q.time = nextN(&p, 4);
			break;
		case 11:	//quote time
			p += 4;
			break;
		case 12: qq.high = q.high = nextF(&p); break;
		case 13: qq.low = q.low = nextF(&p); break;
		case 14: ++p; q.bid_tick = *p++; break;
		case 15: lq.close = qq.prev_close = q.close = nextF(&p);
			 break;
		case 16: ++p; exchange = *p++; break;
		case 17:
			 // Shortable/Marginable flags?
			++p; 	// Unk
			 break;
		case 18:
			 // Shortable/Marginable flags?
			++p; 	// Unk
			break;
		case 19: nextF(&p); break; // island bid
		case 20: nextF(&p); break; // island ask
		case 21: nextN(&p, 8); break; // island vol
		case 255: p = eob; break;
		default: p = eob;
			 if (Debug)
				 fprintf(stderr, "Field %d?\n", p[-1]);
			 //return -1;
		}
	    }
	}

	qq.last_eps = 12345678;
	qq.cur_eps = 12345678;
	qq.sharesout = 12345678;

	if (sp)
	{
		display_quote(&q, is_trade);
		optchain_quote(&q);
		info_quickquote(&qq);
	}

	display_livequote(&lq);

out:
	return (seglen + 5);
}

static int
do_N(STREAMER sr)
{
	unsigned char	buf[2048];
	int		len, desclen;
	int		code;
	short		svcid, rc;
	char		*desc;

	// Read 00 03 31 30 30, a UTF string
	len = streamer_getN(sr, 2);
	if (len > sizeof(buf)-1)
	    error(1, "Insane UTF length %d\n", len);
	streamer_fread(sr, buf, len);
	buf[len] = 0;
	code = atoi(buf);

	// Read remainder of message
	len = streamer_getN(sr, 4);
	streamer_fread(sr, buf, len);

	// Read FF 0A terminator
	streamer_getN(sr, 2);

	// Do we understand this message?
	if (code != 100)
	    return 0;

	// Parse message in buffer
	if (buf[0] == 0x00 && buf[1] == 0x0A)
	{
	    if (buf[2] != 0)
		return 0;
	    svcid = (buf[3]<<8) | buf[4];

	    if (buf[5] != 1)
		return 0;
	    rc = (buf[6]<<8) | buf[7];

	    if (buf[8] != 2)
		return 0;
	    desclen = (buf[9]<<8) | buf[10];
	    desc = buf + 11;
	    switch (rc)
	    {
	    case 0x12:
		    // Server ID number
		    strcpy(sr->id, sr->priv->broker);
		    strcat(sr->id, "/");
		    if (desclen > 3) desclen = 3;
		    desc[desclen] = 0;
		    strcat(sr->id, desc);
		    return 0;
	    case 0x04:
		    // Login failure???
		    if (desclen < 0) desclen = 0;
		    else if (desclen > COLS) desclen = COLS;
		    if (desclen)
			display_prompt(A_BOLD, "%s: %.*s",
			    sr->priv->broker, desclen, desc);
		    if (svcid < 0)
			return SR_AUTH;
		    return 0;
	    }
	}
	return 0;
}

static int
streamer_process(STREAMER sr, int fdindex)
{
	int		len;
	int		rc;
	int		c;

	if (sr->priv->state == 0)
	{
		unsigned char	buf[2048];

		len = streamer_fgets(sr, buf, sizeof(buf));
		if (len <= 0)
		{
		eof:
			if (sr->readfile)
			{
				if (sr->fd[0] >= 0)
					fclose(sr->readfile);
				sr->fd[0] = -1;
				return (0);
			}
			return -1;
		}

		if (buf[0] == '\n' || (buf[0] == '\r' && buf[1] == '\n'))
			++sr->priv->state;
		return 0;
	}

	c = streamer_getc(sr);
	if (c == EOF)
		goto eof;

	switch (c)
	{
	case 'S':
		rc = do_quote(sr);
		break;
	case 'H':
		break;
	case 'N':
		rc = do_N(sr);
		if (rc < 0) return rc;
		break;
	default:
		if (Debug)
			error(0, "Unknown record %d\n", c);
		break;
	}

	return 0;
}

static STREAMER
common_new(int variant)
{
	STREAMER	sr;

	sr = (STREAMER) malloc(sizeof(*sr));
	if (!sr)
		return NULL;
	memset(sr, 0, sizeof(*sr));

	sr->open = streamer_open;
	sr->select = streamer_select;
	sr->close = streamer_close;
	sr->record = streamer_record;
	sr->timetick = streamer_timetick;

	sr->send_quickquote = streamer_send_quickquote;
	sr->send_livequote = streamer_send_livequote;
	sr->send_livequote_end = streamer_send_livequote_end;
	sr->send_symbols = streamer_send_symbols;
	sr->send_symbols_end = streamer_send_symbols_end;

	// sr->send_disconnect = streamer_send_disconnect;
	sr->send_top10 = streamer_send_top10;
	// sr->send_movers = streamer_send_movers;
	// sr->send_info = streamer_send_info;
	// sr->send_optchain = streamer_send_optchain;
	// sr->send_chart = streamer_send_chart;

	sr->process = streamer_process;

	sr->priv = (STREAMERDATA *) malloc(sizeof(*sr->priv));
	if (!sr->priv)
	{
		free(sr);
		return NULL;
	}

	time(&sr->time_start);

	streamer_init(sr, variant);

	return (sr);
}

STREAMER
datek_new(void)
{
	STREAMER	sr;

	sr = common_new(DATEK);
	if (!sr)
		return NULL;

	return (sr);
}
STREAMER
ameritrade_new(void)
{
	STREAMER	sr;

	sr = common_new(AMERITRADE);
	if (!sr)
		return NULL;

	return (sr);
}
STREAMER
freetrade_new(void)
{
	STREAMER	sr;

	sr = common_new(FREETRADE);
	if (!sr)
		return NULL;

	return (sr);
}

