/*
    ldapdiff
    Copyright (C) 2000-2002 Thomas.Reith@rhoen.de

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <lber.h>
#include <ldap.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "ldapdiff.h"

static struct s_conf asdefconf[] = {
 {NULL, SECTIONGLOBAL,  CONFLDAPHOST,        "ldaphost",        NULL },
 {NULL, SECTIONGLOBAL,  CONFLDAPPORT,        "ldapport",        NULL },
 {NULL, SECTIONGLOBAL,  CONFROOTDN,          "rootdn",          NULL },
 {NULL, SECTIONGLOBAL,  CONFROOTPW,          "rootpw",          NULL },
 {NULL, SECTIONGLOBAL,  CONFMODFILE,         "modfile",         NULL },
 {NULL, SECTIONGLOBAL,  CONFADDFILE,         "addfile",         NULL },
 {NULL, SECTIONGLOBAL,  CONFONLINEUPDATE,    "onlineupdate",    NULL },
 {NULL, SECTIONGLOBAL,  CONFOFFLINEUPDATE,   "offlineupdate",   NULL },
 {NULL, SECTIONGLOBAL,  CONFONLINEERRFATAL,  "onlineerrfatal",  NULL },
 {NULL, SECTIONGLOBAL,  CONFPLUGIN,          "plugin",          NULL },
 {NULL, SECTIONGLOBAL,  CONFPLUGINFILE,      "pluginfile",      NULL },
 {NULL, SECTIONGLOBAL,  CONFPLUGINFUNCTION,  "pluginfunction",  NULL },
 {NULL, SECTIONGLOBAL,  CONFICONV,           "iconv",           NULL },
 {NULL, SECTIONGLOBAL,  CONFLDIFCHARSET,     "ldifcharset",     NULL },
 {NULL, SECTIONGLOBAL,  CONFLDAPCHARSET,     "ldapcharset",     NULL },
 {NULL, SECTIONGLOBAL,  CONFSCHEMACHECK,     "schemacheck",     NULL },
 {NULL, SECTIONGLOBAL,  CONFSCHEMABASE,      "schemabase",      NULL },
 {NULL, SECTIONGLOBAL,  CONFSCHEMAFILTER,    "schemafilter",    NULL },
 {NULL, SECTIONGLOBAL,  CONFSCHEMAATTRIBUTE, "schemaattribute", NULL },
 {NULL, SECTIONGLOBAL,  CONFSCHEMAHACK,      "schemahack",      NULL },
 {NULL, SECTIONPROFILE, CONFBASEDN,          "basedn",          NULL },
 {NULL, SECTIONPROFILE, CONFFILTER,          "filter",          NULL },
 {NULL, SECTIONPROFILE, CONFGROUP,           "group",           NULL },
 {NULL, SECTIONPROFILE, CONFIGNORE,          "ignore",          NULL },
 {NULL, SECTIONPROFILE, CONFMULTI,           "multi",           NULL },
 {NULL, SECTIONPROFILE, CONFNOEQUALITY,      "noequality",      NULL },
 {NULL, SECTIONPROFILE, CONFMAPALIAS,        "mapalias",        NULL },
 {NULL, SECTIONPROFILE, CONFDELETEENTRY,     "deleteentry",     NULL },
 {NULL, SECTIONPROFILE, CONFDELETEATTRIBUTE, "deleteattribute", NULL },
 {NULL, SECTIONNONE,    CONFNONE,            NULL,              NULL }
};

static struct s_conf *asconf[MAXCONFVAR];

static char *getconf(char *section,tevarid varid)
{
 int   count = 0;
 char *var   = "unknown";

 while(asconf[count] != NULL){
  if(strcmp(section,asconf[count]->section) == 0 && 
     asconf[count]->varid == varid){
   return asconf[count]->val;
  }
  count++;
 }

 count=0;
 while(asdefconf[count].varid != CONFNONE){
  if(asdefconf[count].varid == varid){
   var = asdefconf[count].var;
  }
  count++;
 }

 fprintf(stderr,"error: requested variable [%s]:%s not found\n",section,var);
 exit(-1);

 return NULL;
}

char *ldifgetgconf(tevarid varid)
{
 return getconf(SECTGLOB,varid);
}

char *ldifgetpconf(tevarid varid)
{
 extern char *profile;
 return getconf(profile,varid);
}

static int confcheck(char *section)
{
 static struct s_conf *ascmpconf[MAXCONFVAR];
 int           count;
 int           cmpcount;
 int           found;
 int           errflag = 0;
 tesectiontype sectiontype;

 /* copy all variables of this section to a temporary array */
 count    = 0;
 cmpcount = 0;
 while(asconf[count] != NULL){
  if(strcmp(asconf[count]->section,section) == 0){
   ascmpconf[cmpcount] = asconf[count];
   cmpcount++;
  }
  count++;
 }
 ascmpconf[cmpcount] = NULL;

 sectiontype = SECTIONPROFILE;
 if(strcmp(section,SECTGLOB) == 0){
  sectiontype = SECTIONGLOBAL;
 }

 count    = 0;
 while(asdefconf[count].varid != CONFNONE){
  if(asdefconf[count].sectiontype == sectiontype){
   cmpcount = 0;
   found    = 0;
   while(ascmpconf[cmpcount] != NULL){
    if(strcmp(asdefconf[count].var,ascmpconf[cmpcount]->var) == 0){
     found++;
    }
    cmpcount++;
   }
   if(found == 0){
    fprintf(stderr,"error: variable [%s]:%s missing\n",section,asdefconf[count].var);
    errflag++;
   }
  }
  count++;
 }

 return errflag;
}

void ldifinitconf(char* cfile)
{
 FILE *f;
 char  line[MAXATTRLEN];
 char  section[MAXATTRLEN];
 char  var[MAXATTRLEN];
 char  val[MAXATTRLEN];
 int   count;
 int   confcount;
 int   clines  = 0;
 int   errflag = 0;

 /* intialize asconf array */
 for(confcount = 0;confcount < MAXCONFVAR;confcount++){
  asconf[confcount] = NULL;
 }

 if((f = fopen(cfile,"r")) == NULL){
  fprintf(stderr,"fopen() of %s failed: file: %s, line: %d\n",cfile,__FILE__,__LINE__);
  exit(-1);
 }

 confcount  = 0;
 section[0] = 0;
 while(fgets(line,sizeof(line),f) != NULL){
  int ccount;

  clines++;

  /* strip '\n' '\r' and '#' */
  ccount = 0;
  while(line[ccount] != '\0'){
   if(line[ccount] == '\n' || line[ccount] == '\r' || line[ccount] == '#'){
    line[ccount] = '\0';
    break;
   }
   ccount++;
  }
  if(strlen(line) == 0){
   continue;
  }

  if(strchr(line,'[') != NULL && strchr(line,']') != NULL){
   int seclength;
   seclength = strchr(line,']') - strchr(line,'[') - 1;
   if(seclength > 0){

    if(strlen(section) != 0){
     errflag += confcheck(section);
    }

    memset(section,0,MAXATTRLEN * sizeof(char));
    strncpy(section,strchr(line,'[') + 1,seclength);
    ldiflog(LOG2,"config: processing section %s",section);
    continue;
   }
  }

  if(ldifconfnormalize(line,var,val) == -1){
   ldiflog(LOG0,"config: warning, parse error in line %d: [%s]",clines,line);
  }

  if(strlen(section) != 0 && strlen(var) != 0 && strlen(val) != 0){
   count = 0;
   while(asdefconf[count].varid != CONFNONE){
    if(strcmp(asdefconf[count].var,var) == 0){
     struct s_conf *psconf;

     psconf = LDALLOC(1,sizeof(struct s_conf));
     psconf->section     = LDDUP(section);
     psconf->sectiontype = asdefconf[count].sectiontype;
     psconf->varid       = asdefconf[count].varid;
     psconf->var         = LDDUP(var);
     psconf->val         = LDDUP(val);
     asconf[confcount] = psconf;
     ldiflog(LOG2,"config: processing variable %s:%s=%s",section,var,val);
     confcount++;
    }
    count++;
   }
  }
 }
 /* the last section should be checked, too */
 if(strlen(section) != 0){
  errflag += confcheck(section);
 }
 
 fclose(f);

 if(errflag){
  fprintf(stderr,"%d error(s) in %s\n",errflag,cfile);
  exit(-1);
 }
}
