/*
 * widgets.h -- Resusable widgets
 * Copyright (C) 2002 Charles Yates <charles.yates@pandora.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef _KINOPLUS_WIDGETS_H
#define _KINOPLUS_WIDGETS_H

#include <stdint.h>
#include <gdk/gdk.h>
#include <gtk/gtk.h>

extern "C" {
#include "support.h"
}

extern GtkWidget *my_lookup_widget( GtkWidget *widget, const gchar *widget_name );

/**	Interface for all Widgets.
*/

class GtkKinoWidget
{
	public:
		virtual GtkWidget *GetGtkWidget( ) = 0;
};

/** Abstract class for containers.
*/

class GtkKinoContainer : public GtkKinoWidget
{
	public:
		// Attach a widget in the named (empty) frame
		void AttachGtkWidget( char *frame, GtkKinoWidget *widget )
		{
			GtkWidget *container = GetGtkWidget( );
			GtkWidget *bin = my_lookup_widget( container, frame );
			gtk_widget_reparent( ( GTK_BIN( widget->GetGtkWidget() ) )->child, bin );
		}

		// Detach the widget in the named frame
		void DetachGtkWidget( char *frame, GtkKinoWidget *widget )
		{
			GtkWidget *container = GetGtkWidget( );
			GtkWidget *bin = my_lookup_widget( container, frame );
			gtk_widget_reparent( ( GTK_BIN( bin ) )->child, GTK_WIDGET( widget->GetGtkWidget() ) );
		}
};

/** Interface for receiving callbacks from the KeyFrame Controller.
*/

class KeyFrameControllerClient
{
	public:
		// Called when the key button is changed to define/undefine a key frame
		virtual void OnControllerKeyChanged( double position, bool editable ) { } 
		// Called when the previous button is pressed
		virtual void OnControllerPrevKey( double position ) { }
		// Called when the next button is pressed
		virtual void OnControllerNextKey( double position ) { }
		// Called when the position is changed
		virtual void OnControllerPositionChanged( double position ) = 0;
};

/** Enumerated type for classifying the current frame, being either a computed frame, a key or
	a locked/non-removable key.
*/

enum frame_type { FRAME = 0, KEY = 1, LOCKED_KEY = 2 };

/** Key frame controller is a GUI widget which provides a simple key frame definition/editing tool. 
 
	This is currently represented by 4 components - a slide bar to select the position, a toggle 
	button which allows you to specify whether a key exists at the current position, and two buttons
	to allow navigation between the key frames. 

	Intended use is for a container to implement the controller client.
*/


class KeyFrameController : public GtkKinoWidget
{
	public:
		// Set the current feedback 
		virtual void ShowCurrentStatus( double position, frame_type type, bool hasPrev, bool hasNext ) = 0;
		// Get the current position
		virtual double GetCurrentPosition( ) = 0;
};

/** Factory method for the key frame controller.
*/

extern KeyFrameController *CreateWidgetKeyFrameController( KeyFrameControllerClient *client );

/** Interface for the Preview Area widget.
*/

class PreviewArea : public GtkKinoWidget
{
	public:
		// Container calls this to refresh the preview
		virtual void ShowImage( int x, int y, uint8_t *image, int width, int height ) = 0;
};

class PreviewAreaClient
{
	public:
		virtual void OnPreviewAreaRefresh( PreviewArea * ) = 0;
};

/** Factory method for the PreviewArea.
*/

extern PreviewArea *CreateWidgetPreviewArea( PreviewAreaClient * );

/** Interface for pair picker clients.
*/

class PairPicker;

class PairPickerClient
{
	public:
		virtual void OnPairPickerChangeValue( PairPicker *picker, double first, double second ) = 0;
};

class PairPicker : public GtkKinoWidget
{
	public:
		virtual void SetValues( double first, double second ) = 0;
		virtual void SetFirstMinMax( double min, double max ) = 0;
		virtual void SetSecondMinMax( double min, double max ) = 0;
};

extern PairPicker *CreateWidgetPairPicker( PairPickerClient *client );

/** File selector
*/

class FileSelector;

class FileSelectorClient
{
	public:
		virtual void OnFileSelected( FileSelector *selector, char *file ) = 0;
};

class FileSelector 
{
	public:
		virtual void SetDefaultDirectory( char *dir ) = 0;
		virtual void SelectFile( ) = 0;
};

extern FileSelector *CreateWidgetFileSelector( FileSelectorClient *client );

/** Interface for luma picker clients.
*/

class LumaPicker;

class LumaPickerClient
{
	public:
		virtual void OnLumaPickerChange( LumaPicker *picker, char *file, double softness ) = 0;
};

class LumaPicker : public GtkKinoWidget
{
	public:
		virtual bool SetLuma( char *file, double softness ) = 0;
};

extern LumaPicker *CreateWidgetLumaPicker( LumaPickerClient *client );

/** Interface for interlace options
*/

class InterlaceOptions : public GtkKinoWidget
{
	public:
		virtual bool IsInterlaceOn( ) = 0;
		virtual bool IsFirstFieldOn( ) = 0;
};

extern InterlaceOptions *CreateWidgetInterlaceOptions( );

/** Interface for multi-map summary and editor.
*/

class MultiMapProperties
{
	public:
		virtual char *GetString( int index ) = 0;
		virtual int Count( ) = 0;
};

class MultiMap;

class MultiMapClient
{
	public:
		virtual void OnPropertySelected( MultiMap *map, int value ) = 0;
};

class MultiMap : public GtkKinoContainer
{
	public:
		virtual void Refresh( MultiMapProperties *set ) = 0;
};

extern MultiMap *CreateWidgetMultiMap( MultiMapClient *client );

#endif

