
/*
 * Copyright (c) Abraham vd Merwe <abz@blio.net>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *	  notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *	  notice, this list of conditions and the following disclaimer in the
 *	  documentation and/or other materials provided with the distribution.
 * 3. Neither the name of the author nor the names of other contributors
 *	  may be used to endorse or promote products derived from this software
 *	  without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO,
 * THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include <ctype.h>
#include <regex.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/procfs.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <dirent.h>

#include "typedefs.h"

/*
 * Search ``string'' for matches to the extended regular expression given in
 * ``pattern'', Returns TRUE if a match for ``pattern'' was found in ``string'',
 * FALSE otherwise.
 */
bool ereg_match (const char *pattern,const char *string)
{
   int errcode;
   regex_t preg;
   if ((errcode = regcomp (&preg,pattern,REG_EXTENDED | REG_NOSUB))) return (FALSE);
   if (regexec (&preg,string,0,NULL,0))
	 {
		regfree (&preg);
		return (FALSE);
	 }
   regfree (&preg);
   return (TRUE);
}

/*
 * Search ``string'' for matches to the extended regular expression given in
 * ``pattern'', Returns TRUE if a match for ``pattern'' was found in ``string'',
 * FALSE otherwise. If a match was found, the results is stored in ``regs''
 * which is of size ``nregs''. The strings in these registers can be extracted
 * with ereg_extract().
 */
bool ereg (const char *pattern,const char *string,regmatch_t regs[],size_t nregs)
{
   int errcode;
   regex_t preg;
   if ((errcode = regcomp (&preg,pattern,REG_EXTENDED))) return (FALSE);
   if (regexec (&preg,string,nregs,regs,0))
	 {
		regfree (&preg);
		return (FALSE);
	 }
   regfree (&preg);
   return (TRUE);
}

/*
 * Extract a substring from ``string'' according to ``reg'' and store
 * the result in ``substring''.
 */
void ereg_extract (char *substring,const char *string,regmatch_t reg)
{
   strncpy (substring,string + reg.rm_so,reg.rm_eo - reg.rm_so);
   substring[reg.rm_eo - reg.rm_so] = '\0';
}

/*
 * Trim blanks from left and right sides of given string.
 */
void trim (char *str)
{
   register char *s = str;
   /* First we trim left side */
   while (isspace (*s)) s++;
   strcpy (str,s);
   /* Now for the right side */
   s = str + strlen (str) - 1;
   while ((s >= str) && isspace (*s)) *(s--) = '\0';
}

/*
 * Check if the given file is one of the types specified in
 * type, and that we have the necessary permissions to open
 * the file in the mode specified. Mode can be one or more of
 * the following characters: 'r' (readable), 'w' (writable),
 * 'x' (executable), e.g. "rw" would mean that we want read
 * and write permissions. Type can be one or more of the
 * following characters: 'l' (symbolic link), 'r' (regular
 * file), 'd' (directory), 'c' (character device), 'b' (block
 * device), 'f' (fifo), 's' (socket), e.g. "bsf" would mean
 * that we want a block device, socket, or a fifo. Both type
 * and mode must contain at least one character.
 */
bool fperm (const char *filename,const char *type,const char *mode)
{
   struct stat filestat;
   int i;
   /* Check parameters */
   if ((strlen (type) > 7) || (*type == '\0')) return FALSE;
   if ((strlen (mode) > 3) || (*mode == '\0')) return FALSE;
   for (i = 0; i < (int) strlen (type); i++)
	 if ((type[i] != 'l') &&
		 (type[i] != 'r') &&
		 (type[i] != 'd') &&
		 (type[i] != 'c') &&
		 (type[i] != 'b') &&
		 (type[i] != 'f') &&
		 (type[i] != 's')) return FALSE;
   for (i = 0; i < (int) strlen (mode); i++)
	 if ((mode[i] != 'r') &&
		 (mode[i] != 'w') &&
		 (mode[i] != 'x')) return FALSE;
   /* Get file permissions */
   if (stat (filename,&filestat) < 0) return FALSE;
   /* Check file types */
   if ((strchr (type,'l') != NULL) && (!S_ISLNK (filestat.st_mode))) return FALSE;
   if ((strchr (type,'r') != NULL) && (!S_ISREG (filestat.st_mode))) return FALSE;
   if ((strchr (type,'d') != NULL) && (!S_ISDIR (filestat.st_mode))) return FALSE;
   if ((strchr (type,'c') != NULL) && (!S_ISCHR (filestat.st_mode))) return FALSE;
   if ((strchr (type,'b') != NULL) && (!S_ISBLK (filestat.st_mode))) return FALSE;
   if ((strchr (type,'f') != NULL) && (!S_ISFIFO (filestat.st_mode))) return FALSE;
   if ((strchr (type,'s') != NULL) && (!S_ISSOCK (filestat.st_mode))) return FALSE;
   /* Are we almighty root? */
   if (!getuid () || !getgid ()) return (TRUE);
   /* Are we the user? */
   if (filestat.st_uid == getuid ())
	 {
		/* Check file permissions for user */
		if ((strchr (mode,'r') != NULL) && (!(S_IRUSR & filestat.st_mode))) return FALSE;
		if ((strchr (mode,'w') != NULL) && (!(S_IWUSR & filestat.st_mode))) return FALSE;
		if ((strchr (mode,'x') != NULL) && (!(S_IXUSR & filestat.st_mode))) return FALSE;
	 }
   /* Are we in the group then? */
   else if (filestat.st_gid == getgid ())
	 {
		/* Check file permissions for group */
		if ((strchr (mode,'r') != NULL) && (!(S_IRGRP & filestat.st_mode))) return FALSE;
		if ((strchr (mode,'w') != NULL) && (!(S_IWGRP & filestat.st_mode))) return FALSE;
		if ((strchr (mode,'x') != NULL) && (!(S_IXGRP & filestat.st_mode))) return FALSE;
	 }
   /* We're a lowly other then */
   else
	 {
		/* Check file permissions for other */
		if ((strchr (mode,'r') != NULL) && (!(S_IROTH & filestat.st_mode))) return FALSE;
		if ((strchr (mode,'w') != NULL) && (!(S_IWOTH & filestat.st_mode))) return FALSE;
		if ((strchr (mode,'x') != NULL) && (!(S_IXOTH & filestat.st_mode))) return FALSE;
	 }
   return TRUE;
}

/*
 * Read the next line from fp''. returns NULL if
 * successful, the error string otherwise.
 */
const char *readline (FILE *fp,char *result)
{
   if (fgets (result,strsize,fp) == NULL && ferror (fp))
	 return ("Read error");
   else if (feof (fp))
	 {
		*result = '\0';
		return (NULL);
	 }
   if (result[strlen (result) - 1] != '\n') return ("String too long: Increase strsize");
   result[strlen (result) - 1] = '\0';
   return (NULL);
}

/*
 * Check if any process with name ``name'' is running. Returns TRUE, if
 * so, FALSE if not or if some error occurred.
 */
bool alive (const char *name)
{
   DIR *dir;
   FILE *fp;
   struct dirent *entry;
   int i,valid;
   char filename[PATH_MAX],procname[PATH_MAX + 7];
   if ((dir = opendir ("/proc")) == NULL) return (false);
   while((entry = readdir (dir)) != NULL)
	 {
		valid = 1;
		for (i = 0; valid && i < strlen (entry->d_name); i++) if (!isdigit (entry->d_name[i])) valid = 0;
		if (valid)
		  {
			 strcpy (filename,"/proc/");
			 strcat (filename,entry->d_name);
			 strcat (filename,"/status");
			 if((fp = fopen (filename,"r")) != NULL)
			   {
				  if (fgets (procname,sizeof (procname),fp) != NULL && !strncmp ("Name:\t",procname,6))
					{
					   procname[strlen (procname) - 1] = '\0';
					   if (!strcmp (name,procname + 6))
						 {
							fclose (fp);
							closedir (dir);
							return (true);
						 }
					}
				  fclose (fp);
			   }
		  }
	 }
   closedir (dir);
   return (false);
}

