#include <config.h>
#include <assert.h>
#include <stdint.h>
#include <stdlib.h>
#include <stdio.h>

#ifdef HAVE_PNG_H
#include <png.h>
#endif
#ifdef HAVE_LIBPNG10_PNG_H
#include <libpng10/png.h>
#endif
#ifdef HAVE_LIBPNG_PNG_H
#include <libpng/png.h>
#endif

#ifdef HAVE_BYTESWAP_H
#include <byteswap.h>
#endif	
#include "common/strbuf.h"
#include "common/string-utils.h"
#include "common/io-utils.h"
#include "common/error.h"
#include "common/memory.h"
#include "icotool.h"
#include "win32-endian.h"

#define ROW_BYTES(bits) ((((bits) + 31) >> 5) << 2)

static uint32_t simple_vec(uint8_t *data, uint32_t ofs, uint8_t size);

static bool
xfread(void *ptr, size_t size, FILE *stream)
{
	if (fread(ptr, size, 1, stream) < 1) {
		if (ferror(stream))
			warn_errno("cannot read file");
		else
			warn("premature end");
		return false;
	}
	return true;
}

int
extract_icons(FILE *in, char *inname, bool listmode, ExtractNameGen outfile_gen, ExtractFilter filter)
{
	Win32CursorIconFileDir dir;
	Win32CursorIconFileDirEntry *entries = NULL;
	uint32_t offset;
	uint32_t c, d;
	int completed;
	int matched = 0;

	set_message_header(inname);

	if (!xfread(&dir, sizeof(Win32CursorIconFileDir), in))
		goto cleanup;
	fix_win32_cursor_icon_file_dir_endian(&dir);

	if (dir.reserved != 0) {
		warn("not an icon or cursor file (reserved non-zero)");
		goto cleanup;
	}
	if (dir.type != 1 && dir.type != 2) {
		warn("not an icon or cursor file (wrong type)");
		goto cleanup;
	}

	entries = xmalloc(dir.count * sizeof(Win32CursorIconFileDirEntry));
	for (c = 0; c < dir.count; c++) {
		if (!xfread(&entries[c], sizeof(Win32CursorIconFileDirEntry), in))
			goto cleanup;
		fix_win32_cursor_icon_file_dir_entry_endian(&entries[c]);
		if (entries[c].reserved != 0)
			warn("reserved is not zero");
	}
	offset = sizeof(Win32CursorIconFileDir) + dir.count * sizeof(Win32CursorIconFileDirEntry);

	for (completed = 0; completed < dir.count; ) {
		uint32_t min_offset = UINT32_MAX;
		int previous = completed;

		for (c = 0; c < dir.count; c++) {
			if (entries[c].dib_offset == offset) {
				Win32BitmapInfoHeader bitmap;
				Win32RGBQuad *palette = NULL;
				uint32_t palette_count = 0;
				uint32_t image_size, mask_size;
				uint32_t width, height;
				uint8_t *image_data = NULL, *mask_data = NULL;
				png_structp png_ptr = NULL;
				png_infop info_ptr = NULL;
				png_byte *row = NULL;
				char *outname = NULL;
				FILE *out = NULL;

				if (!xfread(&bitmap, sizeof(Win32BitmapInfoHeader), in))
					goto local_cleanup;

				fix_win32_bitmap_info_header_endian(&bitmap);
				if (bitmap.size < sizeof(Win32BitmapInfoHeader)) {
					warn("bitmap header is too short");
					goto local_cleanup;
				}
				if (bitmap.compression != 0) {
					warn("compressed image data not supported");
					goto local_cleanup;
				}
				if (bitmap.x_pels_per_meter != 0)
					warn("x_pels_per_meter field in bitmap should be zero");
				if (bitmap.y_pels_per_meter != 0)
					warn("y_pels_per_meter field in bitmap should be zero");
				if (bitmap.clr_important != 0)
					warn("clr_important field in bitmap should be zero");
				if (bitmap.planes != 1)
					warn("planes field in bitmap should be one");
				if (bitmap.size != sizeof(Win32BitmapInfoHeader)) {
					uint32_t skip = bitmap.size - sizeof(Win32BitmapInfoHeader);
					warn("skipping %d bytes of extended bitmap header", skip);
					fskip(in, skip);
				}
				offset += bitmap.size;

				if (bitmap.clr_used != 0 || bitmap.bit_count < 24) {
					palette_count = (bitmap.clr_used != 0 ? bitmap.clr_used : 1 << bitmap.bit_count);
					palette = xmalloc(sizeof(Win32RGBQuad) * palette_count);
					if (!xfread(palette, sizeof(Win32RGBQuad) * palette_count, in))
						goto local_cleanup;
					offset += sizeof(Win32RGBQuad) * palette_count;
				}

				width = bitmap.width;
				height = ABS(bitmap.height)/2;
				
				image_size = height * ROW_BYTES(width * bitmap.bit_count);
				mask_size = height * ROW_BYTES(width);

				if (entries[c].dib_size	!= bitmap.size + image_size + mask_size + palette_count * sizeof(Win32RGBQuad))
					warn("incorrect total size of bitmap (%d specified; %d real)",
					    entries[c].dib_size,
					    bitmap.size + image_size + mask_size + palette_count * sizeof(Win32RGBQuad)
					);

				image_data = xmalloc(image_size);
				if (!xfread(image_data, image_size, in))
					goto local_cleanup;

				mask_data = xmalloc(mask_size);
				if (!xfread(mask_data, mask_size, in))
					goto local_cleanup;

				offset += image_size;
				offset += mask_size;
				completed++;

				if (!filter(completed, width, height, bitmap.bit_count, palette_count, dir.type == 1,
						(dir.type == 1 ? 0 : entries[c].hotspot_x),
						(dir.type == 1 ? 0 : entries[c].hotspot_y)))
					goto done;
				matched++;

				if (!listmode) {
					png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, NULL /*user_error_fn, user_warning_fn*/);
					if (!png_ptr) {
						warn("cannot initialize PNG library");
						goto local_cleanup;
					}
					info_ptr = png_create_info_struct(png_ptr);
					if (!info_ptr) {
						warn("cannot create PNG info structure - out of memory");
						goto local_cleanup;
					}

					outname = inname;
					out = outfile_gen(&outname, width, height, bitmap.bit_count, completed);
					restore_message_header();
					set_message_header(outname);

					if (out == NULL) {
						warn_errno("cannot create file");
						goto local_cleanup;
					}
					png_init_io(png_ptr, out);

					restore_message_header();
					set_message_header(inname);

					png_set_IHDR(png_ptr, info_ptr,	width, height, 8,
							PNG_COLOR_TYPE_RGB_ALPHA,
							PNG_INTERLACE_NONE,
							PNG_COMPRESSION_TYPE_DEFAULT,
							PNG_FILTER_TYPE_DEFAULT);
					png_write_info(png_ptr, info_ptr);
				}

				row = xmalloc(width * 4);

				for (d = 0; d < height; d++) {
					uint32_t x;
					uint32_t y = (bitmap.height < 0 ? d : height - d - 1);
					uint32_t imod = y * (image_size / height) * 8 / bitmap.bit_count;
					uint32_t mmod = y * (mask_size / height) * 8;

					for (x = 0; x < width; x++) {
						uint32_t color = simple_vec(image_data, x + imod, bitmap.bit_count);

						if (bitmap.bit_count <= 16) {
							if (color >= palette_count) {
								warn("color out of range in image data");
								goto local_cleanup;
							}
							row[4*x+0] = palette[color].red;
							row[4*x+1] = palette[color].green;
							row[4*x+2] = palette[color].blue;
						} else {
							row[4*x+0] = (color >> 16) & 0xFF;
							row[4*x+1] = (color >>  8) & 0xFF;
							row[4*x+2] = (color >>  0) & 0xFF;
						}
						if (bitmap.bit_count == 32)
						    row[4*x+3] = (color >> 24) & 0xFF;
						else
						    row[4*x+3] = simple_vec(mask_data, x + mmod, 1) ? 0 : 0xFF;
					}

					if (!listmode)
						png_write_row(png_ptr, row);
				}

				if (listmode) {
					printf("--%s --index=%d --width=%d --height=%d --bit-depth=%d --palette-size=%d",
							(dir.type == 1 ? "icon" : "cursor"), completed, width, height,
							bitmap.bit_count, palette_count);
					if (dir.type == 2)
						printf(" --hotspot-x=%d --hotspot-y=%d", entries[c].hotspot_x, entries[c].hotspot_y);
					printf("\n");
				} else {
					png_write_end(png_ptr, info_ptr);
					png_destroy_write_struct(&png_ptr, &info_ptr);
					/*restore_message_header();*/
				}

			done:

				if (row != NULL)
					free(row);
				if (palette != NULL)
					free(palette);
				if (outname != NULL)
					free(outname);
				if (image_data != NULL) {
					free(image_data);
					free(mask_data);
				}
				if (out != NULL)
					fclose(out);
				continue;

			local_cleanup:

				/*restore_message_header();*/
				if (row != NULL)
					free(row);
				if (palette != NULL)
					free(palette);
				if (image_data != NULL) {
					free(image_data);
					free(mask_data);
				}
				if (out != NULL)
					fclose(out);
				if (png_ptr != NULL)
					png_destroy_read_struct(&png_ptr, &info_ptr, NULL);
				if (outname != NULL)
					free(outname);
				goto cleanup;
			} else {
				if (entries[c].dib_offset > offset)
						min_offset = MIN(min_offset, entries[c].dib_offset);
			}
		}

		if (previous == completed) {
			if (min_offset < offset) {
				warn("offset of bitmap header incorrect (too low)");
				goto cleanup;
			}
			warn("skipping %d bytes of garbage at %d", min_offset-offset, offset);
			fskip(in, min_offset - offset);
			offset = min_offset;
		}
	}

	restore_message_header();
	free(entries);
	return matched;

cleanup:

	restore_message_header();
	free(entries);
	return -1;
}

static uint32_t
simple_vec(uint8_t *data, uint32_t ofs, uint8_t size)
{
	switch (size) {
	case 1:
		return (data[ofs/8] >> (7 - ofs%8)) & 1;
	case 2:
		return (data[ofs/4] >> ((3 - ofs%4) << 1)) & 3;
	case 4:
		return (data[ofs/2] >> ((1 - ofs%2) << 2)) & 15;
	case 8:
		return data[ofs];
	case 16:
		return data[2*ofs] | data[2*ofs+1] << 8;
	case 24:
		return data[3*ofs] | data[3*ofs+1] << 8 | data[3*ofs+2] << 16;
	case 32:
		return data[4*ofs] | data[4*ofs+1] << 8 | data[4*ofs+2] << 16 | data[4*ofs+3] << 24;
	}

	return 0;
}
