//$Id: bitmapfamily-win.cc,v 1.3 2004/01/27 15:27:17 cactus Exp $ -*- c++ -*-

/* Guikachu Copyright (C) 2001-2003 RDI Gerg <cactus@cactus.rulez.org>
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "bitmapfamily-win.h"
#include "bitmapfamily-res-ops.h"

#include "config.h"
#include <libgnome/libgnome.h>

#ifdef GUIKACHU_USE_WM_ICONS
#include <libgnomeui/gnome-window-icon.h>
#endif

#include <sigc++/retbind.h>
#include "argumentsink.h"

#include "io/io.h"

#include "property-ops-resource.h"

#include "widgets/propertytable.h"
#include "widgets/entry.h"

#include "depthlist.h"

#include "form-editor/form-editor-canvas.h"
#include "preferences.h"

#include <gdk-pixbuf/gdk-pixbuf.h>

#include <gtk--/fileselection.h>
#include <gtk--/frame.h>
#include <gtk--/button.h>
#include <gtk--/buttonbox.h>
#include <gtk--/box.h>
#include <gtk--/pixmap.h>
#include <gtk--/main.h>
#include <gtk--/alignment.h>
#include <gtk--/scrolledwindow.h>
#include <gnome--/stock.h>

using namespace Guikachu;

namespace {
    Gtk::Button * create_stock_button (const std::string &stock_id, const std::string &title)
    {
        Gnome::StockPixmap *stock_pm;
        stock_pm = new Gnome::StockPixmap (stock_id.c_str ());

        return Gtk::wrap (GTK_BUTTON (gnome_pixmap_button (GTK_WIDGET (manage (stock_pm)->gtkobj ()),
                                                           title.c_str ())));
    }
    
    class ImageFileSelector: public SigC::Object
    {
        Gtk::FileSelection             fs;
        Resources::BitmapFamily       *res;
        Resources::Bitmap::BitmapType  type;

    public:
        ImageFileSelector (Gtk::Window &parent, Resources::BitmapFamily *res,
                           Resources::Bitmap::BitmapType type, bool save);
        void run ();
        
    private:
        void close_cb ();
        void open_cb ();
	void export_cb ();
    };
    
} // anonymous namespace

GUI::BitmapFamilyWindow::BitmapFamilyWindow (Resources::BitmapFamily *res_):
    res (res_),
    type (Resources::Bitmap::TYPE_COLOR_256)
{
    using namespace SigC;
    using ResourceOps::PropChangeOpFactory;
    using ResourceOps::RenameOpFactory;
    
    window.delete_event.connect (slot (this, &BitmapFamilyWindow::delete_event_impl));
    window.set_policy (false, true, false);
    
#ifdef GUIKACHU_USE_WM_ICONS
    gnome_window_icon_set_from_file (window.gtkobj (), GNOME_ICONDIR "/guikachu.png");
#endif

    Gtk::VBox *main_vbox = new Gtk::VBox (false, 5);
    main_vbox->set_border_width (5);
    
    /* Resource ID */
    // TODO: Underline accelerator for resource entry
    Gtk::Label *id_label = new Gtk::Label (_("Resource ID:"), 1, 0.5);
    Gtk::Widget *id_entry = new GUI::PropertyEditors::Entry (false, res->id, new RenameOpFactory (res));
    Gtk::HBox *id_hbox = new Gtk::HBox (false, 5);
    id_hbox->pack_start (*manage (id_label), false, false);
    id_hbox->pack_start (*manage (id_entry), true, true);
    main_vbox->pack_start (*manage (id_hbox), false, false);
    
    /* Buttons */
#if 0
    Gtk::Box *buttonbox = new Gtk::HButtonBox (GTK_BUTTONBOX_EDGE);
#else
    Gtk::Box *buttonbox = new Gtk::HBox (false, 5);
#endif
    main_vbox->pack_end (*manage (buttonbox), false, false);
    load_button = create_stock_button (GNOME_STOCK_PIXMAP_OPEN, _("Load"));
    load_button->clicked.connect (SigC::slot (this, &BitmapFamilyWindow::load_cb));
    buttonbox->add (*manage (load_button));

    export_button = create_stock_button (GNOME_STOCK_PIXMAP_SAVE_AS, _("Export"));
    export_button->clicked.connect (SigC::slot (this, &BitmapFamilyWindow::export_cb));
    buttonbox->add (*manage (export_button));

    remove_button = create_stock_button (GNOME_STOCK_PIXMAP_TRASH, _("Remove"));
    remove_button->clicked.connect (SigC::slot (this, &BitmapFamilyWindow::remove_cb));
    buttonbox->add (*manage (remove_button));

    /* List of color depths */
    DepthList *depth_list = new DepthList (res);
    depth_list->type_changed.connect (SigC::slot (this, &BitmapFamilyWindow::type_changed_cb));
    
    Gtk::ScrolledWindow *list_frame = new Gtk::ScrolledWindow;
    list_frame->set_policy (GTK_POLICY_NEVER, GTK_POLICY_NEVER);
    list_frame->add_with_viewport (*manage (depth_list));

    /* Preview */
    Gtk::Alignment *align = new Gtk::Alignment (0.5, 0.5, 0, 0);
    Gtk::Frame *frame = new Gtk::Frame;
    frame->add (preview_pixmap);
    align->add (*manage (frame));
    
    Gtk::Box *list_hbox = new Gtk::HBox (false, 5);
    list_hbox->pack_start (*manage (list_frame), false, false);
    list_hbox->pack_end (*manage (align), true, true, 10);
    main_vbox->pack_start (*manage (list_hbox), true, true);
    
    window.add (*manage (main_vbox));
    
    res->changed.connect (slot (this, &BitmapFamilyWindow::update));
    update ();
}

void GUI::BitmapFamilyWindow::show ()
{
    window.show_all ();
    window.get_window ().raise ();
}

int GUI::BitmapFamilyWindow::delete_event_impl (GdkEventAny *e)
{
    window.hide ();
    return true;
}

void GUI::BitmapFamilyWindow::update ()
{
    // Update window title
    gchar *title_buf = g_strdup_printf (_("Bitmap Group: %s"), res->id ().c_str ());
    window.set_title (title_buf);
    g_free (title_buf);

    Magick::Image image = res->get_image (type);
    
    GdkPixbuf *pixbuf = FormEditor::render_bitmap (image, type);

    // Update buttons
    export_button->set_sensitive (true);
    remove_button->set_sensitive (true);    
    
    if (!pixbuf)
    {
        preview_pixmap.set ("");
        export_button->set_sensitive (false);
        remove_button->set_sensitive (false);
    }

    if (type == Resources::Bitmap::TYPE_GREY_16)
    {
        load_button->set_sensitive (!res->get_image (Resources::Bitmap::TYPE_COLOR_16).isValid ());
    }

    if (type == Resources::Bitmap::TYPE_COLOR_16)
    {
        load_button->set_sensitive (!res->get_image (Resources::Bitmap::TYPE_GREY_16).isValid ());
    }
    
    // Update the preview
    if (pixbuf)
    {
        GdkPixmap *pm;
        GdkBitmap *bm;
        
        guint32 background_color = 0xFFFFFF;

        if (!res->get_manager ()->get_target ()->screen_color && type < Resources::Bitmap::TYPE_COLOR_16)
        {
            Gdk_Color c (Preferences::FormEditor::get_color_bg ());
            background_color = ((c.red >> 8) << 16) + ((c.green >> 8) << 8) + (c.blue  >> 8);
        }
        
        GdkPixbuf *pixbuf_with_background = gdk_pixbuf_composite_color_simple (
            pixbuf, gdk_pixbuf_get_width (pixbuf), gdk_pixbuf_get_height (pixbuf),
            GDK_INTERP_NEAREST, 255, 1, background_color, background_color);
        gdk_pixbuf_unref (pixbuf);

        if (pixbuf_with_background)
        {
            gdk_pixbuf_render_pixmap_and_mask (pixbuf_with_background, &pm, &bm, 127);
            preview_pixmap.set (pm, bm);
        }
        
        gdk_pixbuf_unref (pixbuf_with_background);
    }
}

void GUI::BitmapFamilyWindow::load_cb ()
{
    ImageFileSelector ifs (window, res, type, false);
    ifs.run ();
}

void GUI::BitmapFamilyWindow::export_cb ()
{
    ImageFileSelector ifs (window, res, type, true);
    ifs.run ();
}

void GUI::BitmapFamilyWindow::remove_cb ()
{
    UndoOp *op = new ResourceOps::BitmapFamilyOps::ImageRemoveOp (res, type);
    res->set_image (type, Magick::Image ());
    res->get_manager ()->get_undo_manager ().push (op);
}

void GUI::BitmapFamilyWindow::type_changed_cb (Resources::Bitmap::BitmapType type_)
{
    type = type_;
    update ();
}

ImageFileSelector::ImageFileSelector (Gtk::Window                   &parent,
                                      Resources::BitmapFamily       *res_,
                                      Resources::Bitmap::BitmapType  type_,
                                      bool                           save) :
    res (res_),
    type (type_)
{
    char *title;
    if (save)
    {
	title = g_strdup_printf (_("Exporting %s"), res->id ().c_str ());
	fs.show_fileop_buttons ();
	fs.get_ok_button ()->clicked.connect (
	    SigC::slot (this, &ImageFileSelector::export_cb));
    } else {
	title = g_strdup_printf (_("Loading image file to %s"), res->id ().c_str ());
	fs.hide_fileop_buttons ();
	fs.get_ok_button ()->clicked.connect (
	    SigC::slot (this, &ImageFileSelector::open_cb));
    }
    fs.set_title (title);
    g_free (title);
    
    fs.get_cancel_button ()->clicked.connect (SigC::slot (this, &ImageFileSelector::close_cb));
    fs.delete_event.connect (
	SigC::retbind (
	    SigC::hide<void, GdkEventAny*> (SigC::slot (this, &ImageFileSelector::close_cb)),
	    1));
    
    fs.set_transient_for (parent);
    fs.set_modal (true);
}

void ImageFileSelector::close_cb ()
{
    fs.hide ();
    Gtk::Main::quit ();
}

void ImageFileSelector::open_cb ()
{
    std::string uri = IO::create_canonical_uri (fs.get_filename ());

    UndoOp *op = new ResourceOps::BitmapFamilyOps::ImageChangeOp (res, type, res->get_image (type));
    res->load_file (type, uri);
    res->get_manager ()->get_undo_manager ().push (op);
    
    close_cb ();
}

void ImageFileSelector::export_cb ()
{
    // TODO: Show a warning before overwriting files
    Magick::Image image = res->get_image (type);
    image.write (fs.get_filename ());
    
    close_cb ();
}

void ImageFileSelector::run ()
{
    fs.show ();
    Gtk::Main::run ();
}
