/* doodlecb.c - Callbacks to widgets in doodle application
 *
 * Copyright (C) 1996 Eric M. Ludlam
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.ai.mit.edu.
 * 
 * 
 * Description:
 *   Contains callbacks to the doodle application that represent use
 * input of some sort.  This includes menu-item callbacks, tool
 * selection, and the like.   
 * 
 * $Log: doodlecb.c,v $
 * Revision 1.2  1997/03/23 15:58:58  zappo
 * Standardized on eraser/dot size
 *
 * Revision 1.1  1996/03/17 15:22:15  zappo
 * Initial revision
 *
 * History:
 * zappo   3/8/96     Created
 *
 * Tokens: ::Header:: doodle.h
 */

#include <X11/Xlib.h>
#include <X11/keysym.h>

/* Some Athena widgets we need */
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/AsciiTextP.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/SmeBSB.h>
#include <X11/Xaw/Toggle.h>

#include "doodle.h"

char *tool_names[] = {
  "No Tool",
  "Point",
  "Scribble",
  "Line",
  "Rectangle",
  "Elipse",
  "Eraser"
};

char *color_names[] =
{
  "None",
  "white",
  "black",
  "red",
  "green",
  "blue",
  "orange",
  "yellow",
  "purple"
};


/*
 * Function: X_mouse_doodle
 *
 *   Handle mouse input for the doodle space.  Reads XCtxt to see
 * which tool we must currently draw in.
 *
 * Returns:     Nothing
 * Parameters:  w     - Widget w
 *              XCtxt - Context
 *              e     - Pointer to e
 * History:
 * zappo   3/8/96     Created
 */
void X_mouse_doodle(w, XCtxt, e)
     Widget w;
     struct Xcontext *XCtxt;
     XEvent *e;
{
  static int have_push = False;
  static int x[100], y[100];
  static int pntcnt = 0;		/* current point count */

  if(have_push || (e->type == ButtonPress))
    {
      switch(XCtxt->currentTool)
	{
	case Point:
	  x[0] = e->xbutton.x;
	  y[0] = e->xbutton.y;
	  XFillArc(XCtxt->display, XCtxt->doodlePix, XCtxt->gcs[XCtxt->currentColor],
		   x[0]-DOTSIZE/2, y[0]-DOTSIZE/2, 7, 7, 0, 360*64);
	  dnet_send_draw_command(tool_names[XCtxt->currentTool],
				 color_names[XCtxt->currentColor],
				 1, x, y);
	  pntcnt = 0;
	  break;
	case Scribble:
	  if(e->type == ButtonPress)
	    {
	      x[0] = e->xbutton.x;
	      y[0] = e->xbutton.y;
	      pntcnt = 1;
	    }
	  else
	    {
	      x[pntcnt] = e->xbutton.x;
	      y[pntcnt] = e->xbutton.y;
	      XDrawLine(XCtxt->display, XCtxt->doodlePix, 
			XCtxt->gcs[XCtxt->currentColor],
			x[pntcnt-1], y[pntcnt-1], x[pntcnt], y[pntcnt]);
	      pntcnt++;
	      if(e->type == ButtonRelease)
		{
		  dnet_send_draw_command(tool_names[XCtxt->currentTool],
					 color_names[XCtxt->currentColor],
					 pntcnt, x, y);
		  pntcnt = 0;
		}
	      else if(pntcnt == (sizeof(x) / sizeof(int)))
		{
		  dnet_send_draw_command(tool_names[XCtxt->currentTool],
					 color_names[XCtxt->currentColor],
					 pntcnt, x, y);
		  x[0] = x[pntcnt-1];
		  y[0] = y[pntcnt-1];
		  pntcnt = 1;
		}
	    }
	  break;
	case Line:
	  if(e->type == ButtonPress)
	    {
	      x[0] = x[1] = e->xbutton.x;
	      y[0] = y[1] = e->xbutton.y;
	    }
	  else
	    {
	      XDrawLine(XCtxt->display, XCtxt->doodlePix, XCtxt->xor,
			x[0], y[0], x[1], y[1]);
	      x[1] = e->xbutton.x;
	      y[1] = e->xbutton.y;
	      XDrawLine(XCtxt->display, XCtxt->doodlePix,
			(e->type == ButtonRelease)?
			XCtxt->gcs[XCtxt->currentColor]:XCtxt->xor,
			x[0], y[0], x[1], y[1]);
	      if(e->type == ButtonRelease)
		{
		  dnet_send_draw_command(tool_names[XCtxt->currentTool],
					 color_names[XCtxt->currentColor],
					 2, x, y);
		  pntcnt = 0;
		}
	    }
	  break;
	case Rectangle:
	  if(e->type == ButtonPress)
	    {
	      x[0] = x[1] = e->xbutton.x;
	      y[0] = y[1] = e->xbutton.y;
	    }
	  else
	    {
	      
	      XDrawRectangle(XCtxt->display, XCtxt->doodlePix, XCtxt->xor,
			     MIN(x[0],x[1]), MIN(y[0],y[1]),
			     ABS(x[1]-x[0]), ABS(y[1]-y[0]));
	      x[1] = e->xbutton.x;
	      y[1] = e->xbutton.y;
	      XDrawRectangle(XCtxt->display, XCtxt->doodlePix, 
			     (e->type == ButtonRelease)?
			      XCtxt->gcs[XCtxt->currentColor]:XCtxt->xor,
			     MIN(x[0],x[1]), MIN(y[0],y[1]),
			     ABS(x[1]-x[0]), ABS(y[1]-y[0]));
	      if(e->type == ButtonRelease)
		{
		  dnet_send_draw_command(tool_names[XCtxt->currentTool],
					 color_names[XCtxt->currentColor],
					 2, x, y);
		  pntcnt = 0;
		}
	    }
	  break;
	case Elipse:
	  if(e->type == ButtonPress)
	    {
	      x[0] = x[1] = e->xbutton.x;
	      y[0] = y[1] = e->xbutton.y;
	    }
	  else
	    {
	      XDrawArc(XCtxt->display, XCtxt->doodlePix, XCtxt->xor,
		       MIN(x[0],x[1]), MIN(y[0],y[1]),
		       ABS(x[1]-x[0]), ABS(y[1]-y[0]), 0, 64*360);
	      x[1] = e->xbutton.x;
	      y[1] = e->xbutton.y;
	      XDrawArc(XCtxt->display, XCtxt->doodlePix, 
		       (e->type == ButtonRelease)?
		       XCtxt->gcs[XCtxt->currentColor]:XCtxt->xor,
		       MIN(x[0],x[1]), MIN(y[0],y[1]),
		       ABS(x[1]-x[0]), ABS(y[1]-y[0]), 0, 64*360);
	      if(e->type == ButtonRelease)
		{
		  dnet_send_draw_command(tool_names[XCtxt->currentTool],
					 color_names[XCtxt->currentColor],
					 2, x, y);
		  pntcnt = 0;
		}
	    }
	  break;
	case Eraser:
	  x[0] = e->xbutton.x;
	  y[0] = e->xbutton.y;
	  XFillArc(XCtxt->display, XCtxt->doodlePix, XCtxt->white,
		   x[0]-ERASERSIZE/2, y[0]-ERASERSIZE/2, 7, 7, 0, 360*64);
	  dnet_send_draw_command(tool_names[XCtxt->currentTool], 
				 color_names[White],
				 1, x, y);
	  pntcnt = 0;
	  break;
	default:
	  break;
	} /* switch */
      
      X_expose_doodle(XCtxt->doodleCanvas, XCtxt, NULL);

      /* Set this last as a flag for a down-button */
      have_push = True;
    }
  if(e->type == ButtonRelease)
    {
      have_push = False;
    }
} /* X_mouse_doodle( ) */


/*
 * Function: X_toggle_cb
 *
 *   Whenever a toggle for a tool is selected, this gets called.  It
 * usually get's called twice, but the first one is summarily ignored.
 *
 * Returns:     Nothing
 * Parameters:  w         - Widget w
 *              XCtxt     - Context
 *              call_data - Number of call data
 * History:
 * zappo   3/8/96     Created
 */
void X_toggle_cb(w, XCtxt, call_data)
     Widget w;
     struct Xcontext *XCtxt;
     XtPointer call_data;
{
  int which;

  which = (int)XawToggleGetCurrent(w);

  if(which)
    {
      XCtxt->currentTool = which;
    }
} /* XW_build_radio_group( ) */

void X_color_cb(w, XCtxt, call_data)
     Widget w;
     struct Xcontext *XCtxt;
     XtPointer call_data;
{
  int which;

  which = (int)XawToggleGetCurrent(w);

  if(which)
    {
      XCtxt->currentColor = which;
    }
} /* XW_build_radio_group( ) */


/*
 * Function: X_quit
 *
 *   Exit's the doodle program
 *
 * Returns:     Nothing
 * Parameters:  None
 *
 * History:
 * zappo   3/8/96     Created
 */
void X_quit()
{
  exit(0);
} /* X_quit( ) */

