/* gtl_util.c - useful routines that may not be in libc.a
 *
 * Copyright (C) 1995 Eric M. Ludlam
 * Copyright (C) 1997 Free Software Foundation
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 *
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 *
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * Description:
 *  Simple functions such as strcasecmp do not exist on all systems.
 * These are wholesale replacements with different names.
 *
 * $Log: gtl_util.c,v $
 * Revision 1.8  1997/12/14 19:21:56  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.7  1996/03/02 03:29:28  zappo
 * Updated gtl_gethome again.  I can't seem to get this to work reliably.
 *
 * Revision 1.6  1995/12/10  03:47:29  zappo
 * Changed all error messages to use DISP_message
 * Added function GTL_gethome, which tries many methods to return someone
 * home directory
 *
 * Revision 1.5  1995/11/21  03:55:05  zappo
 * Added GTL_gethome, to get any users home directory without
 * environement variables, or other hokey methods
 *
 * Revision 1.4  1995/04/10  21:43:01  zappo
 * Replace GNU version of strtok with my own since theirs was too
 * dependant on thier own library.  Both strdup and strtok will be used
 * only if they don't exist in the system.
 *
 * Revision 1.3  1995/04/08  20:03:49  zappo
 * Added strdup and strtok if not provided by default.
 *
 * Revision 1.2  1995/04/05  00:55:03  zappo
 * Made scmp take const input
 *
 * Revision 1.1  1995/03/30  02:33:32  zappo
 * Initial revision
 *
 *
 * Tokens: ::Header:: gtalklib.h
 */
#include "gtalklib.h"

#ifdef HAVE_PWD_H
#include <pwd.h>
#endif

/*
 * Function: GTL_scmp
 *
 * Compares s1 and s2 for len characters.  Case is ignored.  Success
 * is returned when they are the same.
 *
 * Returns:     int - Success means they are the same
 * Parameters:  s1  - String 1
 *              s2  - String 2
 *              len - length of string
 * History:
 * eml Jul 30, 1994   Created 
 * zappo   3/29/95    Moved to a generalized function, and cleaned it up.
 */
int GTL_scmp(s1, s2, len)
     const char *s1, *s2;
     int   len;
{
  int i;
  for(i = 0; (i<len) && s1[i] && s2[i]; i++)
    {
      char test1, test2;

      if((s1[i] >= 'A') && (s1[i] <= 'Z'))
	test1 = s1[i] - 'A' + 'a';
      else
	test1 = s1[i];

      if((s2[i] >= 'A') && (s2[i] <= 'Z'))
	test2 = s2[i] - 'A' + 'a';
      else
	test2 = s2[i];

      if(test1 != test2)	/* not equal */
	return Fail;
    }
  if(i == len)
    return Success;		/* all match */
  else
    return Fail;		/* to short */
}


/*
 * Function: GTL_gethome
 *
 *   Returns the home directory of user specified by USER.  This is
 * because $HOME doesn't always exist, and searching the PW file is
 * the most reliable way of getting this information.
 *
 * Returns:     char * - 
 * Parameters:  user - user name whose home directory we are looking up
 *              self - If this user is the process owner = non-0
 *
 * History:
 * zappo   11/8/95    Created
 */
char *GTL_gethome(user, self)
     char *user;
     int   self;
{
  struct passwd *pws;

  if(!user)
    {
      return NULL;		/* do not lookup info in this case */
    }
  else
    {
      static char *myhome = NULL;

      if(self && myhome) return myhome;

      pws = getpwnam(user);
  
      if(pws == NULL)
	{
	  if(!self)
	    {
	      DISP_message(NULL, "GTL_gethome: getpwname: Cannot find user",
			   LOG_ERR);
	      return NULL;
	    }
	  else
	    {
	      myhome = (char*)getenv("HOME");

	      if(!myhome)
		{
		  DISP_message(NULL, "GTL_gethome: getenv: Cannot find env HOME",
			       LOG_ERR);
		  return NULL;
		}
	      return myhome;
	    }
	}

      if(self) 
	{
	  /* Make a copy because it's overwritten on some systems */
	  myhome = strdup(pws->pw_dir);
	  if(!myhome)
	    {
	      DISP_message(NULL, "GTL_gethome: strdup: malloc failure",
			   LOG_ERR);
	      exit(1);
	    }
	}
      /* Is PWS allocated???  Must I free it???  Possible memory leak! */
      return pws->pw_dir;
    }
}

#ifndef HAVE_STRDUP

/*
 * Function: strdup
 *
 *   Return a malloced version of the parameter string.  This is
 * supplied if your system doesn't have a strdup of it's own.
 *
 * Returns:     char * - 
 * Parameters:  str - Pointer toCharacter of string
 *
 * History:
 * zappo   4/4/95     Created
 */
char *strdup(__s)
     const char *__s;
{
  char *tmp;

  /* get some space */
  tmp = (char *)malloc(strlen(__s));

  /* check before copy */
  if(tmp) strcpy(tmp, __s);

  /* return the null if that is what happened */
  return tmp;
}
#endif /* HAVE_STRDUP */
#ifndef HAVE_STRTOK

/*
 * Function: strtok
 *
 *   Emulate strtok found in most libraries.  It is not done
 * efficiently, but that's ok.  It's only used during commandline
 * parsing anyways
 *
 * Returns:     char * - 
 * Parameters:  s     - String to tokenize
 *              delim - token seperator set
 * History:
 * zappo   4/8/95     Created
 */
char *strtok(__s, __delim)
     char *__s;
     const char *__delim;
{
  static char *lastmatch = NULL;
  char *end;

  /* initial cases/errors */
  if(!__s) __s = lastmatch;
  if(!__s || !__delim) return NULL;

  /* first scan over first batch of delimiters */
  while(*__s && strchr(__delim, *__s)) __s++;

  /* now scan until we hit the next delimiter */
  end = __s;
  while(*end && !strchr(__delim, *end)) end++;

  /* terminate the token */
  *end = '\0';
  /* prepare for next scan */
  lastmatch = ++end;
  /* return the result */
  if(*__s)
    return __s;
  else
    return NULL;
}
#endif /* HAVE_STRTOK */
