/* gtl_msgm.c - Daemon Message printing, sizes, etc.
 *
 * Copyright (C) 1994, 1995, 1996, 1998 Eric M. Ludlam 
 * Copyright (C) 1997  Free Software Foundation
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * Description:
 * 
 *   Common routines for managing daemon messages, sizes, etc.
 * Created from sources in gt_daem.c which was found to be common with
 * gtalkd source code.
 *
 * $Log: gtl_msgm.c,v $
 * Revision 1.14  1998/05/01 16:45:42  zappo
 * Updated the ControlExtension type, and it's print routine.
 *
 * Revision 1.13  1997/12/14 19:21:30  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.12  1997/12/02 22:56:55  zappo
 * Added new response type to code for printing.
 *
 * Revision 1.11  1997/10/25 01:51:26  zappo
 * When printing message size errors, also display the desired sizes.
 *
 * Revision 1.10  1996/03/02 03:27:57  zappo
 * Moved prototypes to a new header file: gtl_union.h
 *
 * Revision 1.9  1995/12/10  03:43:44  zappo
 * Error message now prints to stdout
 *
 * Revision 1.8  1995/03/25  04:15:06  zappo
 * Updated copyright
 *
 * Revision 1.7  1995/03/03  02:48:40  zappo
 * Added extionsion printing, swap id_num before printing, and second
 * half of messages are not printed unless verbosity >=  2
 *
 * Revision 1.6  1995/02/28  02:58:16  zappo
 * Made sure extension in gtalk message type variable was printed
 *
 * Revision 1.5  1995/02/11  14:12:02  zappo
 * Added new message name "try here" record.
 *
 * Revision 1.4  1995/01/29  14:25:34  zappo
 * Fixed some -Wall warnings
 *
 * Revision 1.3  1995/01/14  17:43:27  zappo
 * Added printing of special message extensions.
 * 
 * History:
 * zappo   9/14/94         Created
 *
 * Tokens: ::Header:: gtl_union.h
 */
#include "gtalklib.h"
#include "otalk.h"
#include "talk.h"
#include "gtalk.h"
#include "gtl_union.h"

char *msg_types[] = {
  "Leave Invite",
  "Look up",
  "Delete",
  "Announce",
  "Reply Query",
};
char *msg_responses[] = {
  "Success",
  "Not Here",
  "Failed",
  "Machine Unknown",
  "Permision Denied",
  "Unknown Request",
  "Bad Version",
  "Bad Address",
  "Bad Control Address",
  "No Caller",
  "Try Here",
  "Selectivly Refused"
};

union ctl_msg      Control;
union ctl_response Response;
CTL_MSG_EXTENSION  ControlExtension;


/*
 * Function: control_size, response_size
 *
 * These functions return the size of each portion of the
 * previously declaired unions.  This allows a send based
 * on message size which, in turn, is based on daemon type.
 * 
 * Parameters:  io   - device we are sending to.  need to know type.
 *              type - Type of
 * History:
 * eml 3/1/94
 * zappo   9/16/94         Added parameter type
 */
int control_size(type)
     enum DaemonType type;
{
  switch(type) {
  case GTALKD: return sizeof(Control.gtalk);
  case NTALKD: return sizeof(Control.talk);
  case OTALKD: return sizeof(Control.otalk);
  default: return 0;
  }
}
int response_size(type)
     enum DaemonType type;
{
  switch(type) {
  case GTALKD: return sizeof(Response.gtalk);
  case NTALKD: return sizeof(Response.talk);
  case OTALKD: return sizeof(Response.otalk);
  default: return 0;
  }
}


/*
 * Function: ETM_control_print
 *
 * the following debugging command will print the contents of the 
 * control message.
 * 
 * Parameters:  type - Type of
 *
 * History:
 * eml 3/1/94
 * zappo   9/16/94         changed parameter io to type
 */
void ETM_control_print(type)
     enum DaemonType type;
{
  printf("Control message is:\n");
  switch(type) {
  case GTALKD:
    printf("Control.gtalk.vers     : %d\n", Control.gtalk.vers);
    if(Control.talk.type > REPLY_QUERY)
      printf("Control.gtalk.type     : Unknown %d\n", Control.gtalk.type);
    else
      printf("Control.gtalk.type     : %s\n", msg_types[Control.gtalk.type]);
    printf("Extension length       : %d\n", Control.gtalk.extended);
    if(Control.gtalk.extended != EXT_extension_length())
      printf("  *** INTERNAL ERROR.  extend field != calculated extension!\n");
    if(verbose >= 2)
      {
	printf("Control.gtalk.addr     : "); 
	print_swapped_sockaddr(&Control.gtalk.addr);
	printf("\nControl.gtalk.ctl_addr : ");
	print_swapped_sockaddr(&Control.gtalk.ctl_addr);
	printf("\nControl.gtalk.pid      : %ld\n", htonl(Control.gtalk.pid));
	printf("Control.gtalk.id_num   : %ld\n", htonl(Control.gtalk.id_num));
	printf("Control.gtalk.l_name   : %s\n", Control.gtalk.l_name);
	printf("Control.gtalk.r_name   : %s\n", Control.gtalk.r_name);
	printf("Control.gtalk.r_tty    : %s\n", Control.gtalk.r_tty);
	if(Control.gtalk.extended)
	  {
	    char *data;
	    int  length;

	    data = EXT_fetch_extension_data(EXTENSION_APPLICATION_NAME,
					    &ControlExtension, &length);
	    if(data)
	      printf("Control.extension.application_name: %*s\n",
		     length-1, data);
	    data = EXT_fetch_extension_data(EXTENSION_PERSONAL_NAME,
					    &ControlExtension, &length);
	    if(data)
	      printf("Control.extension.personal_name   : %*s\n",
		     length-1, data);
	  }
      }
    break;
  case NTALKD:
    printf("Control.talk.vers     : %d\n", Control.talk.vers);
    if(Control.talk.type > REPLY_QUERY)
      printf("Control.talk.type     : Unknown %d\n", Control.talk.type);
    else
      printf("Control.talk.type     : %s\n", msg_types[Control.talk.type]);
    if(verbose >= 2)
      {
	printf("Control.talk.addr     : "); 
	print_swapped_sockaddr(&Control.talk.addr);
	printf("\nControl.talk.ctl_addr : ");
	print_swapped_sockaddr(&Control.talk.ctl_addr);
	printf("\nControl.talk.pid      : %ld\n", htonl(Control.talk.pid));
	printf("Control.talk.id_num   : %ld\n", htonl(Control.talk.id_num));
	printf("Control.talk.l_name   : %s\n", Control.talk.l_name);
	printf("Control.talk.r_name   : %s\n", Control.talk.r_name);
	printf("Control.talk.r_tty    : %s\n", Control.talk.r_tty);
      }
    break;
  case OTALKD:
    if(Control.otalk.type > ANNOUNCE)
      printf("Control.otalk.type     : Unknown %d\n", Control.otalk.type);
    else
      printf("Control.otalk.type     : %s\n", msg_types[Control.otalk.type]);
    if(verbose >= 2)
      {
	printf("Control.otalk.addr     : ");
	print_swapped_sockaddr(&Control.otalk.addr);
	printf("\nControl.otalk.ctl_addr : ");
	print_swapped_sockaddr(&Control.otalk.ctl_addr);
	printf("\nControl.otalk.pid      : %ld\n", htonl(Control.otalk.pid));
	printf("Control.otalk.id_num   : %ld\n", htonl(Control.otalk.id_num));
	printf("Control.otalk.l_name   : %s\n", Control.otalk.l_name);
	printf("Control.otalk.r_name   : %s\n", Control.otalk.r_name);
	printf("Control.otalk.r_tty    : %s\n", Control.otalk.r_tty);
      }
    break;
  default:
    printf("print_control: Input device host has daemon type %d.\n",
	   type);
  }
}

/*
 * Function: ETM_response_print
 *
 * Print out the contents of the last response message.
 * 
 * Parameters:  type - Type of daemon.
 *
 * History:
 * eml 4/15/94
 * zappo   9/16/94         Added parameter type
 */
int ETM_response_print(type)
     enum DaemonType type;
{
  printf("Response message is:\n");
  switch(type)
    {
    case GTALKD:
      printf("Response.talk.vers    : %d\n", Response.talk.vers);
      if(Response.talk.type > BADCTLADDR)
	printf("Response.talk.type    : Unknown %d\n", Response.talk.type);
      else
	printf("Response.talk.type    : %s\n", msg_types[Response.talk.type]);
      printf("Response.talk.answer  : %s\n", msg_responses[Response.talk.answer]);
      if(verbose >= 2)
	{
	  printf("Response.talk.id_num  : %ld\n", ntohl(Response.talk.id_num));
	  printf("Response.talk.addr    : ");
	  print_swapped_sockaddr(&Response.talk.addr);
	  printf("\n");
	  printf("Response.gtalk.r_name : %s\n", Response.gtalk.r_name);
	}
      break;
    case NTALKD:
      printf("Response.talk.vers    : %d\n", Response.talk.vers);
      if(Response.talk.type > BADCTLADDR)
	printf("Response.talk.type    : Unknown %d\n", Response.talk.type);
      else
	printf("Response.talk.type    : %s\n", msg_types[Response.talk.type]);
      printf("Response.talk.answer  : %s\n", msg_responses[Response.talk.answer]);
      if(verbose >= 2)
	{
	  printf("Response.talk.id_num  : %ld\n", ntohl(Response.talk.id_num));
	  printf("Response.talk.addr    : ");
	  print_swapped_sockaddr(&Response.talk.addr);
	  printf("\n");
	}
      break;
    case OTALKD:
      printf("Response.otalk.type   : %s\n", msg_types[Response.otalk.type]);
      if(Response.otalk.type > BADCTLADDR)
	printf("Response.otalk.answer : Unknown %d\n", Response.otalk.answer);
      else
	printf("Response.otalk.answer : %s\n", msg_responses[Response.otalk.answer]);
      if(verbose >= 2)
	{
	  printf("Response.otalk.id_num : %ld\n", ntohl(Response.otalk.id_num));
	  printf("Response.otalk.addr   : ");
	  print_swapped_sockaddr(&Response.otalk.addr);
	  printf("\n");
	}
      break;
    default:
      printf("Response_Display: IO device has wrong host daemon type %d.\n",
	     type);
    }
  return Success;
}


/*
 * Function: DMN_check_compile
 *
 * Checks the compiled sizes of talk daemon messages against those
 * stored in config.h
 *
 * Returns:     int  - status
 * Parameters:  None
 *
 * History:
 * eml     Aug 17, 1994    
 */
int DMN_check_compile()
{
#define SOC sizeof(CTL_MSG_OLD)
#define SOR sizeof(CTL_RESPONSE_OLD)
#define NOC sizeof(CTL_MSG)
#define NOR sizeof(CTL_RESPONSE)
#define GOC sizeof(CTL_MSG_GNU)
#define GOR sizeof(CTL_RESPONSE_GNU)

  if((SOC != OTALK_CM_SIZE) || (SOR != OTALK_CMR_SIZE) ||
     (NOC != NTALK_CM_SIZE) || (NOR != NTALK_CMR_SIZE) ||
     (GOC != GTALK_CM_SIZE) || (GOR != GTALK_CMR_SIZE))
    {
      fprintf(stderr, "\nERROR in compile sizes!  (current/desired)\n\n");
      fprintf(stderr, "Old control %d/%d, Old response %d/%d\n", 
	      SOC, OTALK_CM_SIZE, SOR, OTALK_CMR_SIZE);
      fprintf(stderr, "New control %d/%d, New response %d/%d\n",
	      NOC, NTALK_CM_SIZE, NOR, NTALK_CMR_SIZE);
      fprintf(stderr, "GNU control %d/%d, GNU response %d/%d\n", 
	      GOC, GTALK_CM_SIZE, GOR, GTALK_CMR_SIZE);
      fprintf(stderr, "\nSee config.h for desired sizes.\n");
      return Fail;
    }
  else
    {
      return Success;
    }
}
