/* gt_curses.c - Curses display driver
 *
 * Copyright (C) 1997, 1998, 1999 Free Sorftware Foundation
 * Copyright (C) 1995, 1996 Eric M. Ludlam
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, you can either send email to this
 * program's author (see below) or write to:
 * 
 *              The Free Software Foundation, Inc.
 *              675 Mass Ave.
 *              Cambridge, MA 02139, USA. 
 * 
 * Please send bug reports, etc. to zappo@gnu.org.
 * 
 * Description:
 * 
 *  This file handles the curses display, and keyboard input.
 * 
 * $Log: gt_curses.c,v $
 * Revision 1.16  1999/08/26 12:13:52  zappo
 * Added Ctxt to fix_user_windows.
 *
 * Revision 1.15  1998/01/29 17:26:15  zappo
 * Fixed up some comments.
 *
 * Revision 1.14  1998/01/04 12:40:49  zappo
 * Made changes to better support the BSD curses library.  Includes "move" after
 * clearwin, remembering the popup windows width, and fixing puttext to be
 * clearer, AND work correctly for popups under BSD curses.
 *
 * Revision 1.13  1997/12/14 19:15:17  zappo
 * Renamed package to gtalk, renamed symbols and files apropriately
 * Fixed copyright and email address.
 *
 * Revision 1.12  1997/10/18 02:46:56  zappo
 * Added section that will print out ^H if ^H is not an edit char.
 *
 * Revision 1.11  1997/07/27 17:13:52  zappo
 * Changed to use the generic list for all windows.
 *
 * Revision 1.10  1997/03/12 02:28:02  zappo
 * Added new _suspend method which stop etalk with SIGSTOP
 *
 * Revision 1.9  1997/02/01 14:31:14  zappo
 * Added paramter to CURS_setlabel to make it compatible with the X
 * version.
 *
 * Revision 1.8  1996/03/02  03:13:59  zappo
 * Fixed ncurses.h lookup in configure.in, affecting #include here
 *
 * Revision 1.7  1996/01/20  19:09:24  zappo
 * Updated display code to be cleaner by using a Methods structure.
 *
 * Revision 1.6  1995/12/09  23:53:20  zappo
 * Fixed headers files for ncurses to ncurses/ncurses.h
 *
 * Revision 1.5  1995/11/26  15:36:23  zappo
 * Fixed bug in last checkin
 *
 * Revision 1.4  1995/11/26  06:51:46  zappo
 * Fixed differences in use between curses and ncurses.  Deals with
 * macros verses function calls, scrolling, inverse color, and positions
 *
 * Revision 1.3  1995/09/29  08:28:08  zappo
 * curses init now sets up it's own TTY link, and fix windows uses
 * DISP_window_displyed to filter displayed windows, and made all calls
 * K&R C, and added a bell command
 *
 * Revision 1.2  1995/09/20  23:25:39  zappo
 * Rearanged code which was sharable in X by moving into gt_display.c
 *
 * Revision 1.1  1995/09/14  10:42:03  zappo
 * Initial revision
 *
 * Revision 1.1  1995/08/25  16:46:23  zappo
 * Initial revision
 *
 * Tokens: ::Header:: gtalkc.h
 */

#include "gtalklib.h"
#include "gtalkc.h"

#if HAVE_SIGNAL_H == 1
#include <signal.h>
#endif

/* Not only must we have/not-have these variables, we must check
 * what the user specified for header files as well.
 */
#if CURSES_TAG == ncurses_tag && defined HAVE_NCURSES_H
#include <ncurses.h>
#else
# if CURSES_TAG == curses_tag && defined HAVE_CURSES_H
/* Curses declares these.  To prevent warnings, undefine them */
# undef TRUE
# undef FALSE
# include <curses.h>
# endif
#endif

/* We need this to correctly handle writing into a window with a
 * border.  In addition, BSD curses gets maxx wrong.
 */
static WINDOW *popup;
static int popup_width = 0;

static void *echo_area = NULL, *background = NULL;

static void no_action_here() {};

static void CURS_close();
static char CURS_getch();
static void CURS_suspend();
#ifdef PROTOTYPES
static void CURS_fix_user_windows(struct TalkContext *Ctxt, struct WindowList *list);
static void *CURS_popup(int width, int height);
static int CURS_winwidth(void *window);
static void CURS_clearwin(void *window);
static void CURS_puttext(void *window, char *text, int size);
static void CURS_refresh(void *window, int force);
static void CURS_delchar(void *window);
static void CURS_delword(void *window);
static void CURS_delline(void *window);
static int CURS_fillcolumn(void *window);
static void CURS_bell(struct TalkContext *Ctxt);
static void CURS_setlabel(void *win, char *label);
static void CURS_refresh_screen(struct WindowList *wlist);
static void CURS_map_popup(void *window);
static void CURS_set_echo_area(char *label, int showcursor);
static void CURS_delwin(void *window);
#else
static void CURS_fix_user_windows();
static void *CURS_popup();
static int CURS_winwidth();
static void CURS_clearwin();
static void CURS_puttext();
static void CURS_refresh();
static void CURS_delchar();
static void CURS_delword();
static void CURS_delline();
static int CURS_fillcolumn();
static void CURS_bell();
static void CURS_setlabel();
static void CURS_refresh_screen();
static void CURS_map_popup();
static void CURS_set_echo_area();
static void CURS_delwin();
#endif

static struct IfaceMethods CMethods =
{
  CURS_close,
  CURS_bell,
  CURS_suspend,
  /* Management */
  CURS_fix_user_windows,
  CURS_clearwin,
  CURS_winwidth,
  CURS_setlabel,
  /* editing commands */
  CURS_delchar,
  CURS_delline,
  CURS_delword,
  CURS_fillcolumn,
  CURS_puttext,
  CURS_refresh,
  CURS_refresh_screen,
  /* IO stuff */
  no_action_here,
  CURS_getch,
  CURS_popup,
  CURS_map_popup,
  CURS_delwin,
  CURS_set_echo_area
};


/*
 * Function: CURS_init, close
 *
 *   Initialization routines.
 *      init - the whole system:
 *      close - close the system
 *
 * Returns:     IFaceMethods - Curses control methods
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   8/30/95    Created
 */
struct IfaceMethods *CURS_init(Ctxt)
     struct TalkContext *Ctxt;
{
  /* basic curses stuff. */
  initscr();
#if HAVE_NCURSES_H
  keypad(stdscr, FALSE);
#endif
  cbreak();
  noecho();
  raw();

  /* Create a background window which we can use to refresh the background */
  background = newwin(LINES, COLS, 0, 0);

  /* Initialize some windows.  Make it look like emacs */
  echo_area = newwin(1, COLS, LINES-1, 0);

  /* Set up TTY reading devices */
  Ctxt->tty         = GT_tty();
  Ctxt->tty->readme = DISP_input;

  return &CMethods;
}

static void CURS_close()
{
  clear();
  refresh();
  endwin();
}


/*
 * Function: CURS_suspend
 *
 *   Locally defined function which will suspend the current process
 * using SIGSTOP.
 *
 * Returns:     static void - 
 * Parameters:  None
 *
 * History:
 * zappo   3/11/97    Created
 */
static void CURS_suspend(Ctxt)
     struct TalkContext *Ctxt;
{ 
  kill (getpid (), SIGSTOP);
}


/*
 * Function: CURS_fix_user_windows
 *
 *   Takes the window list pointer, and redoes all the windows to make
 * sure that they are the right size, and that all windows fit on the
 * display.
 *
 * Returns:     struct  - 
 * Parameters:  Ctxt - Context
 *              list - Pointer toWindow list
 *
 * History:
 * zappo   8/30/95    Created
 */
static void CURS_fix_user_windows(Ctxt, list)
     struct TalkContext *Ctxt;
     struct WindowList *list;
{
  struct WindowList *loop;
  int nextfree;
  int winsize, winstat;
  int count, done;

  nextfree = 0;

  /* count up the number of windows. In this loop, LOCAL won't count,
   * so start at 1. */
  for(loop = NextElement(list), count = 1; loop; loop=NextElement(loop))
    {
      if(DISP_window_displayed(loop))
	count++;
    }

  /* rebuild the window list */
  for(loop = list, done = 0; loop; loop = NextElement(loop))
    {
      /* c short circuit protects us from a seg-fault */
      /* a non-user window is local guy, and all others have user pointers */
      /* if a user is cleaned, windows will have been cleaned */
      if(DISP_window_displayed(loop))
	{
	  /*            lines left           windows left    statusline */
	  winsize = (LINES - 1 - nextfree) / (count - done) - 1;
	  winstat = nextfree + winsize;
	  if(loop->window)
	    {
	      /* Check to see if it needs resizing... */
	      if(!((((WINDOW *)loop->window)->_maxx == COLS) &&
		   (((WINDOW *)loop->window)->_maxy == winsize)))
		{
		  WINDOW *new;
		  int     yold, ynew, storey, storex;
		  /* Resize the window by making a new one, and copying stuff 
		   * into it from the old window. */
		  new = newwin(winsize, COLS, nextfree, 0);
		  if(!new)
		    {
		      gtalk_shutdown("Error creating resized window");
		    }
		  CURS_clearwin(new);
		  /* Now copy data from the old window into new... */
		  yold = ((WINDOW *)loop->window)->_cury;
		  if(yold >= new->_maxy) 
		    ynew = new->_maxy-1;
		  else
		    ynew = yold;
		  /* save x and y pos... */
		  storex = ((WINDOW *)loop->window)->_curx;
		  storey = ynew;
		  /* scan each line backwards... */
		  for(; (yold >= 0) && (ynew >= 0); yold--, ynew--)
		    {
		      int xlen, x;
		      char c;

		      xlen = ((WINDOW *)loop->window)->_maxx;
		      if(xlen > new->_maxx) xlen = new->_maxx;
		      
		      /* Copy over the window contents */
		      for(x = 0; x < xlen; x++)
			{
			  c = mvwinch(((WINDOW*)loop->window), yold, x);
			  mvwinsch(new, ynew, x, c);
			}
		    }
		  /* Nuke the old window */
		  delwin((WINDOW *)loop->window);
		  loop->window = new;
		  scrollok(((WINDOW*)loop->window), TRUE);
		  /* move status line to the right line */
		  mvwin((WINDOW *)loop->statusline, winstat, 0);
		  /* reposition the cursor to where it belongs.. */
		  wmove((WINDOW *)loop->window, storey, storex);
		}
	      else
		{
		  /* check to see if it just needs to be moved */
		  /* note: moving costs nothing since we have to
		     refresh anyway, and curses handles effecient
		     refreshing anyway, so just do it. */
		  mvwin((WINDOW *)loop->window, nextfree, 0);
		  mvwin((WINDOW *)loop->statusline, winstat, 0);
		}
	      /* else, it's fine for some reason! */
	    }
	  else /* create a new window of the right size */
	    {
	      loop->window = newwin(winsize, COLS, nextfree, 0);
	      loop->statusline = newwin(1, COLS, winstat, 0);
	      /* set magic settings */
	      scrollok(((WINDOW*)loop->window), TRUE);
#ifdef A_REVERSE
	      wattron(((WINDOW*)loop->statusline), A_REVERSE);
#endif
	    }
	  /* Refresh the window no matter what! */
	  wrefresh(loop->window);
	  wrefresh(loop->statusline);
	  nextfree = winstat +1;
	  done++;
	}
    }
}


/*
 * Function: CURS_winwidth, winheight
 *
 *   Window status and attribute routines
 *
 * Returns:     Nothing
 * Parameters:  window - EmptyPointer to window
 *
 * History:
 * zappo   8/30/95    Created
 */
static int CURS_winwidth(window)
     void *window;
{
  if(window == popup)
    {
      return popup_width - 1;
    }
  else
    {
      return ((WINDOW *)window)->_maxx;
    }
}


/*
 * Function: CURS_clearwin, puttext, refresh, refresh_screen
 *
 *   Window contents editing routines
 *
 * Returns:     Nothing
 * Parameters:  win - EmptyPointer to win
 *
 * History:
 * zappo   8/30/95    Created
 */
static void CURS_clearwin(window)
     void *window;
{
  werase((WINDOW *)window);
  /* This is required for curses, but not ncurses which does this
   * be default.
   */
  wmove((WINDOW *)window, 0, 0);
}

static void CURS_delchar(window)
     void *window;
{
  if(((WINDOW *)window)->_curx == 0)
    {
      /* This SHOULD be Ctxt, but we know (via cheating) that it isn't used */
      CURS_bell(NULL);
    }
  else
    {
      mvwaddch(((WINDOW *)window), ((WINDOW *)window)->_cury,
	       ((WINDOW *)window)->_curx - 1, ' ');
      wmove(((WINDOW *)window), ((WINDOW *)window)->_cury,
	    ((WINDOW *)window)->_curx - 1);
    }
}

static void CURS_delword(window)
     void *window;
{
  int y, x;
  y = ((WINDOW *)window)->_cury;
  x = ((WINDOW *)window)->_curx;

  /* Go backwards, removing initial space characters... */
  while((x != 0) && (mvwinch(((WINDOW *)window), y, x) == ' '))
    {
      waddch(((WINDOW *)window), ' ');
      x--;
    }
  /* Go backwards, removing text characters... */
  while((x != 0) && (mvwinch(((WINDOW *)window), y, x) != ' '))
    {
      waddch(((WINDOW *)window), ' ');
      x--;
    }
  /* Put the found space back in there */
  waddch(((WINDOW *)window), ' ');
}

static void CURS_delline(window)
     void *window;
{
  wmove(((WINDOW *)window), ((WINDOW *)window)->_cury, 0);
  wclrtoeol(((WINDOW *)window));
}

static int CURS_fillcolumn(window)
     void *window;
{
  return (((WINDOW *)window)->_curx >= (((WINDOW *)window)->_maxx - 10));
}

static void CURS_puttext(window, text, size)
     void *window;
     char *text;
     int   size;
{
  int i, mw, cw;

  cw = (((WINDOW *)window)->_curx);

  mw = CURS_winwidth(window);

  for(i=0;
      (i < size) && ((window == popup)? (mw > cw) : TRUE);
      i++, cw = (((WINDOW *)window)->_curx))
    {
      waddch((WINDOW *)window, text[i]);
      if(((text[i] == '\n') || (((WINDOW *)window)->_curx == 0))
	 && (window != popup))
	{
	  wclrtoeol((WINDOW *)window); /* curses doesn't do this for us */
	}
      else if(((text[i] == 8/*C-h*/) || (((WINDOW *)window)->_curx == 0))
	      && (window != popup)) 
	{
	  /* C-h interprets in some version of curses to a backspace,
	   * which we don't want.  If it's a real edit char, then we
	   * won't ever make it here.
	   */
	  waddch((WINDOW *)window, '^');
	  waddch((WINDOW *)window, 'H');
	}
    }

  /* if drawing to popup - auto newline this puppy */
  if(window == popup)
    {
      wmove(popup, popup->_cury + 1, 1);		  
    }
}

static void CURS_refresh(window, force)
     void *window;
     int force;
{
  if(force)
    {
      clearok(((WINDOW *)window), TRUE);
      touchwin((WINDOW *)window);
    }
  wrefresh(((WINDOW *)window));
}

static void CURS_map_popup(window)
     void *window;
{
  CURS_refresh(window, FALSE);
}

static void CURS_refresh_screen(wlist)
     struct WindowList *wlist;
{
  struct WindowList *loop;
  
  CURS_refresh(background, TRUE);
  for(loop = wlist; loop; loop = NextElement(loop))
    {
      if(DISP_window_displayed(loop))
	{
	  CURS_refresh(loop->window, TRUE);
	  CURS_refresh(loop->statusline, TRUE);
	}
    }
  CURS_refresh(echo_area, TRUE);
  
  /* lastly, if we have a popup, refresh that too. */
  if(popup)
    {
      CURS_refresh(popup, TRUE);
    }
}


/*
 * Function: CURS_getch
 *
 *   Curses input functions (from keyboard)
 *
 * Returns:     Nothing
 * Parameters:  None
 *
 * History:
 * zappo   8/31/95    Created
 */
static char CURS_getch()
{
  return getch();
}


/*
 * Function: CURS_setlabel
 *
 *   Sets the label under a talk window
 *
 * Returns:     Nothing
 * Parameters:  win   - EmptyPointer to win
 *              label - Pointer toCharacter of label
 * History:
 * zappo   1/23/96    Created
 */
static void CURS_setlabel(win, label)
     void *win;
     char *label;
{
  int len;

  if(strlen(label) > CURS_winwidth(win))
    len = CURS_winwidth(win);
  else
    len = strlen(label);

#ifndef A_REVERSE
  /* The following behavior is for status lines only. */
  if(win != echo_area)
    {
      /* We know that "label" is not used after this routine
       * is called, so we can modify it.  If A_REVERSE is not set, then
       * that means that we won't get reverse video on the mode-line.
       * This makes it look silly.  Here we look for space pairs and stick
       * '-' chars in them so we get a line effect.
       */
      int cnt;
      for(cnt = 1; cnt < len; cnt++)
	{
	  if ((label[cnt] == ' ') && ((label[cnt-1] == ' ') ||
				      (label[cnt-1] == '-')))
	    {
	      label[cnt] = '-';
	    }
	}
    }
#endif

  CURS_clearwin(win);
  CURS_puttext(win, label, len);
  CURS_refresh(win, FALSE);
}
static void CURS_set_echo_area(label, showcursor)
     char *label;
     int showcursor;
{
  if(echo_area)
    CURS_setlabel(echo_area, label);
}

/*
 * Function: CURS_popup
 *
 *   Popup window routines
 *
 * Returns:     Nothing
 * Parameters:  width  - Width of the popup window to create.
 *              height - Height of the popup window to create.
 * History:
 * zappo   8/30/95    Created
 */
static void *CURS_popup(width, height)
     int width;
     int height;
{
  if(height >= LINES) height = LINES - 2;
  if(width >= COLS) width = COLS - 2;

  popup = newwin(height, width, (LINES - height) / 2, (COLS - width) / 2);

  if(!popup)
    return NULL;

  wmove(popup, 1, 0);

  box(popup, '|', '-');

  wmove(popup, 1, 1);

  wrefresh(popup);

  popup_width = width;

  return popup;
}

/*
 * Function: CURS_delwin
 *
 *   Deletes a curses window.  Additionally checks for popup so we can
 * delete our version of popup as well.
 *
 * Returns:     Nothing
 * Parameters:  window - EmptyPointer to window
 *
 * History:
 * zappo   8/31/95    Created
 */
static void CURS_delwin(window)
     void *window;
{
  delwin((WINDOW *)window);

  if(window == popup) 
    popup = NULL;
}


/*
 * Function: CURS_bell
 *
 *   Rings the bell via curses.
 *
 * Returns:     Nothing
 * Parameters:  Ctxt - Context
 *
 * History:
 * zappo   9/24/95    Created
 */
static void CURS_bell(Ctxt)
     struct TalkContext *Ctxt;
{
#if HAVE_NCURSES_NCURSES_H
  beep();
#else
  /* We don't have a convenient BEEP, so be clever and push a BELL char
   * to the terminal */
  putc('\07', stderr);
#endif
}
